// Membrane Dynamics in 3D using Discrete Differential Geometry (Mem3DG)
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright (c) 2020:
//     Laboratory for Computational Cellular Mechanobiology
//     Cuncheng Zhu (cuzhu@eng.ucsd.edu)
//     Christopher T. Lee (ctlee@ucsd.edu)
//     Ravi Ramamoorthi (ravir@cs.ucsd.edu)
//     Padmini Rangamani (prangamani@eng.ucsd.edu)
//

/**
 * @file  typetraits.h
 * @brief Helper functions for debugging template types.
 *
 * This is copied directly from this very helpful post from
 * <a
 * href="https://stackoverflow.com/questions/81870/is-it-possible-to-print-a-variables-type-in-standard-c/20170989#20170989">Stackoverflow</a>.
 */

/// @cond hidden
#pragma once

#include <cstddef>
#include <cstring>
#include <ostream>
#include <stdexcept>

#ifndef _MSC_VER
#if __cplusplus < 201103
#define CONSTEXPR11_TN
#define CONSTEXPR14_TN
#define NOEXCEPT_TN
#elif __cplusplus < 201402
#define CONSTEXPR11_TN constexpr
#define CONSTEXPR14_TN
#define NOEXCEPT_TN noexcept
#else
#define CONSTEXPR11_TN constexpr
#define CONSTEXPR14_TN constexpr
#define NOEXCEPT_TN noexcept
#endif
#else // _MSC_VER
#if _MSC_VER < 1900
#define CONSTEXPR11_TN
#define CONSTEXPR14_TN
#define NOEXCEPT_TN
#elif _MSC_VER < 2000
#define CONSTEXPR11_TN constexpr
#define CONSTEXPR14_TN
#define NOEXCEPT_TN noexcept
#else
#define CONSTEXPR11_TN constexpr
#define CONSTEXPR14_TN constexpr
#define NOEXCEPT_TN noexcept
#endif
#endif // _MSC_VER

class static_string {
  const char *const p_;
  const std::size_t sz_;

public:
  typedef const char *const_iterator;

  template <std::size_t N>
  CONSTEXPR11_TN static_string(const char (&a)[N]) NOEXCEPT_TN : p_(a),
                                                                 sz_(N - 1) {}

  CONSTEXPR11_TN static_string(const char *p, std::size_t N) NOEXCEPT_TN
      : p_(p),
        sz_(N) {}

  CONSTEXPR11_TN const char *data() const NOEXCEPT_TN { return p_; }
  CONSTEXPR11_TN std::size_t size() const NOEXCEPT_TN { return sz_; }

  CONSTEXPR11_TN const_iterator begin() const NOEXCEPT_TN { return p_; }
  CONSTEXPR11_TN const_iterator end() const NOEXCEPT_TN { return p_ + sz_; }

  CONSTEXPR11_TN char operator[](std::size_t n) const {
    return n < sz_ ? p_[n] : throw std::out_of_range("static_string");
  }
};

inline std::ostream &operator<<(std::ostream &os, static_string const &s) {
  return os.write(s.data(), s.size());
}
/// @endcond

/**
 * @brief      Print the typename of an object at compile time.
 *
 * Example usage:
 * ~~~~~~~~~~~~~~~~{.cpp}
 * std::cout << "decltype(i) is " << type_name<decltype(i)>() << '\n';
 * ~~~~~~~~~~~~~~~~
 */
template <class T> CONSTEXPR14_TN static_string type_name() {
#ifdef __clang__
  static_string p = __PRETTY_FUNCTION__;
  return static_string(p.data() + 31, p.size() - 31 - 1);
#elif defined(__GNUC__)
  static_string p = __PRETTY_FUNCTION__;
#if __cplusplus < 201402
  return static_string(p.data() + 36, p.size() - 36 - 1);
#else
  return static_string(p.data() + 46, p.size() - 46 - 1);
#endif
#elif defined(_MSC_VER)
  static_string p = __FUNCSIG__;
  return static_string(p.data() + 38, p.size() - 38 - 7);
#endif
}
