"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscriptionProtocol = exports.Subscription = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
            redrivePolicy: this.buildDeadLetterConfig(this.deadLetterQueue),
        });
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue;
        deadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [deadLetterQueue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new aws_iam_1.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': props.topic.topicArn },
            },
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                deadLetterTargetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    SubscriptionProtocol["HTTP"] = "http";
    SubscriptionProtocol["HTTPS"] = "https";
    SubscriptionProtocol["EMAIL"] = "email";
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    SubscriptionProtocol["SMS"] = "sms";
    SubscriptionProtocol["SQS"] = "sqs";
    SubscriptionProtocol["APPLICATION"] = "application";
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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