# pylint: disable=missing-function-docstring
"""Events generated by the TS3Connection class."""
import logging
import sys
import traceback
from enum import IntEnum, Enum


class EventType(Enum):
    """
    Enum of event names used by TS3.
    """
    CLIENT_ENTER = "notifycliententerview"
    CLIENT_LEFT = "notifyclientleftview"
    CLIENT_MOVED = "notifyclientmoved"
    TEXT_MESSAGE = "notifytextmessage"
    SERVER_EDITED = "notifyserveredited"
    CHANNEL_DESC_CHANGED = "notifychanneldescriptionchanged"
    CHANNEL_EDITED = "notifychanneledited"
    CHANNEL_CREATED = "notifychannelcreated"
    CHANNEL_DELETED = "notifychanneldeleted"
    CHANNEL_MOVED = "notifychannelmoved"
    CHANNEL_PASSWORD_CHANGED = "notifychannelpasswordchanged"
    UNKNOWN = None


# Sadly, there seems to be no way to differentiate between the last two server events and the
# client events ...
server_events = [EventType.SERVER_EDITED, EventType.CLIENT_ENTER, EventType.CLIENT_LEFT]
text_events = [EventType.TEXT_MESSAGE]
channel_events = [EventType.CLIENT_ENTER, EventType.CLIENT_LEFT, EventType.CLIENT_MOVED,
                  EventType.CHANNEL_DESC_CHANGED, EventType.CHANNEL_EDITED,
                  EventType.CHANNEL_CREATED, EventType.CHANNEL_MOVED,
                  EventType.CHANNEL_DELETED, EventType.CHANNEL_PASSWORD_CHANGED]


class ReasonID(IntEnum):
    """
    Enum of ReasonID given for events.
    """
    SELF_JOINED = 0
    MOVED = 1
    TIMEOUT = 3
    CHANNEL_KICK = 4
    SERVER_KICK = 5
    BAN = 6
    LEFT = 8
    EDITED = 10
    SERVER_SHUTDOWN = 11


class TS3Event():
    """
    Event class for Teamspeak 3 events. This is a stub for all other events.
    """
    event_type = EventType.UNKNOWN

    def __init__(self, data, event_type=None):
        self._data = data
        if event_type is not None:
            self._event_type = event_type
        self._logger = logging.getLogger(__name__)

    @property
    def data(self):
        """
        Get event data.
        """
        return self._data

    def __getattr__(self, item):
        """
        Allows access to not explicitly specified attributes in event.
        :param item: Attribute name
        :type item: str
        :return: Attribute value string
        :rtype: str
        """
        print(self._data, item)
        return self._data[item]


class EventParser():
    @staticmethod
    def parse_event(event, event_type):
        """
        Parse an event.
        :param event: Event dictionary
        :type event: dict
        :param event_type: Event type to parse.
        :type event_type: str
        :return: Parsed Event
        :rtype: TS3Event
        """
        if EventType.TEXT_MESSAGE.value == event_type:
            parsed_event = TextMessageEvent(event)
            return parsed_event
        if EventType.CLIENT_MOVED.value == event_type:
            if 'invokerid' in event:
                parsed_event = ClientMovedEvent(event)
            else:
                parsed_event = ClientMovedSelfEvent(event)
            return parsed_event
        if EventType.CLIENT_ENTER.value == event_type:
            parsed_event = ClientEnteredEvent(event)
            return parsed_event
        if EventType.CLIENT_LEFT.value == event_type:
            reason_id = int(event.get("reasonid", '-1'))
            if reason_id == int(ReasonID.SERVER_KICK) or reason_id == int(ReasonID.CHANNEL_KICK):
                parsed_event = ClientKickedEvent(event)
            elif reason_id == int(ReasonID.BAN):
                parsed_event = ClientBannedEvent(event)
            else:
                parsed_event = ClientLeftEvent(event)
            return parsed_event
        if EventType.CHANNEL_DESC_CHANGED.value == event_type:
            parsed_event = ChannelDescriptionEditedEvent(event)
            return parsed_event
        if EventType.CHANNEL_EDITED.value == event_type:
            parsed_event = ChannelEditedEvent(event)
            return parsed_event
        if EventType.CHANNEL_CREATED.value == event_type:
            parsed_event = ChannelCreatedEvent(event)
            return parsed_event
        if EventType.CHANNEL_DELETED.value == event_type:
            parsed_event = ChannelDeletedEvent(event)
            return parsed_event
        if EventType.CHANNEL_MOVED.value == event_type:
            parsed_event = ChannelMovedEvent(event)
            return parsed_event
        if EventType.SERVER_EDITED.value == event_type:
            parsed_event = ServerEditedEvent(event)
            return parsed_event
        if EventType.CHANNEL_PASSWORD_CHANGED.value == event_type:
            parsed_event = ChannelPasswordChangedEvent(event)
            return parsed_event
        logging.warning("Unknown event! %s", str(event_type))
        return TS3Event(event, event_type)


class ServerEditedEvent(TS3Event):
    """
    Event created when the server is edited.
    """
    event_type = EventType.SERVER_EDITED

    def __init__(self, data):
        super().__init__(data)
        self._reason_id = data.get("reasonid", "")
        del data["reasonid"]
        self._invoker_id = data.get("invokerid", "-1")
        del data["invokerid"]
        self._invoker_uid = data.get("invokeruid", "-1")
        del data["invokeruid"]
        self._invoker_name = data.get("invokername", "")
        del data["invokername"]
        self._changed_properties = data

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

    @property
    def reason_id(self):
        return self._reason_id

    @property
    def changed_properties(self):
        return self._changed_properties


class ChannelEditedEvent(TS3Event):
    """
    Event created on channel edits.
    """
    event_type = EventType.CHANNEL_EDITED

    def __init__(self, data):
        super().__init__(data)
        self._channel_id = data.get('cid', '-1')
        self._channel_topic = data.get('channel_topic', '')
        self._invoker_id = data.get('invokerid', '-1')
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '-1')
        self._reason_id = data.get('reasonid', '-1')

    @property
    def channel_id(self):
        return self._channel_id

    @property
    def channel_topic(self):
        return self._channel_topic

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

    @property
    def reason_id(self):
        return self._reason_id


class ChannelCreatedEvent(TS3Event):
    """
    Event created on channel creation.
    """
    event_type = EventType.CHANNEL_CREATED

    def __init__(self, data):
        super().__init__(data)
        self._channel_id = data.get('cid', '-1')
        self._channel_topic = data.get('channel_topic', '')
        self._invoker_id = data.get('invokerid', '-1')
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '-1')
        self._reason_id = data.get('reasonid', '-1')

    @property
    def channel_id(self):
        return self._channel_id

    @property
    def channel_topic(self):
        return self._channel_topic

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

    @property
    def reason_id(self):
        return self._reason_id

class ChannelDeletedEvent(TS3Event):
    """
    Event created on channel deletion.
    """
    event_type = EventType.CHANNEL_DELETED

    def __init__(self, data):
        super().__init__(data)
        self._channel_id = data.get('cid', '-1')
        self._invoker_id = data.get('invokerid', '-1')
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '-1')

    @property
    def channel_id(self):
        return self._channel_id

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

class ChannelMovedEvent(TS3Event):
    """
    Event created when a channel is moved in the server hierarchy.
    """
    event_type = EventType.CHANNEL_MOVED

    def __init__(self, data):
        super().__init__(data)
        self._cid = data.get('cid', '-1')
        self._cpid = data.get('cpid', '-1')
        self._order = data.get('order', '-1')
        self._reasonid = data.get('reasonid', '-1')
        self._invoker_id = data.get('invokerid', '-1')
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '-1')

    @property
    def channel_id(self):
        return self._cid

    @property
    def channel_pid(self):
        return self._cpid

    @property
    def channel_order(self):
        return self._order

    @property
    def reason_id(self):
        return self._reasonid

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

class ChannelDescriptionEditedEvent(TS3Event):
    """
    Event created on channel description change.
    """
    event_type = EventType.CHANNEL_DESC_CHANGED

    def __init__(self, data):
        super().__init__(data)
        self._channel_id = int(data.get('cid', '-1'))

    @property
    def channel_id(self):
        return self._channel_id

class ChannelPasswordChangedEvent(TS3Event):
    """
    Event created on channel password change.
    """
    event_type = EventType.CHANNEL_PASSWORD_CHANGED

    def __init__(self, data):
        super().__init__(data)
        self._channel_id = int(data.get('cid', '-1'))

    @property
    def channel_id(self):
        return self._channel_id


class ClientEnteredEvent(TS3Event):
    """
    Event created when a client enters a channel.
    """
    event_type = EventType.CLIENT_ENTER

    def __init__(self, data):
        super().__init__(data)
        try:
            self._client_id = int(data.get('clid', '-1'))
            self._client_name = data.get('client_nickname', '')
            self._client_uid = data.get('client_unique_identifier', '')
            self._client_description = data.get('client_description', '')
            self._client_country = data.get('client_country', '')
            self._client_away = data.get('client_away', '')
            self._client_away_msg = data.get('client_away_message', '')
            self._client_input_muted = data.get('client_input_muted', '')
            self._client_output_muted = data.get('client_output_muted', '')
            self._client_outputonly_muted = data.get('client_outputonly_muted', '')
            self._client_input_hardware = data.get('client_input_hardware', '')
            self._client_output_hardware = data.get('client_output_hardware', '')
            self._target_channel_id = int(data.get('ctid', '-1'))
            self._from_channel_id = int(data.get('cfid', '-1'))
            self._reason_id = int(data.get('reasonid', '-1'))
            self._client_is_recording = data.get('client_is_recording', '')
            self._client_dbid = data.get('client_database_id', '')
            self._client_servergroups = data.get('client_servergroups', '')
            self._client_channel_group_id = int(data.get('client_channel_group_id', '-1'))
        except:
            self._logger.error("Failed to parse ClientEnterEvent:")
            self._logger.error(data)
            self._logger.error("\n\n")
            self._logger.error("Uncaught exception: %s", str(sys.exc_info()[0]))
            self._logger.error(str(sys.exc_info()[1]))
            self._logger.error(traceback.format_exc())

    @property
    def client_id(self):
        return self._client_id

    @property
    def client_name(self):
        return self._client_name

    @property
    def client_uid(self):
        return self._client_uid

    @property
    def client_description(self):
        return self._client_description

    @property
    def client_country(self):
        return self._client_country

    @property
    def client_away(self):
        return self._client_away

    @property
    def client_away_msg(self):
        return self._client_away_msg

    @property
    def client_input_muted(self):
        return self._client_input_muted

    @property
    def client_output_muted(self):
        return self._client_output_muted

    @property
    def client_outputonly_muted(self):
        return self._client_outputonly_muted

    @property
    def client_input_hardware(self):
        return self._client_input_hardware

    @property
    def client_output_hardware(self):
        return self._client_output_hardware

    @property
    def target_channel_id(self):
        return self._target_channel_id

    @property
    def from_channel_id(self):
        return self._from_channel_id

    @property
    def reason_id(self):
        return self._reason_id

    @property
    def client_is_recording(self):
        return self._client_is_recording

    @property
    def client_dbid(self):
        return self._client_dbid

    @property
    def client_servergroups(self):
        return self._client_servergroups

    @property
    def client_channel_group_id(self):
        return self._client_channel_group_id


class ClientLeftEvent(TS3Event):
    """
    Event created when a client leaves a channel.
    """
    event_type = EventType.CLIENT_LEFT

    def __init__(self, data):
        super().__init__(data)
        self._client_id = int(data.get('clid', '-1'))
        self._target_channel_id = int(data.get('ctid', '-1'))
        self._from_channel_id = int(data.get('cfid', '-1'))
        self._reason_id = int(data.get('reasonid', '-1'))
        self._reason_msg = data.get('reasonmsg', '')

    @property
    def client_id(self):
        return self._client_id

    @property
    def target_channel_id(self):
        return self._target_channel_id

    @property
    def reason_id(self):
        return self._reason_id

    @property
    def reason_msg(self):
        return self._reason_msg


class ClientKickedEvent(ClientLeftEvent):
    """
    Event created when a client is kicked.
    """
    def __init__(self, data):
        super().__init__(data)
        self._invoker_id = int(data.get('invokerid', '-1'))
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '')

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid


class ClientBannedEvent(ClientKickedEvent):
    """
    Event created when a client is banned.
    """
    def __init__(self, data):
        super().__init__(data)
        self._ban_time = int(data.get("bantime", '-1'))

    @property
    def ban_time(self):
        return self._ban_time


class ClientMovedEvent(TS3Event):
    """
    Event created when a client is moved from/to a channel.
    """
    event_type = EventType.CLIENT_MOVED

    def __init__(self, data):
        super().__init__(data)
        self._client_id = int(data.get('clid', '-1'))
        self._target_channel_id = int(data.get('ctid', '-1'))
        self._reason_id = int(data.get('reasonid', '-1'))
        self._invoker_id = int(data.get('invokerid', '-1'))
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '')

    @property
    def client_id(self):
        return self._client_id

    @property
    def target_channel_id(self):
        return self._target_channel_id

    @property
    def reason_id(self):
        return self._reason_id

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid


class ClientMovedSelfEvent(TS3Event):
    """
    Event created when a client is moves themselves from/to a channel.
    """
    event_type = EventType.CLIENT_MOVED

    def __init__(self, data):
        super().__init__(data)
        self._client_id = int(data.get('clid', '-1'))
        self._target_channel_id = int(data.get('ctid', '-1'))
        self._reason_id = int(data.get('reasonid', '-1'))

    @property
    def client_id(self):
        return self._client_id

    @property
    def target_channel_id(self):
        return self._target_channel_id

    @property
    def reason_id(self):
        return self._reason_id


class TextMessageEvent(TS3Event):
    """
    Event created when a Text Message is received.
    """
    event_type = EventType.TEXT_MESSAGE

    def __init__(self, data):
        super().__init__(data)
        if data.get('targetmode') == '1':
            self._targetmode = 'Private'
            self._target = data.get('target')
        elif data.get('targetmode') == '2':
            self._targetmode = 'Channel'
        elif data.get('targetmode') == '3':
            self._targetmode = 'Server'

        self._message = data.get('msg')
        self._invoker_id = int(data.get('invokerid', '-1'))
        self._invoker_name = data.get('invokername', '')
        self._invoker_uid = data.get('invokeruid', '-1')

    @property
    def invoker_id(self):
        return self._invoker_id

    @property
    def invoker_name(self):
        return self._invoker_name

    @property
    def invoker_uid(self):
        return self._invoker_uid

    @property
    def message(self):
        return self._message

    @property
    def targetmode(self):
        return self._targetmode

    @property
    def target(self):
        if self.targetmode == 'Private':
            return self._target
        return None
