# AUTOGENERATED! DO NOT EDIT! File to edit: 05_losses.ipynb (unless otherwise specified).

__all__ = ['aa_cosine', 'GeodesicLossR', 'ContinuousRotReprDecoder', 'ForwardKinematicLoss', 'VPoserLikelihood',
           'discretize', 'DiscretizedEulerLoss', 'euler_angle_mse', 'GeodesicLossSPL', 'PositionalLossSPL', 'PCK_SPL',
           'calculate_auc', 'all_univariate_tensors_in']

# Cell
import math

import torch
import torch.nn as nn
import torch.nn.functional as F

import numpy as np

import llamass.transforms as transforms

# Cell
def aa_cosine(out, target):
    if out.ndim == 2:
        b, d = out.size()
    elif out.ndim == 3:
        b, f, d = out.size()
        assert f == 1, f'{out.size()}'
    j = d//3
    out, target = out.view(b, j, 3), target.view(b, j, 3)
    def theta(x, eps=1e-6):
        return torch.sqrt(torch.clamp(torch.sum(x**2, 2, keepdims=True), eps, 2*math.pi))
    theta_a = theta(out)
    theta_b = theta(target)
    cosine_sim = F.cosine_similarity(out, target, dim=2)
    cosine_sim_loss = 1. - cosine_sim
    cosine_angle_diff = 1. - torch.cos(theta_a - theta_b)
    return cosine_sim_loss + cosine_angle_diff[:,:,0]

# Cell
class GeodesicLossR(nn.Module):
    def __init__(self, reduction='batchmean'):
        super(GeodesicLossR, self).__init__()

        self.reduction = reduction
        self.eps = 1e-6

    # batch geodesic loss for rotation matrices
    def bgdR(self,m1,m2):
        assert m1.ndim == m2.ndim, \
            f"Rotation matrices ndim must be equal but was {m1.ndim} {m2.ndim}"
        for m in [m1, m2]:
            assert m.size(-1) == 3 and m.size(-2) == 3, \
                f"Trailing 2 dimensions must 3x3 rotation matrices {m.size()}"
        if m1.ndim == 2:
            # ndim 2 must be single rotation matrix
            m1 = m1.view(1, 3, 3)
            m2 = m2.view(1, 3, 3)
        elif m1.ndim > 3:
            m1 = m1.view(-1, 3, 3)
            m2 = m2.view(-1, 3, 3)
        batch = m1.shape[0]
        m = torch.bmm(m1, m2.transpose(1, 2))  # batch*3*3

        cos = (m[:, 0, 0] + m[:, 1, 1] + m[:, 2, 2] - 1) / 2
        cos = torch.min(cos, m1.new(np.ones(batch)))
        cos = torch.max(cos, m1.new(np.ones(batch)) * -1)

        return torch.acos(cos)

    def forward(self, ypred, ytrue):
        theta = self.bgdR(ypred,ytrue)
        if self.reduction == 'mean':
            return torch.mean(theta)
        elif self.reduction == 'batchmean':
            return torch.mean(torch.sum(theta, dim=theta.shape[1:]))
        elif self.reduction == 'none':
            return theta
        else:
            raise NotImplementedError(f"Reduction {self.reduction} not known")

# Cell
class ContinuousRotReprDecoder(nn.Module):
    def __init__(self):
        super().__init__()

    def forward(self, module_input):
        b, d = module_input.size()
        assert d%6 == 0
        reshaped_input = module_input.view(-1, 3, 2)

        b1 = F.normalize(reshaped_input[:, :, 0], dim=1)

        dot_prod = torch.sum(b1 * reshaped_input[:, :, 1], dim=1, keepdim=True)
        b2 = F.normalize(reshaped_input[:, :, 1] - dot_prod * b1, dim=-1)
        b3 = torch.cross(b1, b2, dim=1)

        return torch.stack([b1, b2, b3], dim=-1).view(b, -1, 3, 3)

# Cell
class ForwardKinematicLoss(nn.Module):
    "Must be initialized with an SMPL-like `body_model`."
    def __init__(self, body_model):
        super().__init__()
        self.bm = body_model
        self.geodesic_loss = GeodesicLossR(reduction="mean")

    def kinematics(self, aa_out, pose_target):
        with torch.no_grad():
            bm_orig = self.bm(pose_body=pose_target)
        bm_rec = self.bm(pose_body=aa_out.contiguous())
        return bm_orig, bm_rec

# Cell
class VPoserLikelihood(ForwardKinematicLoss):
    def forward(
        self,
        dec_out,
        aa_out,
        pose_target,
        pose_target_rotmat,
        bm_orig=None,
        bm_rec=None,
        loss_rec_wt=torch.tensor(4),
        loss_matrot_wt=torch.tensor(2),
        loss_jtr_wt=torch.tensor(2),
        callback=None
    ):
        """
        Default settings for loss weights taken from:
        https://github.com/nghorbani/human_body_prior/blob/master/src/human_body_prior/train/V02_05/V02_05.yaml
        Inputs:
            - dec_out: output of network as rotation matrix, shape (batch, frames, joints, 3, 3)
            - aa_out: output of network as axis-angle vectors, shape (batch, frames, joints, 3)
            - pose_target: target as axis-angle vectors, shape (batch, frames, joints, 3)
            - pose_target_rotmat: target as rotation matrix, shape (batch, frames, joints, 3, 3)
        """
        l1_loss = torch.nn.L1Loss(reduction="mean")

        # cast decoder output to aa
        bs, f, d = pose_target.size()

        # forward kinematics
        if bm_orig is None or bm_rec is None:
            bm_orig, bm_rec = self.kinematics(aa_out.view(bs*f, -1), pose_target.view(bs*f, d))

        # Reconstruction loss - L1 on the output mesh
        v2v = l1_loss(bm_rec.v, bm_orig.v)

        # Geodesic loss between rotation matrices
        matrot_loss = self.geodesic_loss(
            dec_out.view(-1, 3, 3),
            pose_target_rotmat.view(-1, 3, 3)
        )
        # L1 Loss on joint positions
        jtr_loss = l1_loss(bm_rec.Jtr, bm_orig.Jtr)

        # apply weights to make weighted loss
        weighted_loss = (
            loss_matrot_wt * matrot_loss + loss_rec_wt * v2v + loss_jtr_wt * jtr_loss
        )


        # log results
        with torch.no_grad():
            unweighted_loss = matrot_loss + v2v + jtr_loss
            if callback is not None:
                callback(all_univariate_tensors_in(locals()))

        return weighted_loss

# Cell
def discretize(x, nquant, eps=1e-6, dither=False, zero_centered=True):
    if zero_centered:
        x = x + math.pi
    m = math.pi*2
    assert x.max() < m
    x = x/m # scale to between zero and 1
    x = x*nquant
    if dither:
        d = 2.*(torch.rand_like(x)-0.5)
        x = torch.clamp(x+d, 0, nquant-eps)
    return torch.floor(x).long() # bin account to nquant levels

class DiscretizedEulerLoss(nn.Module):
    def __init__(self, nquant, dither=False, zero_centered=True):
        super().__init__()
        self.nquant, self.dither, self.zero_centered = nquant, dither, zero_centered

    def forward(self, out, target):
        assert out.size(-1) == self.nquant, f'trailing dimension should hold logits {out.size()}'
        target = discretize(target, self.nquant, dither=self.dither, zero_centered=self.zero_centered)
        return F.nll_loss(out.view(-1, self.nquant), target.view(-1))

# Cell
def euler_angle_mse(predictions, targets, n_joints=21):
    "Inputs predictions and targets are assumed to be rotation matrices"
    predictions = transforms.Rotation.from_matrix(predictions).as_euler()
    targets = transforms.Rotation.from_matrix(targets).as_euler()

    predictions = predictions.view(-1, n_joints*3)
    targets = targets.view(-1, n_joints*3)

    # l2 error on euler angles
    #idx_to_use = np.where(np.std(euler_targs, 0) > 1e-4)[0]
    mask = (torch.std(targets, 0) > 1e-4).float().view(1, -1)
    euc_error = torch.square(targets*mask - predictions*mask)
    euc_error = torch.sqrt(torch.sum(euc_error, 1))  # (-1, ...)
    return euc_error

# Cell
class GeodesicLossSPL(GeodesicLossR):
    def __init__(self, reduction='none'):
        super().__init__(reduction=reduction)

    def bgdR(self, m1, m2):
        assert m1.ndim == m2.ndim, \
            f"Rotation matrices ndim must be equal but was {m1.ndim} {m2.ndim}"
        for m in [m1, m2]:
            assert m.size(-1) == 3 and m.size(-2) == 3, \
                f"Trailing 2 dimensions must 3x3 rotation matrices {m.size()}"
        if m1.ndim == 2:
            # ndim 2 must be single rotation matrix
            m1 = m1.view(1, 3, 3)
            m2 = m2.view(1, 3, 3)
        elif m1.ndim > 3:
            m1 = m1.view(-1, 3, 3)
            m2 = m2.view(-1, 3, 3)
        batch = m1.shape[0]
        m = torch.bmm(m1, m2.transpose(1, 2))  # batch*3*3

        aa = transforms.Rotation.from_matrix(m).as_rotvec().view(batch, 3)
        angles = torch.linalg.norm(aa, axis=1)

        return angles

# Cell
class PositionalLossSPL(ForwardKinematicLoss):
    def forward(
        self,
        aa_out=None,
        pose_target=None,
        bm_orig=None,
        bm_rec=None,
        positions=None,
        target_positions=None
    ):
        for p in [positions, target_positions]:
            if p is not None:
                assert p.ndim == 3
                assert p.size(-1) == 3, "final dim must contain 3D locations"
        if pose_target is not None:
            if pose_target.ndim == 3:
                bs, f, d = pose_target.size()
                n = bs*f
                assert d == n_joints*3
            elif pose_target.ndim == 2:
                n, d = pose_target.size()

        # forward kinematics
        no_bm_output = bm_orig is None or bm_rec is None
        no_positions = positions is None or target_positions is None
        if no_bm_output and no_positions:
            bm_orig, bm_rec = self.kinematics(aa_out.reshape(n, d), pose_target.reshape(n, d))
            positions = bm_rec.Jtr
            target_positions = bm_orig.Jtr

        return torch.sqrt(torch.square(positions - target_positions).sum(2))

# Cell
class PCK_SPL(ForwardKinematicLoss):
    def forward(
        self,
        aa_out=None,
        pose_target=None,
        positions=None,
        target_positions=None,
        thresh=None,
        bm_orig=None,
        bm_rec=None,
        n_joints=21
    ):
        assert thresh is not None
        for p in [positions, target_positions]:
            if p is not None:
                assert p.ndim == 3
                assert p.size(-1) == 3, "final dim must contain 3D locations"
        if pose_target is not None:
            if pose_target.ndim == 3:
                bs, f, d = pose_target.size()
                n = bs*f
                assert d == n_joints*3
            elif pose_target.ndim == 2:
                n, d = pose_target.size()

        # forward kinematics
        no_bm_output = bm_orig is None or bm_rec is None
        no_positions = positions is None or target_positions is None
        if no_bm_output and no_positions:
            bm_orig, bm_rec = self.kinematics(aa_out.reshape(n, d), pose_target.reshape(n, d))
            positions = bm_rec.Jtr
            target_positions = bm_orig.Jtr

        # percentage of coordinates in the ball defined by thresh around a joint
        n, d, _ = positions.size()
        dist = torch.sqrt(torch.square(positions - target_positions).sum(2))
        return torch.mean((dist <= thresh).float(), 1)

# Cell
"""
SPL: training and evaluation of neural networks with a structured prediction layer.
Copyright (C) 2019 ETH Zurich, Emre Aksan, Manuel Kaufmann

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
"""

def calculate_auc(pck_values, pck_thresholds, target_length):
    """Calculate area under a curve (AUC) metric for PCK.

    If the sequence length is shorter, we ignore some of the high-tolerance PCK values in order to have less
    saturated AUC.
    Args:
        pck_values (list): PCK values.
        pck_thresholds (list): PCK threshold values.
        target_length (int): determines for which time-step we calculate AUC.
    Returns:
    """
    # Due to the saturation effect, we consider a limited number of PCK thresholds in AUC calculation.
    if target_length < 6:
        n_pck = 6
    elif target_length < 12:
        n_pck = 7
    elif target_length < 18:
        n_pck = 8
    else:
        n_pck = len(pck_thresholds)

    norm_factor = np.diff(pck_thresholds[:n_pck]).sum()
    auc_values = []
    for i in range(n_pck - 1):
        auc = (pck_values[i] + pck_values[i + 1]) / 2 * (pck_thresholds[i + 1] - pck_thresholds[i])
        auc_values.append(auc)
    return np.array(auc_values).sum() / norm_factor

# Cell
def all_univariate_tensors_in(d):
    "Utility function for logging with a callback function"
    def is_univariate_tensor(x):
        if isinstance(x, torch.Tensor):
            return x.nelement() == 1

    return {k: v for k, v in d.items() if is_univariate_tensor(v)}