"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Document = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document
 */
class Document extends cdk.Construct {
    /**
     * Defines a new SSM document
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Name of the document
         */
        this.name = '';
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag('CreatedBy', ID);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new statement.Ssm().allow().toListDocuments().toListTagsForResource(),
                new statement.Ssm()
                    .allow()
                    .toCreateDocument()
                    .toAddTagsToResource()
                    .ifAwsRequestTag('CreatedBy', ID),
                new statement.Ssm()
                    .allow()
                    .toDeleteDocument()
                    .toDescribeDocument()
                    .toGetDocument()
                    .toListDocumentVersions()
                    .toModifyDocumentPermission()
                    .toUpdateDocument()
                    .toUpdateDocumentDefaultVersion()
                    .toAddTagsToResource()
                    .toRemoveTagsFromResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,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