"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const common_2 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 */
class AmiRecipe extends common_2.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_2.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        let parentAmi;
        let workingDirectory;
        if (props.platform == 'Linux') {
            let archUrl;
            if (props.architecture.is(common_1.Architecture.X86_64)) {
                archUrl = 'amd64';
            }
            else if (props.architecture.is(common_1.Architecture.ARM64)) {
                archUrl = 'arm64';
            }
            else {
                throw new Error(`Unsupported architecture for parent AMI: ${props.architecture.name}`);
            }
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter(`/aws/service/canonical/ubuntu/server/focal/stable/current/${archUrl}/hvm/ebs-gp2/ami-id`, {
                os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
            }).getImage(this).imageId;
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.latestWindows(aws_cdk_lib_1.aws_ec2.WindowsVersion.WINDOWS_SERVER_2022_ENGLISH_FULL_CONTAINERSLATEST).getImage(this).imageId;
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ImageRecipe', name, {
                platform: props.platform,
                components,
                parentAmi,
            }),
            parentImage: parentAmi,
            components,
            workingDirectory,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An AMI builder that uses AWS Image Builder to build AMIs pre-baked with all the GitHub Actions runner requirements. Builders can be used with {@link Ec2Runner}.
 *
 * Each builder re-runs automatically at a set interval to make sure the AMIs contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the AMI used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the EC2 provider, use:
 *
 * ```
 * const builder = new AmiBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.addComponent(new ImageBuilderComponent(scope, id, {
 *   platform: 'Linux',
 *   displayName: 'p7zip',
 *   description: 'Install some more packages',
 *   commands: [
 *     'set -ex',
 *     'apt-get install p7zip',
 *   ],
 * }));
 * new Ec2Runner(this, 'EC2 provider', {
 *     label: 'custom-ec2',
 *     amiBuilder: builder,
 * });
 * ```
 */
class AmiBuilder extends common_2.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_1.Os.LINUX, common_1.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_1.Architecture.X86_64, common_1.Architecture.ARM64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroups: props?.securityGroup ? [props.securityGroup] : props?.securityGroups,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'AMI',
        });
        // add all basic components
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.addBaseWindowsComponents();
        }
        else if (this.os.is(common_1.Os.LINUX)) {
            this.addBaseLinuxComponents();
        }
    }
    addBaseWindowsComponents() {
        this.addComponent(windows_components_1.WindowsComponents.cloudwatchAgent(this, 'CloudWatch agent'));
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
        this.addComponent(windows_components_1.WindowsComponents.docker(this, 'Docker'));
    }
    addBaseLinuxComponents() {
        this.addComponent(linux_components_1.LinuxUbuntuComponents.requiredPackages(this, 'Upgrade packages and install basics', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.runnerUser(this, 'User', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.awsCli(this, 'AWS CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubCli(this, 'GitHub CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.git(this, 'git', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion, this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.docker(this, 'Docker', this.architecture));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.platform == 'Linux') {
            this.prependComponent(linux_components_1.LinuxUbuntuComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else if (this.platform == 'Windows') {
            this.prependComponent(windows_components_1.WindowsComponents.extraCertificates(this, 'Extra Certs', path));
        }
        else {
            throw new Error(`Unknown platform: ${this.platform}`);
        }
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the AMI builder.
     */
    bind() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_2.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform,
            components: this.components,
            architecture: this.architecture,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        this.imageCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    imageCleaner(launchTemplate, stackName, builderName) {
        const deleter = utils_1.BundledNodejsFunction.singleton(this, 'delete-ami', {
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
}
exports.AmiBuilder = AmiBuilder;
_a = JSII_RTTI_SYMBOL_1;
AmiBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.AmiBuilder", version: "0.7.2" };
//# sourceMappingURL=data:application/json;base64,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