"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * GitHub Actions runner provider using CodeBuild to execute jobs.
 *
 * Creates a project that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class CodeBuildRunner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id);
        this.labels = this.labelsFromProperties('codebuild', props?.label, props?.labels);
        this.vpc = props?.vpc;
        if (props?.securityGroup) {
            this.securityGroups = [props.securityGroup];
        }
        else {
            if (props?.securityGroups) {
                this.securityGroups = props.securityGroups;
            }
            else {
                if (this.vpc) {
                    this.securityGroups = [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
                }
            }
        }
        let buildSpec = {
            version: '0.2',
            env: {
                variables: {
                    RUNNER_TOKEN: 'unspecified',
                    RUNNER_NAME: 'unspecified',
                    RUNNER_LABEL: 'unspecified',
                    OWNER: 'unspecified',
                    REPO: 'unspecified',
                    GITHUB_DOMAIN: 'github.com',
                },
            },
            phases: {
                install: {
                    commands: [
                        'nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://127.0.0.1:2375 --storage-driver=overlay2 &',
                        'timeout 15 sh -c "until docker info; do echo .; sleep 1; done"',
                        'if [ "${RUNNER_VERSION}" = "latest" ]; then RUNNER_FLAGS=""; else RUNNER_FLAGS="--disableupdate"; fi',
                        'sudo -Hu runner /home/runner/config.sh --unattended --url "https://${GITHUB_DOMAIN}/${OWNER}/${REPO}" --token "${RUNNER_TOKEN}" --ephemeral --work _work --labels "${RUNNER_LABEL}" ${RUNNER_FLAGS} --name "${RUNNER_NAME}"',
                    ],
                },
                build: {
                    commands: [
                        'sudo --preserve-env=AWS_CONTAINER_CREDENTIALS_RELATIVE_URI,AWS_DEFAULT_REGION,AWS_REGION -Hu runner /home/runner/run.sh',
                    ],
                },
            },
        };
        const imageBuilder = props?.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: CodeBuildRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        if (image.os.is(common_1.Os.WINDOWS)) {
            buildSpec.phases.install.commands = [
                'cd \\actions',
                'if (${Env:RUNNER_VERSION} -eq "latest") { $RunnerFlags = "" } else { $RunnerFlags = "--disableupdate" }',
                './config.cmd --unattended --url "https://${Env:GITHUB_DOMAIN}/${Env:OWNER}/${Env:REPO}" --token "${Env:RUNNER_TOKEN}" --ephemeral --work _work --labels "${Env:RUNNER_LABEL}" ${RunnerFlags} --name "${Env:RUNNER_NAME}"',
            ];
            buildSpec.phases.build.commands = [
                'cd \\actions',
                './run.cmd',
            ];
        }
        // choose build image
        let buildImage;
        if (image.os.is(common_1.Os.LINUX)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.fromEcrRepository(image.imageRepository, image.imageTag);
            }
            else if (image.architecture.is(common_1.Architecture.ARM64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.fromEcrRepository(image.imageRepository, image.imageTag);
            }
        }
        if (image.os.is(common_1.Os.WINDOWS)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.WindowsBuildImage.fromEcrRepository(image.imageRepository, image.imageTag, aws_cdk_lib_1.aws_codebuild.WindowsImageType.SERVER_2019);
            }
        }
        if (buildImage === undefined) {
            throw new Error(`Unable to find supported CodeBuild image for ${image.os.name}/${image.architecture.name}`);
        }
        // create project
        this.project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `GitHub Actions self-hosted runner for labels ${this.labels}`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            vpc: this.vpc,
            securityGroups: this.securityGroups,
            subnetSelection: props?.subnetSelection,
            timeout: props?.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage,
                computeType: props?.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: image.os.is(common_1.Os.LINUX),
            },
            logging: {
                cloudWatch: {
                    logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                        retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
                        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    }),
                },
            },
        });
        this.grantPrincipal = this.project.grantPrincipal;
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CodeBuildStartBuild(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            project: this.project,
            environmentVariablesOverride: {
                RUNNER_TOKEN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerTokenPath,
                },
                RUNNER_NAME: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerNamePath,
                },
                RUNNER_LABEL: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: this.labels.join(','),
                },
                GITHUB_DOMAIN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.githubDomainPath,
                },
                OWNER: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.ownerPath,
                },
                REPO: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.repoPath,
                },
            },
        });
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroups: this.securityGroups?.map(sg => sg.securityGroupId),
            roleArn: this.project.role?.roleArn,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.project.connections;
    }
}
exports.CodeBuildRunner = CodeBuildRunner;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildRunner", version: "0.7.2" };
/**
 * Path to Dockerfile for Linux x64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stable"`.
 * * `DIND_COMMIT` overrides the commit where dind is found.
 * * `DOCKER_VERSION` overrides the installed Docker version.
 * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
 */
CodeBuildRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'codebuild', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stable"`.
 * * `DIND_COMMIT` overrides the commit where dind is found.
 * * `DOCKER_VERSION` overrides the installed Docker version.
 * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
 */
CodeBuildRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'codebuild', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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