

def mars_modeling(
    x_train,
    y_train,
    x_test,
    model_weights:list=[1, 1],
    allow_missing:bool=True,
    max_degree:int=1,
    max_terms:int=100,
    max_iter:int=100,
    model_out:str="model.joblib"
):

    """
      Create a predictive model using MARS (pyearth). Further documentation on the algorythm can be found at https://contrib.scikit-learn.org/py-earth/

      Ex: gitlabds.mars_modeling(x_train, y_train, x_test , model_weights=[1,1], allow_missing =True, max_degree=1, max_terms=100, max_iter=100, model_out='model.joblib'):

      Parameters:


      x_train : train "predictors" dataframe

      y_train : train outcome/dv/target dataframe

      x_test : test "predictors" dataframe

      model_weights : (Optional) Pass model weights if up-/down-sampling was performed on the datasets. Otherwise defaults to a weight of 1 for x and y dataframes

      allow_missing : If True, use missing argument to determine missingness or,if X is a pandas DataFrame, infer missingness from X.

      max_degree : The maximum degree of terms generated by the forward pass. Setting = 2 will look at all first-order interactions between predictor columns (i.e x_train) and the outcome (y_train)

      max_terms : The maximum number of terms generated by the forward pass. All memory is allocated at the beginning of the forward pass, so setting max_terms to a very high number on a system with insufficient memory may cause a MemoryError at the start of the forward pass.

      max_iter : Maximum number of iterations taken for the solvers to converge.

      model_out : Where to save the resultant model file (packages as joblib).


      Returns

      model fit, model equation (string), pared-down x_train and x_test dataframes that contain only the columns that hit in the model.
    """
    import pandas as pd
    import copy
    import joblib
    from pyearth import Earth, export
    from sklearn.pipeline import Pipeline
    from sklearn.linear_model import LogisticRegression, LinearRegression
    pd.set_option("display.float_format", lambda x: "%.10f" % x)

    model = Pipeline([("earth",
                       Earth(feature_importance_type="gcv", 
                             allow_missing=allow_missing, 
                             verbose=1,
                             max_degree=max_degree,
                             max_terms=max_terms,
                            ),
                      ),
                      ("log",
                       LogisticRegression(class_weight={0: model_weights[0], 
                                                        1: model_weights[1]},
                                          verbose=1,
                                          solver="liblinear",
                                          max_iter=max_iter,
                                         ),
                      ),
                     ])
    
    model.fit(x_train, y_train)

    # Export Scoring Equation
    # equation = copy.deepcopy(model)
    # equation.named_steps['earth'].coef_ = equation.named_steps['log'].coef_
    # equation = str(export.export_sympy(equation.named_steps['earth']))
    # print(equation)

    # Second Pass using only variables that hit in the first pass. This makes the scoring easier as not all the columns from training dataset are needed -- just the ones that actually hit in the model
    rerun = pd.DataFrame()
    rerun["vars"] = model.named_steps["earth"].xlabels_
    rerun["imps"] = model.named_steps["earth"].feature_importances_
    rerun = rerun[rerun["imps"] > 0]

    x_train = x_train[rerun["vars"].to_list()]
    x_test = x_test[rerun["vars"].to_list()]

    # Re-initialize the model
    model = Pipeline([("earth",
                       Earth(feature_importance_type="gcv", 
                             allow_missing=allow_missing, 
                             verbose=1,
                             max_degree=max_degree,
                             max_terms=max_terms,
                            ),
                      ),
                      ("log",
                       LogisticRegression(class_weight={0: model_weights[0], 
                                                        1: model_weights[1]},
                                          verbose=1,
                                          solver="liblinear",
                                          max_iter=max_iter,
                                         ),
                      ),
                     ])
    
    model.fit(x_train, y_train)

    equation = copy.deepcopy(model)
    equation.named_steps["earth"].coef_ = equation.named_steps["log"].coef_
    equation = str(export.export_sympy(equation.named_steps["earth"]))
    print("\nModeling Equation:")
    print(equation)

    # Save the joblib model file
    joblib.dump(model, model_out)
    print(f"\njoblib model file saved as '{model_out}'")

    return model, equation, x_train, x_test
