/**
 * Represents a matcher that can perform special data matching capabilities between a given pattern and a target.
 *
 * @stability stable
 */
export declare abstract class Matcher {
    /**
     * Check whether the provided object is a subtype of the `IMatcher`.
     *
     * @stability stable
     */
    static isMatcher(x: any): x is Matcher;
    /**
     * A name for the matcher.
     *
     * This is collected as part of the result and may be presented to the user.
     *
     * @stability stable
     */
    abstract readonly name: string;
    /**
     * Test whether a target matches the provided pattern.
     *
     * Every Matcher must implement this method.
     * This method will be invoked by the assertions framework. Do not call this method directly.
     *
     * @param actual the target to match.
     * @returns the list of match failures. An empty array denotes a successful match.
     * @stability stable
     */
    abstract test(actual: any): MatchResult;
}
/**
 * The result of `Match.test()`.
 *
 * @stability stable
 */
export declare class MatchResult {
    /**
     * The target for which this result was generated.
     *
     * @stability stable
     */
    readonly target: any;
    private readonly failures;
    /**
     * @stability stable
     */
    constructor(target: any);
    /**
     * Push a new failure into this result at a specific path.
     *
     * If the failure occurred at root of the match tree, set the path to an empty list.
     * If it occurs in the 5th index of an array nested within the 'foo' key of an object,
     * set the path as `['/foo', '[5]']`.
     *
     * @param path the path at which the failure occurred.
     * @param message the failure.
     * @stability stable
     */
    push(matcher: Matcher, path: string[], message: string): this;
    /**
     * Does the result contain any failures.
     *
     * If not, the result is a success
     *
     * @stability stable
     */
    hasFailed(): boolean;
    /**
     * The number of failures.
     *
     * @stability stable
     */
    get failCount(): number;
    /**
     * Compose the results of a previous match as a subtree.
     *
     * @param id the id of the parent tree.
     * @stability stable
     */
    compose(id: string, inner: MatchResult): this;
    /**
     * Get the list of failures as human readable strings.
     *
     * @stability stable
     */
    toHumanStrings(): string[];
}
