"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const absent_1 = require("./private/matchers/absent");
const type_1 = require("./private/type");
/**
 * Partial and special matching during template assertions.
 *
 * @stability stable
 */
class Match {
    /**
     * Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @stability stable
     */
    static absent() {
        return new absent_1.AbsentMatch('absent');
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @stability stable
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * Matches any string-encoded JSON and applies the specified pattern after parsing it.
     *
     * @param pattern the pattern to match after parsing the encoded JSON.
     * @stability stable
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * Matches any non-null value at the target.
     *
     * @stability stable
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.134.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.push(this, [], `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`);
            return result;
        }
        if (actual !== this.pattern) {
            result.push(this, [], `Expected ${this.pattern} but received ${actual}`);
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type array but received ${type_1.getType(actual)}`);
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected array of length ${this.pattern.length} but received ${actual.length}`);
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            const matcherName = matcher.name;
            if (this.subsequence && (matcherName == 'absent' || matcherName == 'anyValue')) {
                // array subsequence matcher is not compatible with anyValue() or absent() matcher. They don't make sense to be used together.
                throw new Error(`The Matcher ${matcherName}() cannot be nested within arrayWith()`);
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.push(this, [], `Missing element${element}at pattern index ${patternIdx}`);
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).push(this, [], `Expected type object but received ${type_1.getType(actual)}`);
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.push(this, [`/${a}`], 'Unexpected key');
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (!(patternKey in actual) && !(patternVal instanceof absent_1.AbsentMatch)) {
                result.push(this, [`/${patternKey}`], 'Missing key');
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.push(this, [], `Expected JSON as a string but found ${type_1.getType(actual)}`);
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.push(this, [], `Invalid JSON string: ${actual}`);
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.push(this, [], `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`);
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.push(this, [], 'Expected a value but found none');
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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