import { IDependable } from './dependency';
import { MetadataEntry } from './metadata';
/**
 * Represents a construct.
 *
 * @stability stable
 */
export interface IConstruct extends IDependable {
    /**
     * The tree node.
     *
     * @stability stable
     */
    readonly node: Node;
}
/**
 * Represents the construct node in the scope tree.
 *
 * @stability stable
 */
export declare class Node {
    private readonly host;
    /**
     * Separator used to delimit construct path components.
     *
     * @stability stable
     */
    static readonly PATH_SEP = "/";
    /**
     * (deprecated) Returns the node associated with a construct.
     *
     * @param construct the construct.
     * @deprecated use `construct.node` instead
     */
    static of(construct: IConstruct): Node;
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     *
     * @stability stable
     */
    readonly scope?: IConstruct;
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     *
     * @stability stable
     */
    readonly id: string;
    private _locked;
    private readonly _children;
    private readonly _context;
    private readonly _metadata;
    private readonly _dependencies;
    private _defaultChild;
    private readonly _validations;
    private _addr?;
    /**
     * @stability stable
     */
    constructor(host: Construct, scope: IConstruct, id: string);
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     *
     * @stability stable
     */
    get path(): string;
    /**
     * Returns an opaque tree-unique address for this construct.
     *
     * Addresses are 42 characters hexadecimal strings. They begin with "c8"
     * followed by 40 lowercase hexadecimal characters (0-9a-f).
     *
     * Addresses are calculated using a SHA-1 of the components of the construct
     * path.
     *
     * To enable refactorings of construct trees, constructs with the ID `Default`
     * will be excluded from the calculation. In those cases constructs in the
     * same tree may have the same addreess.
     *
     * @stability stable
     * @example
     *
     * c83a2846e506bcc5f10682b564084bca2d275709ee
     */
    get addr(): string;
    /**
     * Return a direct child by id, or undefined.
     *
     * @param id Identifier of direct child.
     * @returns the child if found, or undefined
     * @stability stable
     */
    tryFindChild(id: string): IConstruct | undefined;
    /**
     * Return a direct child by id.
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child.
     * @returns Child with the given id.
     * @stability stable
     */
    findChild(id: string): IConstruct;
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     *
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     *
     * @returns a construct or undefined if there is no default child
     * @stability stable
     * @throws if there is more than one child
     */
    get defaultChild(): IConstruct | undefined;
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     *
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     *
     * @returns a construct or undefined if there is no default child
     * @stability stable
     * @throws if there is more than one child
     */
    set defaultChild(value: IConstruct | undefined);
    /**
     * All direct children of this construct.
     *
     * @stability stable
     */
    get children(): IConstruct[];
    /**
     * Return this construct and all of its children in the given order.
     *
     * @stability stable
     */
    findAll(order?: ConstructOrder): IConstruct[];
    /**
     * This can be used to set contextual values.
     *
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     *
     * @param key The context key.
     * @param value The context value.
     * @stability stable
     */
    setContext(key: string, value: any): void;
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key.
     * @returns The context value or `undefined` if there is no context value for thie key.
     * @stability stable
     */
    tryGetContext(key: string): any;
    /**
     * An immutable array of metadata objects associated with this construct.
     *
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     *
     * @stability stable
     */
    get metadata(): MetadataEntry[];
    /**
     * Adds a metadata entry to this construct.
     *
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata.
     * @param data the value of the metadata (can be a Token).
     * @param options options.
     * @stability stable
     */
    addMetadata(type: string, data: any, options?: MetadataOptions): void;
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     * @stability stable
     */
    get scopes(): IConstruct[];
    /**
     * Returns the root of the construct tree.
     *
     * @returns The root of the construct tree.
     * @stability stable
     */
    get root(): IConstruct;
    /**
     * Returns true if this construct or the scopes in which it is defined are locked.
     *
     * @stability stable
     */
    get locked(): boolean;
    /**
     * Add an ordering dependency on another construct.
     *
     * An `IDependable`
     *
     * @stability stable
     */
    addDependency(...deps: IDependable[]): void;
    /**
     * Return all dependencies registered on this node (non-recursive).
     *
     * @stability stable
     */
    get dependencies(): IConstruct[];
    /**
     * (experimental) Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName: string): boolean;
    /**
     * Adds a validation to this construct.
     *
     * When `node.validate()` is called, the `validate()` method will be called on
     * all validations and all errors will be returned.
     *
     * @param validation The validation object.
     * @stability stable
     */
    addValidation(validation: IValidation): void;
    /**
     * Validates this construct.
     *
     * Invokes the `validate()` method on all validations added through
     * `addValidation()`.
     *
     * @returns an array of validation error messages associated with this
     * construct.
     * @stability stable
     */
    validate(): string[];
    /**
     * Locks this construct from allowing more children to be added.
     *
     * After this
     * call, no more children can be added to this construct or to any children.
     *
     * @stability stable
     */
    lock(): void;
    /**
     * Adds a child construct to this node.
     *
     * @param child The child construct
     * @param childName The type name of the child construct.
     * @returns The resolved path part name of the child
     */
    private addChild;
}
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 *
 * @stability stable
 */
export declare class Construct implements IConstruct {
    /**
     * (deprecated) Checks if `x` is a construct.
     *
     * @param x Any object.
     * @returns true if `x` is an object created from a class which extends `Construct`.
     * @deprecated use `x instanceof Construct` instead
     */
    static isConstruct(x: any): x is Construct;
    /**
     * The tree node.
     *
     * @stability stable
     */
    readonly node: Node;
    /**
     * Creates a new construct node.
     *
     * @param scope The scope in which to define this construct.
     * @param id The scoped construct ID.
     * @stability stable
     */
    constructor(scope: Construct, id: string);
    /**
     * Returns a string representation of this construct.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Implement this interface in order for the construct to be able to validate itself.
 *
 * Implement this interface in order for the construct to be able to validate itself.
 *
 * @stability stable
 */
export interface IValidation {
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     * @stability stable
     */
    validate(): string[];
}
/**
 * In what order to return constructs.
 *
 * @stability stable
 */
export declare enum ConstructOrder {
    /**
     * Depth-first, pre-order.
     *
     * @stability stable
     */
    PREORDER = 0,
    /**
     * Depth-first, post-order (leaf nodes first).
     *
     * @stability stable
     */
    POSTORDER = 1
}
/**
 * Implement this interface in order for the construct to be able to validate itself.
 *
 * Implement this interface in order for the construct to be able to validate itself.
 *
 * @stability stable
 */
export interface IValidation {
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     * @stability stable
     */
    validate(): string[];
}
/**
 * Options for `construct.addMetadata()`.
 *
 * @stability stable
 */
export interface MetadataOptions {
    /**
     * Include stack trace with metadata entry.
     *
     * @default false
     * @stability stable
     */
    readonly stackTrace?: boolean;
    /**
     * A JavaScript function to begin tracing from.
     *
     * This option is ignored unless `stackTrace` is `true`.
     *
     * @default addMetadata()
     * @stability stable
     */
    readonly traceFromFunction?: any;
}
