# AUTOGENERATED! DO NOT EDIT! File to edit: ../04_gui.ipynb.

# %% auto 0
__all__ = ['launch_gui', 'GUI', 'PlainTab', 'StatisticsTab', 'PlotTab', 'ConfigsTab']

# %% ../04_gui.ipynb 3
from typing import List, Tuple, Dict, Optional, Union
from abc import ABC, abstractmethod

import pandas as pd
from pathlib import Path
from ipywidgets.widgets.widget_box import VBox
import ipywidgets as w
from IPython.display import display
import inspect
import itertools
import pickle
import os

from .main import Session
from .database import Configs
from . import stats
from . import plots

# %% ../04_gui.ipynb 4
def launch_gui():
    display(GUI().widget)

# %% ../04_gui.ipynb 5
class GUI:
    
    def __init__(self):
        self.uploader = w.FileUpload(accept=('.xlsx, .csv'), multiple=False, layout = {'width': '15%'})
        available_statistical_tests = self.get_available_statistical_tests()
        self.stats_selection = w.Dropdown(options = available_statistical_tests, layout = {'width': '60%'})
        self.confirm_stats_selection_button = w.Button(description = 'confirm selection and compute stats', layout = {'width': '25%'})
        user_info_0 = w.Label(value = 'Please select a file (.xlsx or .csv) you would like to upload,\
                             choose the appropriate statistical test & then confirm your selection to get started:')
        self.widget = w.VBox([user_info_0,
                              w.HBox([self.uploader, self.stats_selection, self.confirm_stats_selection_button])])
        
        self.confirm_stats_selection_button.on_click(self.build_and_change_to_tabs_ui)
        
        
    def get_available_statistical_tests(self) -> List[Tuple[stats.StatisticalTest, str]]:
        available_statistical_tests = [] # list of tuples like (description_string, stats_test_obj)
        for name, obj in inspect.getmembers(stats):
            if name.endswith('Stats'):
                available_statistical_tests.append((obj().name_displayed_in_gui, obj))
        return available_statistical_tests
    
    
    def build_and_change_to_tabs_ui(self, b) -> None:
        self.session = Session()
        self.session.upload_data_via_gui(uploader_value = self.uploader.value)
        
        
        self.stats_tab = StatisticsTab(gui = self)
        self.plot_tab = PlotTab(gui = self)
        self.configurations_tab = ConfigsTab(gui = self)
        #self.io_tab = IOTab(gui = self)
        
        tabs_ui = w.Tab([self.stats_tab.widget, self.plot_tab.widget, self.configurations_tab.widget])
        tabs_ui.set_title(0, 'statistics corner')
        tabs_ui.set_title(1, 'generated plot')
        tabs_ui.set_title(2, 'customize your plot')
        
        # might require some initalization of configs?
        self.plot_tab.update_plot.click()
        
        self.widget.children = (tabs_ui, )
        

# %% ../04_gui.ipynb 6
class PlainTab(ABC):
    
    def __init__(self, gui: GUI) -> None:
        self.gui = gui
        self.widget = self.create_widget()
        
    
    @abstractmethod
    def create_widget(self) -> VBox:
        pass

# %% ../04_gui.ipynb 7
class StatisticsTab(PlainTab):
    
    def create_widget(self) -> VBox:
        user_information = w.VBox([])
        self.display_stats_df = w.Output()
        self.export_stats = w.Button(description = 'export statistical results', layout = {'width': '25%'})
        widget = w.VBox([user_information, self.display_stats_df, self.export_stats])
        self.export_stats.on_click(self.export_stats_results)
        with self.display_stats_df:
            self.gui.session.calculate_stats(statistical_test = self.gui.stats_selection.value)
            user_information_strings = self.create_user_information_strings()
            user_information_labels = []
            for user_info in user_information_strings:
                user_information.children += (w.Label(value = user_info), )
            if type(self.gui.session.database.stats_results['pairwise_comparisons']) == pd.DataFrame:
                display(self.gui.session.database.stats_results['pairwise_comparisons'])
            else:
                display(self.gui.session.database.stats_results['summary_stats']['full_test_results'])
        return widget
        
                                        
    def create_user_information_strings(self) -> List[str]:
        user_info_strings = []
        summary_stats = self.gui.session.database.stats_results['summary_stats'].copy()
        performed_test = summary_stats['performed_test']
        summary_p_value = summary_stats['p_value']
        summary_stars_str = summary_stats['stars_str']
        user_info_strings.append(f'I used a {performed_test} to analyze your data \
                                 and the resulting p-value is: {round(summary_p_value, 4)}.')
        if summary_p_value <= 0.05:
            user_info_strings.append('Thus, there is a *significant* difference in your data. \
                                     See the table below for more detailed information:')
        else:
            user_info_strings.append('Thus, there is *no* significant difference in your data. \
                                     Unfortunately, this also means that you have to')
            user_info_strings.append('ignore the results of pairwise comparisons that may be listed below, \
                                     even if they show p-values <= 0.05')
        return user_info_strings
    
    
    def export_stats_results(self, b) -> None:
        self.gui.session.export_stats_results()

# %% ../04_gui.ipynb 8
class PlotTab(PlainTab):
    
    def create_widget(self) -> VBox:
        available_plot_types = self.get_available_plot_types()
        self.select_plot_type = w.Dropdown(description = 'Please select a plot type: ',
                                           options = available_plot_types, 
                                           layout = {'width': '40%'},
                                           style = {'description_width': 'initial'})
        self.update_plot = w.Button(description = 'update the plot', layout = {'width': '20%'})
        self.export_plot = w.Button(description = 'export the plot', layout = {'width': '20%'})
        self.select_export_type = w.Dropdown(description = 'Export filetype: ',
                                             options = ['png', 'pdf', 'svg'], 
                                             layout = {'width': '35%'},
                                             style = {'description_width': 'initial'})        
        self.export_dpi = w.IntSlider(description = 'Resolution (dpi): ', min = 100, max = 1200, value = 300, step = 50,
                                      layout = {'width': '60%'},
                                      style = {'description_width': 'initial'})
        self.export_filename = w.Text(description = 'Filename (without filetpye extension): ',
                                      value = 'customized_plot',
                                      placeholder = 'Use no extension like .png or .pdf here!',
                                      layout = {'width': '95%'},
                                      style = {'description_width': 'initial'})
        export_accordion = w.Accordion(children = [w.VBox([w.HBox([self.select_export_type, self.export_dpi]), self.export_filename])],
                                      selected_index = None,
                                      layout = {'width': '80%'})
        export_accordion.set_title(0, 'Additional export settings')
        self.display_plot = w.Output()
        self.update_plot.on_click(self.update_the_plot)
        self.export_plot.on_click(self.export_the_plot)
        widget = w.VBox([w.HBox([self.select_plot_type, self.update_plot, self.export_plot]),
                         export_accordion,
                         self.display_plot])
        return widget
    
    
    def get_available_plot_types(self) -> List[str]:
        return self.gui.stats_selection.value().plot_handler().plot_options_displayed_in_gui        
        
    
    def update_the_plot(self, b):
        self.gui.session.database.configs = self.gui.configurations_tab.update_configs()
        with self.display_plot:
            self.display_plot.clear_output()
            self.gui.session.create_plot()
            

    def export_the_plot(self, b):
        self.gui.session.database.configs = self.gui.configurations_tab.update_configs()
        current_working_directory = Path(os.getcwd())
        export_filepath = current_working_directory.joinpath(f'{self.export_filename.value}.{self.select_export_type.value}')
        self.gui.session.create_plot(filepath = export_filepath, dpi = self.export_dpi.value, show = False, save = True)

# %% ../04_gui.ipynb 9
class ConfigsTab(PlainTab):
    
    def create_widget(self):
        user_info0 = w.Label(value = 'Please use the interactive widgets below to customize the plot to your needs.')
        user_info1 = w.Label(value = 'You also have the option to export your current settings,\
                                      in order to upload & set them again in a subsequent session.')
        user_information = w.VBox([user_info0, user_info1])
        self.export_configs = w.Button(description = 'export your current settings', layout = {'width': '25%'})
        self.upload_configs = w.FileUpload(accept=('.p'), multiple=False, layout = {'width': '15%'})
        self.set_uploaded_configs = w.Button(description = 'update to uploaded settings', layout = {'width': '25%'})
        
        stats_annotation_accordion_tab = self.initialize_stats_annotation_accordion_tab()
        axes_accordion_tab = self.initialize_axes_accordion_tab()
        additional_configs_accordion_tab = self.initialize_additional_configs_accordion_tab()
        
        accordion = w.Accordion([stats_annotation_accordion_tab, axes_accordion_tab, additional_configs_accordion_tab])
        accordion.set_title(0, 'Customize how statistics are annotated in the plot')
        accordion.set_title(1, 'Customize axes')
        accordion.set_title(2, 'Customize additional other features of the plot')
        
        widget = w.VBox([user_information,
                         w.HBox([self.export_configs, self.upload_configs, self.set_uploaded_configs]),
                         accordion])
        
        self.export_configs.on_click(self.export_current_settings)
        self.set_uploaded_configs.on_click(self.extract_and_set_uploaded_configs)

        return widget


    def initialize_stats_annotation_accordion_tab(self):
        select_annotations_vbox = self.initialize_select_annotations_vbox()
        customize_annotations_vbox = self.initialize_customize_annotations_vbox()
        annotations_accordion = w.Accordion([select_annotations_vbox, customize_annotations_vbox])
        annotations_accordion.set_title(0, 'Select which stats shall be annotated')
        annotations_accordion.set_title(1, 'Customize annotations')
        return w.VBox([annotations_accordion])

    
    def initialize_select_annotations_vbox(self) -> VBox:
        user_information = w.Label(value = 'Please note: if "Annotate all" is checked, individual selections will be ignored.')
        self.annotate_all = w.Checkbox(value = False, description = 'Annotate all', indent = False)
        stats_results = self.gui.session.database.stats_results.copy()
        if stats_results['summary_stats']['performed_test'] == 'Mixed-model ANOVA':
            self._mma = True
            l_all_sorted_checkboxes, self._pw_comparison_checkboxes = self.create_all_pw_comparison_checkboxes_mma(stats_results = stats_results)
            select_individual_pw_comparisons_accordion = w.Accordion(l_all_sorted_checkboxes)
            for i in range(len(stats_results['df_infos']['all_session_ids'])):
                select_individual_pw_comparisons_accordion.set_title(i, stats_results['df_infos']['all_session_ids'][i])
        else:
            self._mma = False
            l_checkboxes_sorted_into_hboxes, self._pw_comparison_checkboxes = self.create_all_pw_comparison_checkboxes(stats_results = stats_results)
            select_individual_pw_comparisons_accordion = w.Accordion([w.VBox(l_checkboxes_sorted_into_hboxes)])
            select_individual_pw_comparisons_accordion.set_title(0, 'Select individual comparisons for annotation')
        return w.VBox([user_information, self.annotate_all, select_individual_pw_comparisons_accordion])
    
    
    def create_all_pw_comparison_checkboxes(self, stats_results: Dict) -> Tuple[List, List]:
        l_groups = stats_results['df_infos']['all_group_ids']
        if len(l_groups) == 1:
            fixed_value_colum_name = stats_results['df_infos']['fixed_value_column_name']
            l_checkboxes = [w.Checkbox(value = False, description = f'{l_groups[0]} vs. {fixed_value_colum_name}')]
        else:
            # Create a checkbox for each pairwise comparison
            l_checkboxes = [w.Checkbox(value = False, description = f'{group1} vs. {group2}')
                            for group1, group2 in list(itertools.combinations(l_groups, 2))]
        l_checkboxes_sorted_into_hboxes = self.assign_checkboxes_to_rowwise_hboxes(l_checkboxes = l_checkboxes, checkboxes_per_row = 3)
        return l_checkboxes_sorted_into_hboxes, l_checkboxes
    
    
    def assign_checkboxes_to_rowwise_hboxes(self, l_checkboxes: List, checkboxes_per_row: int) -> List:
        # Arrange checkboxes in a HBoxes with up to 3 checkboxes per HBox
        l_hboxes = []
        elem = 0
        for i in range(int(len(l_checkboxes)/checkboxes_per_row)):
            l_hboxes.append(w.HBox(l_checkboxes[elem:elem+checkboxes_per_row]))
            elem = elem + checkboxes_per_row
        if len(l_checkboxes) % checkboxes_per_row != 0:
            l_hboxes.append(w.HBox(l_checkboxes[elem:]))
        return l_hboxes
    
    
    def create_all_pw_comparison_checkboxes_mma(self, stats_results: Dict) -> List:
        l_sessions = stats_results['df_infos']['all_session_ids']
        l_all_checkboxes = []
        l_all_sorted_checkboxes = []
        for session_id in l_sessions:
            l_checkboxes_sorted_into_hboxes, l_checkboxes = self.create_all_pw_comparison_checkboxes(stats_results = stats_results)
            l_checkboxes = [(session_id, checkbox) for checkbox in l_checkboxes]
            l_all_checkboxes += l_checkboxes
            l_all_sorted_checkboxes.append(w.VBox(l_checkboxes_sorted_into_hboxes))
        return l_all_sorted_checkboxes, l_all_checkboxes

        
        
    def initialize_customize_annotations_vbox(self):
        self.distance_brackets_to_data = w.BoundedFloatText(description = 'Distance of the annotation bars to the graph',
                                                            value = 0.1, min = 0, max = 1, step = 0.005,
                                                            style = {'description_width': 'initial'},
                                                            layout = {'width':'75%'})
        self.annotation_brackets_factor = w.RadioButtons(description='Annotation bar style:',
                                                         options=[('no brackets', 0), ('brackets', 1)],
                                                         value=1,
                                                         style={'description_width': 'initial'},
                                                         layout={'width': '50%', 'height': '50px'})
        self.distance_stars_to_brackets = w.BoundedFloatText(description = 'Distance of the stars to the annotation bars',
                                                             value = 0.5, step = 0.05, min = 0, max = 3,
                                                             style = {'description_width': 'initial'},
                                                             layout = {'width':'75%'})
        self.fontsize_stars = w.BoundedFloatText(description='Fontsize of the stars',
                                                 value = 10, min = 1, max = 50,
                                                 style = {'description_width': 'initial'},
                                                 layout = {'width': '75%'})
        self.linewidth_annotations = w.BoundedFloatText(description = 'Linewidth of the annotation bars',
                                                        value = 1.5, min = 0, max = 10, step = 0.1,
                                                        layout = {'width':'initial'},
                                                        style = {'description_width': 'initial'})
        self.fontweight_stars = w.Checkbox(description = 'Stars bold', value = False, layout = {'width': '25%'})
        row_0 = w.HBox([self.fontweight_stars, self.annotation_brackets_factor])
        row_1 = self.distance_stars_to_brackets
        row_2 = self.distance_brackets_to_data
        row_3 = self.fontsize_stars
        row_4 = self.linewidth_annotations
        return w.VBox([row_0, row_1, row_2, row_3, row_4])


    def initialize_axes_accordion_tab(self):
        x_axis_vbox = self.initialize_x_axis_vbox()
        y_axis_vbox = self.initialize_y_axis_vbox()
        common_axis_features_vbox = self.initialize_common_axis_features_vbox()
        accordion = w.Accordion([x_axis_vbox, y_axis_vbox, common_axis_features_vbox])
        accordion.set_title(0, 'Customize the x axis')
        accordion.set_title(1, 'Customize the y axis')
        accordion.set_title(2, 'Customize general axis features')
        return w.VBox([accordion])

    
    def initialize_x_axis_vbox(self) -> VBox:
        initial_xlabel_order, initial_hue_order = self.create_group_order_text()
        self.l_xlabel_order = w.Text(value = initial_xlabel_order,
                                     placeholder = 'Specify the desired order of the x-axis \
                                     labels with individual labels separated by a comma',
                                     description = 'x-axis label order (separated by comma):',
                                     layout = {'width': '90%'},
                                     style = {'description_width': 'initial'})
        self.xaxis_label_text = w.Text(value = 'group_IDs', placeholder = 'group_IDs', 
                                       description = 'x-axis title:', layout = {'width': '36%'})
        self.xaxis_label_fontsize = w.IntSlider(value = 12, min = 6, max = 30, 
                                                step = 1, description = 'fontsize:', layout = {'width': '28%'})
        self.xaxis_label_color = w.ColorPicker(concise = False, description = 'font color', 
                                               value = '#000000', layout = {'width': '28%'})
        if initial_hue_order == '':
            visibility = 'hidden'
        else:
            visibility = 'visible'
        self.l_hue_order = w.Text(value = initial_hue_order,
                                 placeholder = 'Specify the desired group order with individual groups separated by a comma (MMA only)',
                                 description = 'group order (separated by comma):',
                                 layout = {'width': '90%', 'visibility': visibility},
                                 style = {'description_width': 'initial'})
        row_0 = w.HBox([self.xaxis_label_text, self.xaxis_label_fontsize, self.xaxis_label_color])
        row_1 = self.l_xlabel_order
        row_2 = self.l_hue_order
        return w.VBox([row_0, row_1, row_2])


    def create_group_order_text(self):
        df_infos = self.gui.session.database.stats_results['df_infos'].copy()
        l_groups = df_infos['all_group_ids']
        if self._mma == False:
            initial_hue_order = ''
            initial_xlabel_order = ''
            for group_id in l_groups:
                if group_id != l_groups[-1]:
                    initial_xlabel_order += f'{group_id}, '
                else:
                    initial_xlabel_order += f'{group_id}'
        else:
            initial_hue_order = ''
            initial_xlabel_order = ''
            for session_id in df_infos['all_session_ids']:
                if session_id != df_infos['all_session_ids'][-1]:
                    initial_xlabel_order += f'{session_id}, '
                else:
                    initial_xlabel_order += f'{session_id}'        
            for group_id in l_groups:
                if group_id != l_groups[-1]:
                    initial_hue_order += f'{group_id}, '
                else:
                    initial_hue_order += f'{group_id}'                
        return initial_xlabel_order, initial_hue_order
    
    
    def initialize_y_axis_vbox(self) -> VBox:
        self.yaxis_label_text = w.Text(value = 'data', placeholder = 'data', description = 'y-axis title:', 
                                       layout = {'width': '36%'})
        self.yaxis_label_fontsize = w.IntSlider(value = 12, min = 6, max = 30, step = 1,
                                                description = 'fontsize:', layout = {'width': '28%'})
        self.yaxis_label_color = w.ColorPicker(concise = False, description = 'font color', 
                                               value = '#000000', layout = {'width': '28%'})
        self.yaxis_scaling_mode = w.RadioButtons(description = 'Please select whether you want to use \
                                                 automatic or manual scaling of the yaxis:',
                                                 options = [('Use automatic scaling', 'auto'), 
                                                            ('Use manual scaling', 'manual')],
                                                 value = 'auto', layout = {'width': '90%', 'height': '75px'}, 
                                                 style = {'description_width': 'initial'})
        data_column_values = self.gui.session.database.stats_results['df_infos']['data_column_values']
        if data_column_values.min() < 0:
            yaxis_lower_lim_value = round(data_column_values.min() + data_column_values.min()*0.1, 2)
        else:
            yaxis_lower_lim_value = round(data_column_values.min() - data_column_values.min()*0.1, 2)
        if data_column_values.max() < 0:
            yaxis_upper_lim_value = round(data_column_values.max() - data_column_values.max()*0.1, 2)
        else:
            yaxis_upper_lim_value = round(data_column_values.max() + data_column_values.max()*0.1, 2)
        self.yaxis_lower_lim = w.FloatText(value = yaxis_lower_lim_value, description = 'lower limit:', 
                                           style = {'description_width': 'initial'}, layout = {'width': '45%'})
        self.yaxis_upper_lim = w.FloatText(value = yaxis_upper_lim_value, description = 'upper limit:', 
                                           style = {'description_width': 'initial'}, 
                                           layout = {'width': '45%'})
        row_0 = w.HBox([self.yaxis_label_text, self.yaxis_label_fontsize, self.yaxis_label_color])
        row_1 = self.yaxis_scaling_mode
        row_2 = w.HBox([self.yaxis_lower_lim, self.yaxis_upper_lim])
        return w.VBox([row_0, row_1, row_2])
    
    
    def initialize_common_axis_features_vbox(self):
        
        self.axes_linewidth = w.BoundedFloatText(value = 1, min = 0, max = 5, step = 0.1, description = 'Axes linewidth',
                                                 style = {'description_width': 'initial'}, layout = {'width': '30%'})
        self.axes_color = w.ColorPicker(concise = False, description = 'Axes and tick label color',
                                        value = '#000000', style = {'description_width': 'initial'}, 
                                        layout={'width': '30%'})
        self.axes_tick_size = w.BoundedFloatText(value = 10, min = 1, max = 30, step = 0.5, description = 'Tick label size',
                                                style = {'description_width': 'initial'}, layout = {'width': '30%'})
        row_0 = w.HBox([self.axes_linewidth, self.axes_color, self.axes_tick_size])
        return w.VBox([row_0])
        
        

    def initialize_additional_configs_accordion_tab(self) -> VBox:
        self.fig_width = w.FloatSlider(value=28, min=1, max=50, description='Figure width:', 
                                       style={'description_width': 'inital'})
        self.fig_height = w.FloatSlider(value=16, min=1, max=50, description='Figure height:', 
                                        style={'description_width': 'inital'})
        self._preset_color_palette = w.Dropdown(options = ['colorblind', 'Spectral', 'viridis', 'rocket', 'cubehelix'],
                                                value = 'viridis',
                                                description = 'Select a color palette',
                                                layout = {'width': '35%'},
                                                style = {'description_width': 'initial'})
        color_pickers = []
        for group_id in self.gui.session.database.stats_results['df_infos']['all_group_ids']:
            color_pickers.append(w.ColorPicker(concise = False, description = group_id, style = {'description_width': 'initial'}))
        self._custom_color_palette = w.VBox(color_pickers)
        self.color_palette = w.RadioButtons(description = 'Please select a color code option and chose from the respective options below:',
                                            options = [('Use a pre-defined palette', 'preset'), 
                                                       ('Define colors individually', 'custom')],
                                            value = 'preset', layout = {'width': '80%', 'height': '75px'}, 
                                            style = {'description_width': 'initial'})
        user_info = w.Label(value = 'Below, you can find several additional customization features that are, however, applicable only for certain plots:')
        self.show_legend = w.Checkbox(value=True, description='Show legend (MMA plots):', 
                                      style={'description_width': 'initial'})
        self.marker_size = w.FloatText(value=5,description='marker size (any stripplot):', 
                                       style={'description_width': 'initial'})
        self.boxplot_linewidth = w.BoundedFloatText(value = 1.5, min = 0, max = 10, step = 0.5,
                                                    description = 'Boxplot linewidth:',
                                                    style = {'description_width': 'initial'})
        self.boxplot_width = w.BoundedFloatText(value = 0.8, min = 0, max = 1, step = 0.1, 
                                                description = 'Boxplot width:',
                                                style = {'description_width': 'initial'})
        self.rm_linewidth = w.BoundedFloatText(value = 0.5, min = 0, max = 10, step = 0.1,
                                               description = 'Line width (repeated measurements):',
                                               style = {'description_width': 'initial'})
        self.rm_linestyle = w.Dropdown(options = ['solid', 'dashed', 'dashdot', 'dotted', 'none'],
                                       value = 'dashed',
                                       description = 'Line style (repeated measurements):',
                                       style = {'description_width': 'initial'})
        self.rm_linecolor = w.ColorPicker(concise = False, description = 'Line color (repeated measurements)', style = {'description_width': 'initial'})
        self.rm_alpha = w.BoundedFloatText(value = 0.8, min = 0, max = 1, step = 0.05,
                                           description = 'Line transparency (repeated measurements):',
                                           style = {'description_width': 'initial'})
        
        row_0 = w.HBox([self.fig_height, self.fig_width])
        row_1 = w.VBox([self.color_palette, 
                        w.HBox([self._preset_color_palette, self._custom_color_palette])])
        row_2 = user_info
        row_3 = w.HBox([self.show_legend, self.marker_size])
        row_4 = w.HBox([self.boxplot_width, self.boxplot_linewidth])
        row_5 = w.HBox([self.rm_linewidth, self.rm_linestyle])
        row_6 = w.HBox([self.rm_linecolor, self.rm_alpha])
        return w.VBox([row_0, row_1, row_2, row_3, row_4, row_5, row_6])
        
                                             
    def export_current_settings(self, b) -> None:
        all_attributes = self.__dict__.copy()
        keys_to_pop = ['l_xlabel_order']
        for key in all_attributes.keys():
            if key not in self.gui.session.database.configs.__dict__.keys():
                keys_to_pop.append(key)
        for key in keys_to_pop:
            all_attributes.pop(key)
        exportable_configs = dict()
        for key, value in all_attributes.items():
            exportable_configs[key] = value.value
        with open('configs.p', 'wb') as io:
            pickle.dump(exportable_configs, io)
    
    
    def extract_and_set_uploaded_configs(self, b) -> None:
        with open('uploaded_configs.p', 'w+b') as io:
            io.write(self.upload_configs.data[0])
        with open('uploaded_configs.p', 'rb') as i:
            imported_configs = pickle.load(i)
        for key, value in imported_configs.items():
            if key not in ['l_xlabel_order']:
                widget = self.__dict__[key]
                widget.value = value
        
        
    def update_configs(self) -> Configs:
        configs = self.gui.session.database.configs
        setattr(configs, 'plot_type', self.gui.plot_tab.select_plot_type.value)
        for attr in self.__dict__.keys():
            if attr == 'l_xlabel_order':
                l_xlabel_order = self.handle_l_xlabel_order_exception()
                setattr(configs, 'l_xlabel_order', l_xlabel_order)
            elif attr == 'l_hue_order':
                l_hue_order = self.handle_hue_order_exception()
                setattr(configs, 'l_hue_order', l_hue_order)
            elif attr == 'color_palette':
                color_palette = self.handle_color_palette_exception()
                setattr(configs, 'color_palette', color_palette)
            elif attr == 'annotate_all':
                l_stats_to_annotate = self.handle_annotate_all_exception()
                setattr(configs, 'l_stats_to_annotate', l_stats_to_annotate)
            elif attr == 'fontweight_stars':
                if self.fontweight_stars:
                    fontweight_stars = 'bold'
                else:
                    fontweight_stars = 'normal'
                setattr(configs, 'fontweight_stars', fontweight_stars)
            elif hasattr(configs, attr):
                widget = getattr(self, attr)
                value = widget.value
                setattr(configs, attr, value)
        return configs
        
    
    
    def handle_color_palette_exception(self) -> Union[str, Dict]:
        if self.color_palette.value == 'preset':
            color_palette = self._preset_color_palette.value
        else:
            color_palette = dict()
            for group_id, color_picker in zip(self.gui.session.database.stats_results['df_infos']['all_group_ids'], 
                                              self._custom_color_palette.children):
                color_palette[group_id] = color_picker.value
        return color_palette
        
    
    def handle_l_xlabel_order_exception(self) -> List[str]:
        l_xlabel_order = []
        l_xlabel_string = self.l_xlabel_order.value
        if ',' in l_xlabel_string:
            while ', ' in l_xlabel_string:
                l_xlabel_order.append(l_xlabel_string[:l_xlabel_string.index(', ')])
                l_xlabel_string = l_xlabel_string[l_xlabel_string.index(', ')+2:]
            l_xlabel_order.append(l_xlabel_string)
        else:
            l_xlabel_order.append(self.l_xlabel_order.value)
        return l_xlabel_order


    def handle_hue_order_exception(self) -> List[str]:
        l_hue_order = []
        hue_order_string = self.l_hue_order.value
        if ',' in hue_order_string:
            while ', ' in hue_order_string:
                l_hue_order.append(hue_order_string[:hue_order_string.index(', ')])
                hue_order_string = hue_order_string[hue_order_string.index(', ')+2:]
            l_hue_order.append(hue_order_string)
        else:
            l_hue_order.append(self.l_hue_order.value)
        return l_hue_order

    
    def handle_annotate_all_exception(self) -> List[str]:
        l_stats_to_annotate = []
        if self._mma:
            if self.annotate_all.value:
                for i in range(len(self._pw_comparison_checkboxes)):
                    self._pw_comparison_checkboxes[i][1].value = True
            for i in range(len(self._pw_comparison_checkboxes)):
                if self._pw_comparison_checkboxes[i][1].value:
                    checkbox_description = self._pw_comparison_checkboxes[i][1].description
                    group1 = checkbox_description[:checkbox_description.index(' ')]
                    group2 = checkbox_description[checkbox_description.index(' vs. ') + 5 :]
                    session_id = self._pw_comparison_checkboxes[i][0]
                    l_stats_to_annotate.append((group1, group2, session_id))         
        else:
            if self.annotate_all.value == True:
                for i in range(len(self._pw_comparison_checkboxes)):
                    self._pw_comparison_checkboxes[i].value = True
            for i in range(len(self._pw_comparison_checkboxes)):
                if self._pw_comparison_checkboxes[i].value:
                    checkbox_description = self._pw_comparison_checkboxes[i].description
                    group1 = checkbox_description[:checkbox_description.index(' ')]
                    group2 = checkbox_description[checkbox_description.index(' vs. ') + 5 :]
                    l_stats_to_annotate.append((group1, group2))
        return l_stats_to_annotate
