from typing import Iterable, List, Union, cast

import numpy as np
from scipy.stats import norm

from physrisk.kernel.assets import Asset, IndustrialActivity
from physrisk.kernel.hazard_model import HazardDataRequest, HazardDataResponse, HazardParameterDataResponse
from physrisk.kernel.hazards import ChronicHeat
from physrisk.kernel.impact_distrib import ImpactDistrib, ImpactType
from physrisk.kernel.vulnerability_model import VulnerabilityModelBase


class ChronicHeatGZNModel(VulnerabilityModelBase):
    """Model which estiamtes the labour productivity impact based on chronic heat based on the paper "Neidell M,
    Graff Zivin J, Sheahan M,  Willwerth J, Fant C, Sarofim M, et al. (2021) Temperature and work:
    Time allocated to work under varying climate and labor market conditions."
    Average annual work hours are based on USA values reported by the OECD for 2021."""

    def __init__(self, indicator_id: str = "mean_degree_days/above/32c", delta=True):
        super().__init__(
            indicator_id=indicator_id, hazard_type=ChronicHeat, impact_type=ImpactType.disruption
        )  # opportunity to give a model hint, but blank here
        self.time_lost_per_degree_day = 4.671  # This comes from the paper converted to celsius
        self.time_lost_per_degree_day_se = 2.2302  # This comes from the paper converted to celsius
        self.total_labour_hours = 107460
        self.delta = delta

    def get_data_requests(
        self, asset: Asset, *, scenario: str, year: int
    ) -> Union[HazardDataRequest, Iterable[HazardDataRequest]]:
        """Request the hazard data needed by the vulnerability model for a specific asset
        (this is a Google-style doc string)

        Args:
            asset: Asset for which data is requested.
            scenario: Climate scenario of calculation.
            year: Projection year of calculation.

        Returns:
            Single data requests.
        """

        return [
            HazardDataRequest(
                self.hazard_type,
                asset.longitude,
                asset.latitude,
                scenario="historical",
                year=1980,
                indicator_id=self.indicator_id,
            ),
            HazardDataRequest(
                self.hazard_type,
                asset.longitude,
                asset.latitude,
                scenario=scenario,
                year=year,
                indicator_id=self.indicator_id,
            ),
        ]

    def get_impact(self, asset: Asset, data_responses: List[HazardDataResponse]) -> ImpactDistrib:
        """Calcaulate impact (disruption) of asset based on the hazard data returned.

        Args:
            asset: Asset for which impact is calculated.
            data_responses: responses to the hazard data requests generated in get_data_requests.

        Returns:
            Probability distribution of impacts.
        """
        assert isinstance(asset, IndustrialActivity)
        baseline_dd_above_mean, scenario_dd_above_mean = data_responses

        assert isinstance(baseline_dd_above_mean, HazardParameterDataResponse)
        assert isinstance(scenario_dd_above_mean, HazardParameterDataResponse)

        delta_dd_above_mean: float = scenario_dd_above_mean.parameter - baseline_dd_above_mean.parameter * self.delta

        hours_worked = self.total_labour_hours
        fraction_loss_mean = (delta_dd_above_mean * self.time_lost_per_degree_day) / hours_worked
        fraction_loss_std = (delta_dd_above_mean * self.time_lost_per_degree_day_se) / hours_worked

        return get_impact_distrib(fraction_loss_mean, fraction_loss_std, ChronicHeat, ImpactType.disruption)


class ChronicHeatWBGTGZNModel(ChronicHeatGZNModel):
    """Implementation of the WBGT/GZN chronic heat model. This model
    inherits attributes from the ChronicHeatGZN model and estimates the
    results based on applying both GZN and WBGT."""

    def __init__(self, indicator_id: str = "mean_work_loss_high"):
        super().__init__(indicator_id=indicator_id)  # opportunity to give a model hint, but blank here

    def work_type_mapping(self):
        return {"low": ["low", "medium"], "medium": ["medium", "low", "high"], "high": ["high", "medium"]}

    def get_data_requests(
        self, asset: Asset, *, scenario: str, year: int
    ) -> Union[HazardDataRequest, Iterable[HazardDataRequest]]:
        """Request the hazard data needed by the vulnerability model for a specific asset
        (this is a Google-style doc string)

        Args:
            asset: Asset for which data is requested.
            scenario: Climate scenario of calculation.
            year: Projection year of calculation.

        Returns:
            Single or multiple data requests.
        """

        work_type_mapping = self.work_type_mapping()
        assert isinstance(asset, IndustrialActivity)
        # specify hazard data needed. Model string is hierarchical and '/' separated.
        model_gzn = "mean_degree_days/above/32c"
        model_wbgt = "mean_work_loss/"

        asset_types = [type_asset for type_asset in work_type_mapping[asset.type]]
        wbgt_data_requests = []
        for i_asset_types in asset_types:
            wbgt_data_requests.append(
                HazardDataRequest(
                    self.hazard_type,
                    asset.longitude,
                    asset.latitude,
                    scenario="historical",
                    year=2010,
                    indicator_id=model_wbgt + i_asset_types,
                )
            )

            wbgt_data_requests.append(
                HazardDataRequest(
                    self.hazard_type,
                    asset.longitude,
                    asset.latitude,
                    scenario=scenario,
                    year=year,
                    indicator_id=model_wbgt + i_asset_types,
                )
            )

        return [
            HazardDataRequest(
                self.hazard_type,
                asset.longitude,
                asset.latitude,
                scenario="historical",
                year=1980,
                indicator_id=model_gzn,
            ),
            HazardDataRequest(
                self.hazard_type,
                asset.longitude,
                asset.latitude,
                scenario=scenario,
                year=year,
                indicator_id=model_gzn,
            ),
        ] + wbgt_data_requests

    def get_impact(self, asset: Asset, data_responses: List[HazardDataResponse]) -> ImpactDistrib:
        """
        Function to return the impact distribution of the wbgt model.
        """

        assert isinstance(asset, IndustrialActivity)
        wbgt_responses = [cast(HazardParameterDataResponse, r) for r in data_responses[2:]]

        baseline_dd_above_mean = cast(HazardParameterDataResponse, data_responses[0])
        scenario_dd_above_mean = cast(HazardParameterDataResponse, data_responses[1])

        hours_worked = self.total_labour_hours
        fraction_loss_mean_base_gzn = (baseline_dd_above_mean.parameter * self.time_lost_per_degree_day) / hours_worked

        fraction_loss_mean_scenario_gzn = (
            scenario_dd_above_mean.parameter * self.time_lost_per_degree_day
        ) / hours_worked

        fraction_loss_std_base = (baseline_dd_above_mean.parameter * self.time_lost_per_degree_day_se) / hours_worked

        fraction_loss_std_scenario = (
            scenario_dd_above_mean.parameter * self.time_lost_per_degree_day_se
        ) / hours_worked

        baseline_work_ability = (1 - fraction_loss_mean_base_gzn) * (1 - wbgt_responses[0].parameter)
        scenario_work_ability = (1 - fraction_loss_mean_scenario_gzn) * (1 - wbgt_responses[1].parameter)

        # Getting the parameters required for the uniform distribution.
        if asset.type in ["low", "high"]:
            a_historical = (
                wbgt_responses[0].parameter - abs((wbgt_responses[2].parameter - wbgt_responses[0].parameter)) / 2
            )
            b_historical = (
                wbgt_responses[0].parameter + abs((wbgt_responses[2].parameter - wbgt_responses[0].parameter)) / 2
            )
            a_scenario = (
                wbgt_responses[1].parameter - abs((wbgt_responses[3].parameter - wbgt_responses[1].parameter)) / 2
            )
            b_scenario = (
                wbgt_responses[1].parameter + abs((wbgt_responses[3].parameter - wbgt_responses[1].parameter)) / 2
            )
        elif asset.type == "medium":
            a_historical = wbgt_responses[0].parameter - (wbgt_responses[2].parameter - wbgt_responses[0].parameter) / 2
            b_historical = wbgt_responses[0].parameter + (wbgt_responses[4].parameter - wbgt_responses[0].parameter) / 2
            a_scenario = (
                wbgt_responses[1].parameter - abs((wbgt_responses[3].parameter - wbgt_responses[1].parameter)) / 2
            )
            b_scenario = (
                wbgt_responses[1].parameter + abs((wbgt_responses[5].parameter - wbgt_responses[1].parameter)) / 2
            )

        # Estimation of the variance
        variance_historical_uni = ((b_historical - a_historical) ** 2) / 12
        variance_scenario_uni = ((b_scenario - a_scenario) ** 2) / 12

        variance_historical = two_variable_joint_variance(
            (1 - fraction_loss_mean_base_gzn),
            fraction_loss_std_base**2,
            (1 - wbgt_responses[0].parameter),
            variance_historical_uni,
        )
        variance_scenario = two_variable_joint_variance(
            (1 - fraction_loss_mean_scenario_gzn),
            fraction_loss_std_scenario**2,
            (1 - wbgt_responses[1].parameter),
            variance_scenario_uni,
        )

        std_delta = variance_scenario ** (0.5) - variance_historical ** (0.5)

        total_work_loss_delta: float = baseline_work_ability - scenario_work_ability

        return get_impact_distrib(total_work_loss_delta, std_delta, self.hazard_type, self.impact_type)


def two_variable_joint_variance(ex, varx, ey, vary):
    """
    Function to estimate the variance of two uncorrelated variables.
    """
    return varx * vary + varx * (ey**2) + vary * (ex**2)


def get_impact_distrib(
    fraction_loss_mean: float, fraction_loss_std: float, hazard_type: type, impact_type: ImpactType
) -> ImpactDistrib:
    """Calculate impact (disruption) of asset based on the hazard data returned.

    Args:
        fraction_loss_mean: mean of the impact distribution
        fraction_loss_std: standard deviation of the impact distribution
        hazard_type: Hazard Type.
        impact_type: Impact Type.

    Returns:
        Probability distribution of impacts.
    """
    impact_bins = np.concatenate(
        [
            np.linspace(-0.001, 0.001, 1, endpoint=False),
            np.linspace(0.001, 0.01, 9, endpoint=False),
            np.linspace(0.01, 0.1, 10, endpoint=False),
            np.linspace(0.1, 0.999, 10, endpoint=False),
            np.linspace(0.999, 1.001, 2),
        ]
    )

    probs_cumulative = np.vectorize(lambda x: norm.cdf(x, loc=fraction_loss_mean, scale=max(1e-12, fraction_loss_std)))(
        impact_bins
    )
    probs_cumulative[-1] = np.maximum(probs_cumulative[-1], 1.0)
    probs = np.diff(probs_cumulative)

    probs_norm = np.sum(probs)
    prob_differential = 1 - probs_norm
    if probs_norm < 1e-8:
        if fraction_loss_mean <= 0.0:
            probs = np.concatenate((np.array([1.0]), np.zeros(len(impact_bins) - 2)))
        elif fraction_loss_mean >= 1.0:
            probs = np.concatenate((np.zeros(len(impact_bins) - 2), np.array([1.0])))
    else:
        probs[0] = probs[0] + prob_differential

    return ImpactDistrib(hazard_type, impact_bins, probs, impact_type)
