from typing import Callable, Sequence

import numpy as np


class Distribution:
    def __init__(self, mean, std_dev):
        self.mean = mean
        self.std_dev = std_dev


class VulnMatrixProvider:
    __slots__ = ["intensity_bin_centres", "impact_cdfs"]

    def __init__(
        self,
        intensity_bin_centres: np.ndarray,
        impact_cdfs: Sequence[Callable[[np.ndarray], np.ndarray]],
    ):
        """Via to_prob_matrix method, provides the probability that the impact on an asset falls within
        a specified bin, an impact being either a fractional damage or disruption that occurs as a result of a
        hazard event of a given intensity.

        Args:
            intensity_bin_centres (Iterable[float]): The centres of the intensity bins.
            impact_cdfs (Iterable[Callable[[float], float]]): For each intensity bin centre, provides a function
            that takes parameter, d, and returns the probability that the impact is less than d. This is used to
            construct the probability matrix.
        """

        if not np.all(np.diff(intensity_bin_centres) >= 0):
            raise ValueError("intensities must be sorted and increasing")

        if len(intensity_bin_centres) != len(impact_cdfs):
            raise ValueError("one impact_cdf expected for each intensity_bin_centre")

        self.intensity_bin_centres = np.array(intensity_bin_centres)
        self.impact_cdfs = impact_cdfs

    def to_prob_matrix(self, impact_bin_edges: np.ndarray) -> np.ndarray:
        """Return probability matrix, p with dimension (number intensity bins, number impact bins)
        where p[i, j] is the conditional probability that given the intensity falls in bin i, the impact is
        in bin j.

        Args:
            impact_bin_edges (Iterable[float]): Bin edges of the impact bins.

        Returns:
            np.ndarray: Probability matrix.
        """
        # construct a cdf probability matrix at each intensity point
        # the probability is the prob that the impact is greater than the specified
        cdf_matrix = np.empty([len(self.intensity_bin_centres), len(impact_bin_edges)])

        for i, _ in enumerate(self.intensity_bin_centres):
            cdf_matrix[i, :] = self.impact_cdfs[i](impact_bin_edges)  # type: ignore

        prob_matrix = cdf_matrix[:, 1:] - cdf_matrix[:, :-1]

        return prob_matrix
