#!/usr/bin/env python3
# Copyright (c) Meta Platforms, Inc. and affiliates.
# SPDX-License-Identifier: GPL-3.0-or-later

r"""
This script reads a drgn architecture definition file ("arch_foo.defs") and
outputs architecture definition code which must then be processed by
gen_strswitch.py to produce the final "arch_foo.inc" file.

The definition file comprises a list of register definitions which have the
following syntax:

  register-definition ::= (identifier ":")? register-names "\n"
  identifier ::= [a-zA-Z_][a-zA-Z0-9_]*
  register-names ::= string-literal ("," string-literal)*
  string-literal ::= C string literal

Comments start with "#" outside of a string literal; the rest of the line is
ignored. Whitespace other than newlines is not significant. Lines consisting of
only comments and whitespace are ignored.

A register definition denotes that the given names map to the register with the
given identifier (defined by DRGN_ARCH_REGISTER_LAYOUT in "arch_foo.c"). If the
identifier is omitted, then it is assumed to be the same as the first name with
non-alphanumeric characters replaced by "_" and prepended with "_" if the first
character is numeric.

Registers should be defined in the architecture's logical order.

The generated file includes "arch_register_layout.h" and defines three things:

1. An array of register definitions:
   static const struct drgn_register registers[];

2. A name lookup function:
   static const struct drgn_register *register_by_name(const char *name);

3. A macro containing initializers for the "register_layout",
   "dwarf_regno_to_internal", "registers", "num_registers", and
   "register_by_name" members of "struct drgn_architecture_info":
   #define DRGN_ARCHITECTURE_REGISTERS ...
"""

import re
import sys
from typing import Iterator, List, NamedTuple, Sequence, TextIO

from codegen_utils import CodeGenError, c_string_literal, parse_c_string_literal

_TOKEN_REGEXES = (
    ("STRING", r'"(?:[^\"]|\\")*"'),
    ("IDENTIFIER", r"[a-zA-Z_][a-zA-Z0-9_]*"),
    ("COMMA", r","),
    ("COLON", r":"),
    ("NEWLINE", r"\n"),
    ("SPACE", r"[ \t]+"),
    ("COMMENT", r"#.*"),
    ("UNKNOWN", "."),
)
_TOKEN_REGEX = re.compile(
    "|".join(f"(?P<{name}>{regex})" for name, regex in _TOKEN_REGEXES)
)


class _Token(NamedTuple):
    kind: str
    value: str
    lineno: int
    columnno: int


def _tokenize(file: TextIO) -> Iterator[_Token]:
    lineno = 1
    line = ""
    for lineno, line in enumerate(file, 1):
        for match in _TOKEN_REGEX.finditer(line):
            if match.lastgroup != "SPACE" and match.lastgroup != "COMMENT":
                assert match.lastgroup is not None
                yield _Token(match.lastgroup, match.group(), lineno, match.start() + 1)
    yield _Token("EOF", "", lineno, len(line) + 1)


class Register(NamedTuple):
    identifier: str
    names: Sequence[str]


def parse_defs(file: TextIO, filename: str) -> List[Register]:
    def error(message: str) -> None:
        raise CodeGenError(message, filename, token.lineno, token.columnno)

    registers = []
    used_names = set()
    tokens = _tokenize(file)
    while True:
        token = next(tokens)
        if token.kind == "EOF":
            break
        if token.kind == "NEWLINE":
            continue

        if token.kind == "IDENTIFIER":
            identifier = token.value
            token = next(tokens)
            if token.kind != "COLON":
                error(f"expected COLON, got {token.kind}")
            token = next(tokens)
        elif token.kind == "STRING":
            identifier = None
        else:
            error(f"expected IDENTIFIER or STRING, got {token.kind}")

        names = []
        while True:
            if token.kind != "STRING":
                error(f"expected STRING, got {token.kind}")
            name = parse_c_string_literal(token.value)
            if name in used_names:
                error("duplicate register name")
            used_names.add(name)
            names.append(name)

            token = next(tokens)
            if token.kind != "COMMA":
                break
            token = next(tokens)

        if token.kind != "NEWLINE":
            error(f"expected COMMA or NEWLINE, got {token.kind}")

        if identifier is None:
            identifier = re.sub(r"[^a-zA-Z0-9_]", "_", names[0])
            if not re.match(r"^[a-zA-Z_]", identifier):
                identifier = "_" + identifier
        registers.append(Register(identifier, names))
    return registers


def gen_arch_inc_strswitch(in_file: TextIO, in_filename: str, out_file: TextIO) -> None:
    registers = parse_defs(in_file, in_filename)

    out_file.write("/* Generated by libdrgn/build-aux/gen_arch_inc_strswitch.py. */\n")
    out_file.write("\n")
    out_file.write('#include "arch_register_layout.h" // IWYU pragma: export\n')

    out_file.write("\n")
    out_file.write("static const struct drgn_register registers[] = {\n")
    for register in registers:
        out_file.write("\t{\n")
        out_file.write("\t\t.names = (const char * const []){\n")
        for name in register.names:
            out_file.write(f"\t\t\t{c_string_literal(name)},\n")
        out_file.write("\t\t},\n")
        out_file.write(f"\t\t.num_names = {len(register.names)},\n")
        out_file.write(f"\t\t.regno = DRGN_REGISTER_NUMBER({register.identifier}),\n")
        out_file.write("\t},\n")
    out_file.write("};\n")

    out_file.write("\n")
    out_file.write(
        "static const struct drgn_register *register_by_name(const char *name)\n"
    )
    out_file.write("{\n")
    out_file.write("\t@strswitch (name)@\n")
    for i, register in enumerate(registers):
        for name in register.names:
            out_file.write(f"\t@case {c_string_literal(name)}@\n")
            out_file.write(f"\t\treturn &registers[{i}];\n")
    out_file.write(f"\t@default@\n")
    out_file.write(f"\t\treturn NULL;\n")
    out_file.write("\t@endswitch@\n")
    out_file.write("}\n")

    out_file.write("\n")
    out_file.write("#define DRGN_ARCHITECTURE_REGISTERS \\\n")
    out_file.write("\t.register_layout = register_layout, \\\n")
    out_file.write("\t.dwarf_regno_to_internal = dwarf_regno_to_internal, \\\n")
    out_file.write("\t.registers = registers, \\\n")
    out_file.write(f"\t.num_registers = {len(registers)}, \\\n")
    out_file.write("\t.register_by_name = register_by_name\n")


if __name__ == "__main__":
    try:
        gen_arch_inc_strswitch(sys.stdin, "-", sys.stdout)
    except CodeGenError as e:
        sys.exit(e)
