"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("@aws-cdk/aws-iam");
const cache_policy_1 = require("../cache-policy");
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            cachePolicyId: ((_c = this.props.cachePolicy) !== null && _c !== void 0 ? _c : cache_policy_1.CachePolicy.CACHING_OPTIMIZED).cachePolicyId,
            compress: (_d = this.props.compress) !== null && _d !== void 0 ? _d : true,
            originRequestPolicyId: (_e = this.props.originRequestPolicy) === null || _e === void 0 ? void 0 : _e.originRequestPolicyId,
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_f = this.props.viewerProtocolPolicy) !== null && _f !== void 0 ? _f : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: (_g = this.props.edgeLambdas) === null || _g === void 0 ? void 0 : _g.map(edgeLambda => ({
                lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                eventType: edgeLambda.eventType.toString(),
                includeBody: edgeLambda.includeBody,
            })),
            trustedKeyGroups: (_h = this.props.trustedKeyGroups) === null || _h === void 0 ? void 0 : _h.map(keyGroup => keyGroup.keyGroupId),
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,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