import { CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { ApiCallBase, IApiCall } from './api-call-base';
import { ExpectedResult } from './common';
import { AssertionsProvider } from './providers';
/**
 * Options to perform an AWS JavaScript V2 API call
 */
export interface AwsApiCallOptions {
    /**
     * The AWS service, i.e. S3
     */
    readonly service: string;
    /**
     * The api call to make, i.e. getBucketLifecycle
     */
    readonly api: string;
    /**
     * Any parameters to pass to the api call
     *
     * @default - no parameters
     */
    readonly parameters?: any;
}
/**
 * Options for creating an SDKQuery provider
 */
export interface AwsApiCallProps extends AwsApiCallOptions {
}
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
export declare class AwsApiCall extends ApiCallBase {
    protected readonly apiCallResource: CustomResource;
    private readonly name;
    readonly provider: AssertionsProvider;
    private _assertAtPath?;
    constructor(scope: Construct, id: string, props: AwsApiCallProps);
    assertAtPath(path: string, expected: ExpectedResult): IApiCall;
}
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
export declare enum LogType {
    /**
     * The log messages are not returned in the response
     */
    NONE = "None",
    /**
     * The log messages are returned in the response
     */
    TAIL = "Tail"
}
/**
 * The type of invocation. Default is REQUEST_RESPONE
 */
export declare enum InvocationType {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    EVENT = "Event",
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    REQUEST_RESPONE = "RequestResponse",
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    DRY_RUN = "DryRun"
}
/**
 * Options to pass to the Lambda invokeFunction API call
 */
export interface LambdaInvokeFunctionProps {
    /**
     * The name of the function to invoke
     */
    readonly functionName: string;
    /**
     * The type of invocation to use
     *
     * @default InvocationType.REQUEST_RESPONE
     */
    readonly invocationType?: InvocationType;
    /**
     * Whether to return the logs as part of the response
     *
     * @default LogType.NONE
     */
    readonly logType?: LogType;
    /**
     * Payload to send as part of the invoke
     *
     * @default - no payload
     */
    readonly payload?: string;
}
/**
 * An AWS Lambda Invoke function API call.
 * Use this istead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
export declare class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope: Construct, id: string, props: LambdaInvokeFunctionProps);
}
