"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
const assertion_1 = require("./assertion");
const sdk_1 = require("./sdk");
const types = require("./types");
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify({ ...event, ResponseURL: '...' })}`);
    const provider = createResourceHandler(event, context);
    try {
        // if we are deleting the custom resource, just respond
        // with 'SUCCESS' since there is nothing to do.
        if (event.RequestType === 'Delete') {
            await provider.respond({
                status: 'SUCCESS',
                reason: 'OK',
            });
            return;
        }
        const result = await provider.handle();
        const actualPath = event.ResourceProperties.actualPath;
        // if we are providing a path to make the assertion at, that means that we have
        // flattened the response, otherwise the path to assert against in the entire response
        const actual = actualPath ? result[`apiCallResponse.${actualPath}`] : result.apiCallResponse;
        if ('expected' in event.ResourceProperties) {
            const assertion = new assertion_1.AssertionHandler({
                ...event,
                ResourceProperties: {
                    ServiceToken: event.ServiceToken,
                    actual,
                    expected: event.ResourceProperties.expected,
                },
            }, context);
            try {
                const assertionResult = await assertion.handle();
                await provider.respond({
                    status: 'SUCCESS',
                    reason: 'OK',
                    // return both the result of the API call _and_ the assertion results
                    data: {
                        ...assertionResult,
                        ...result,
                    },
                });
                return;
            }
            catch (e) {
                await provider.respond({
                    status: 'FAILED',
                    reason: e.message ?? 'Internal Error',
                });
                return;
            }
        }
        await provider.respond({
            status: 'SUCCESS',
            reason: 'OK',
            data: result,
        });
    }
    catch (e) {
        await provider.respond({
            status: 'FAILED',
            reason: e.message ?? 'Internal Error',
        });
        return;
    }
    return;
}
exports.handler = handler;
function createResourceHandler(event, context) {
    if (event.ResourceType.startsWith(types.SDK_RESOURCE_TYPE_PREFIX)) {
        return new sdk_1.AwsApiCallHandler(event, context);
    }
    else if (event.ResourceType.startsWith(types.ASSERT_RESOURCE_TYPE)) {
        return new assertion_1.AssertionHandler(event, context);
    }
    else {
        throw new Error(`Unsupported resource type "${event.ResourceType}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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