# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['degiro_connector',
 'degiro_connector.core',
 'degiro_connector.core.abstracts',
 'degiro_connector.core.constants',
 'degiro_connector.core.helpers',
 'degiro_connector.core.models',
 'degiro_connector.migration',
 'degiro_connector.quotecast',
 'degiro_connector.quotecast.actions',
 'degiro_connector.quotecast.models',
 'degiro_connector.trading',
 'degiro_connector.trading.actions',
 'degiro_connector.trading.models']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.41.1,<2.0.0',
 'onetimepass>=1.0.1,<2.0.0',
 'orjson>=3.6.0,<4.0.0',
 'pandas>=1.1.5,<2.0.0',
 'protobuf>=3.19.1,<4.0.0',
 'requests>=2.26.0,<3.0.0',
 'wrapt>=1.12.1,<2.0.0']

setup_kwargs = {
    'name': 'degiro-connector',
    'version': '2.0.7',
    'description': "This is yet another library to access Degiro's API.",
    'long_description': '# 1. **Degiro Connector**\n\nThis is yet another library to access Degiro\'s API.\n\nNotes :\n- Migration scripts are available :\n```bash\npython -m degiro_connector.migration.from_0_1_3_to_1_0_0\npython -m degiro_connector.migration.from_1_0_4_to_1_0_5\npython -m degiro_connector.migration.from_1_0_10_to_2_0_0\npython -m degiro_connector.migration.from_2_0_2_to_2_0_3\n```\n- `GRPC` services are available to let you consume this library through other languages like Javascript, Java, Go, C++, Rust, etc :\n```bash\npython -m examples.quotecast.relay_server\npython -m examples.trading.relay_server\n\n```\n\n## 1.1. Which features ?\nHere are the features you can access through this library :\n\n|**Endpoint**|**Feature(s)**|\n|:-|:-|\n|AccountCashReport|Export cash movements in a specific format : CSV, HTML, PDF or XLS.|\n|AccountInfo|Retrieve a table containing : "clientId" and Currencies.|\n|AccountOverview|Retrieve all the cash movements between two dates.|\n|Agenda|Crucial events regarding products : Dividend, Economic, Earnings, Holiday, IPO or Split.|\n|Bonds<br>ETFs<br>Funds<br>Futures<br>Leverageds<br>Lookup<br>Options<br>Stocks<br>Warrants|Search list of products according their name, type and other criterias. <br> For instance all the stocks from NASDAQ 100.|\n|Chart|Retrieve chart data.|\n|ClientDetails|Retrieve a table containing : "clientId", "intAccount" and other account information.|\n|CompanyProfile|Retrieve a company\'s profile using its ISIN code.|\n|CompanyRatios|Retrieve a company\'s ratios using its ISIN code.|\n|Config|Retrieve a table containing : "clientId" and URLs which are constitutive of Degiro\'s API.|\n|Favourites|Retrieve favorite products lists.|\n|FinancialStatements|Retrieve a company\'s financial statements using its ISIN code.|\n|LatestNews|Retrieve latest news about all the companies.|\n|LoginQuotecast|Establish a connection for quotecast operations.|\n|LoginTrading|Establish a connection for trading operations.|\n|LogoutTrading|Destroy previously established connection for trading operations.|\n|NewsByCompany|Retrieve news related to a specific company.|\n|Order|Create, update, delete an Order.|\n|OrderHistory|Retrieve all Orders created between two dates.|\n|Orders|List pending Orders.|\n|Portoflio|List products in your Portoflio.|\n|ProductsConfig|Retrieve a table containing : useful parameters to filter products.|\n|ProductsInfo|Search for products using their ids.|\n|Quotecasts|Fetch real-time data on financial products. <br> For instance the real-time stock Price.|\n|TopNewsPreview|Retrieve some news preview about all the companies.|\n|TotalPorfolio|Retrieve aggregated information about your assets.|\n|TransactionsHistory|Retrieve all Transactions created between two dates.|\n\n## 1.2. How to install ?\n\n```bash\npip install degiro-connector\n```\n\n## 1.3. How to upgrade ?\n\n```bash\npip install --no-cache-dir --upgrade degiro-connector\n```\n\n## 1.4. How to uninstall ?\n\n```bash\npip uninstall degiro-connector\n```\n\n## 1.5. Table of contents\n- [1. **Degiro Connector**](#1-degiro-connector)\n  * [1.1. Which features ?](#11-which-features-)\n  * [1.2. How to install ?](#12-how-to-install-)\n  * [1.3. How to upgrade ?](#13-how-to-upgrade-)\n  * [1.4. How to uninstall ?](#14-how-to-uninstall-)\n  * [1.5. Table of contents](#15-table-of-contents)\n- [2. Real-time data](#2-real-time-data)\n  * [2.1. What are the workflows ?](#21-what-are-the-workflows-)\n  * [2.2. What are the credentials ?](#22-what-are-the-credentials-)\n  * [2.3. How to find your : user_token ?](#23-how-to-find-your--user_token-)\n  * [2.4. How to login ?](#24-how-to-login-)\n  * [2.5. Is there a timeout ?](#25-is-there-a-timeout-)\n  * [2.6. How to subscribe to a data-stream ?](#26-how-to-subscribe-to-a-data-stream-)\n  * [2.7. How to unsubscribe to a data-stream ?](#27-how-to-unsubscribe-to-a-data-stream-)\n  * [2.8. How to fetch the data ?](#28-how-to-fetch-the-data-)\n  * [2.9. How to use this data ?](#29-how-to-use-this-data-)\n  * [2.10. Which are the available data types ?](#210-which-are-the-available-data-types-)\n  * [2.11. What is a Ticker ?](#211-what-is-a-ticker-)\n  * [2.12. What is inside the Dictionnary ?](#212-what-is-inside-the-dictionnary-)\n  * [2.13. What is inside the DataFrame ?](#213-what-is-inside-the-dataframe-)\n  * [2.14. How to get chart data ?](#214-how-to-get-chart-data-)\n  * [2.15. How to find a : vwd_id ?](#215-how-to-find-a--vwd_id-)\n- [3. Trading connection](#3-trading-connection)\n  * [3.1. How to login ?](#31-how-to-login-)\n  * [3.2. How to logout ?](#32-how-to-logout-)\n  * [3.3. What are the credentials ?](#33-what-are-the-credentials-)\n  * [3.4. How to find your : int_account ?](#34-how-to-find-your--int_account-)\n  * [3.5. How to use 2FA ?](#35-how-to-use-2fa-)\n  * [3.6. How to find your : totp_secret_key ?](#36-how-to-find-your--totp_secret_key-)\n  * [3.7. How to find your : one_time_password ?](#37-how-to-find-your--one_time_password-)\n  * [3.8. Is there a timeout ?](#38-is-there-a-timeout-)\n- [4. Order](#4-order)\n  * [4.1. How to create an Order ?](#41-how-to-create-an-order-)\n  * [4.2. How to update an Order ?](#42-how-to-update-an-order-)\n  * [4.3. How to delete an Order ?](#43-how-to-delete-an-order-)\n- [5. Portfolio](#5-portfolio)\n  * [5.1. How to retrieve pending Orders ?](#51-how-to-retrieve-pending-orders-)\n  * [5.2. How to get the Portfolio ?](#52-how-to-get-the-portfolio-)\n  * [5.3. How to get the TotalPortfolio ?](#53-how-to-get-the-totalportfolio-)\n  * [5.4. How to retrieve the OrdersHistory ?](#54-how-to-retrieve-the-ordershistory-)\n  * [5.5. How to retrieve the TransactionsHistory ?](#55-how-to-retrieve-the-transactionshistory-)\n- [6. Account](#6-account)\n  * [6.1. How to retrieve the table : Config ?](#61-how-to-retrieve-the-table--config-)\n  * [6.2. How to retrieve the table : ClientDetails ?](#62-how-to-retrieve-the-table--clientdetails-)\n  * [6.3. How to retrieve the table : AccountInfo ?](#63-how-to-retrieve-the-table--accountinfo-)\n  * [6.4. How to get the table : AccountOverview ?](#64-how-to-get-the-table--accountoverview-)\n  * [6.5. How to export the table : CashAccountReport ?](#65-how-to-export-the-table--cashaccountreport-)\n- [7. Products](#7-products)\n  * [7.1. How to get the table : ProductsConfig ?](#71-how-to-get-the-table--productsconfig-)\n  * [7.2. How to get my favourite products ?](#72-how-to-get-my-favourite-products-)\n  * [7.3. How to lookup products (search by name) ?](#73-how-to-lookup-products-search-by-name-)\n  * [7.4. How to search bonds ?](#74-how-to-search-bonds-)\n  * [7.5. How to search etfs ?](#75-how-to-search-etfs-)\n  * [7.6. How to search funds ?](#76-how-to-search-funds-)\n  * [7.7. How to search futures ?](#77-how-to-search-futures-)\n  * [7.8. How to search leverageds ?](#78-how-to-search-leverageds-)\n  * [7.9. How to search options ?](#79-how-to-search-options-)\n  * [7.10. How to search stocks ?](#710-how-to-search-stocks-)\n  * [7.11. How to search warrants ?](#711-how-to-search-warrants-)\n  * [7.12. How to search products from ids ?](#712-how-to-search-products-from-ids-)\n- [8. Companies](#8-companies)\n  * [8.1. How to get : CompanyProfile ?](#81-how-to-get--companyprofile-)\n  * [8.2. How to get : CompanyRatios ?](#82-how-to-get--companyratios-)\n  * [8.3. How to get : FinancialStatements ?](#83-how-to-get--financialstatements-)\n  * [8.4. How to get : LatestNews ?](#84-how-to-get--latestnews-)\n  * [8.5. How to get : TopNewsPreview ?](#85-how-to-get--topnewspreview-)\n  * [8.6. How to get : NewsByCompany ?](#86-how-to-get--newsbycompany-)\n  * [8.7. How to get : Agenda ?](#87-how-to-get--agenda-)\n- [9. Contributing](#9-contributing)\n- [10. License](#10-license)\n\n# 2. Real-time data\n\nIt is possible to fetch a stream of data in real-time from Degiro\'s API.\n\nFor instance if one needs the following data from the "AAPL" stock :\n* LastDate\n* LastTime\n* LastPrice\n* LastVolume\n\nHe can use this library to retrieve update like this :\n\n    LastDate    LastTime    LastPrice LastVolume\n    2020-11-13  22:00:00    119.26    4697040\n\n## 2.1. What are the workflows ?\n\nThis is the workflow for consuming real-time data-stream :\n\n    A. Find your "user_token".\n    B. Setup the API object with your "user_token".\n    C. Connect.\n    D. Subscribe to data-stream.\n    E. Fetch data-stream.\n\nThis is the worflow for consuming charts :\n\n    A. Find your "user_token".\n    B. Setup the API object with your "user_token".\n    C. Fetch charts.\n\nAll the details of these steps are explained in the rest of this section.\n\n## 2.2. What are the credentials ?\n\nThe only credential you need in order to fetch real-time data and charts is the :\n* user_token\n\nBeware, these two identifiers are not the same thing :\n* user_token : used to fetch real-time data and charts.\n* int_account : used for some trading operations.\n\n## 2.3. How to find your : user_token ?\nYou can find your "user_token" inside one of these tables :\n* "Config" : attribute "clientId"\n* "ClientDetails" : attribute "id"\n\nSee section related to "Config" and "ClientDetails" tables.\n\n## 2.4. How to login ?\n\nIn order to fetch data you need to establish a connection.\n\nYou can use the following code to connect :\n\n```python\n# SETUP QUOTECAST API\nquotecast_api = API(user_token=YOUR_USER_TOKEN)\n\n# CONNECTION\nquotecast_api.connect()\n```\n\n## 2.5. Is there a timeout ?\n\nConnection timeout is around 15 seconds.\n\nWhich means a connection will cease to work after this timeout.\n\nThis timeout is reset each time you use this connection to :\n* Subscribe to a metric (for instance a stock Price)\n* Fetch the data-stream\n\nSo if you use it nonstop (in a loop) you won\'t need to reconnect.\n\n## 2.6. How to subscribe to a data-stream ?\n\nTo subscribe to a data-stream you need to setup a Request message.\n\nA Request has the following parameters :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|subscriptions|MessageMap|List of products and metrics to subscribe to.|\n|unsubscriptions|MessageMap|List of products and metrics to unsubscribe to.|\n\nHere is an example of request :\n```python\nrequest = Quotecast.Request()\nrequest.subscriptions[\'360015751\'].extend([\n    \'LastDate\',\n    \'LastTime\',\n    \'LastPrice\',\n    \'LastVolume\',\n    \'LastPrice\',\n    \'AskPrice\',\n    \'BidPrice\',\n])\nrequest.subscriptions[\'AAPL.BATS,E\'].extend([\n    \'LastDate\',\n    \'LastTime\',\n    \'LastPrice\',\n    \'LastVolume\',\n    \'LastPrice\',\n    \'AskPrice\',\n    \'BidPrice\',\n])\n```\n\nIn this example these are the `vwd_id` of the product from which you want `Real-time data` :\n- 360015751\n- AAPL.BATS,E\n\nSee the section related to `vwd_id` for more information.\n\nOnce you have built this Request object you can send it to Degiro\'s API like this :\n```python\nquotecast_api.subscribe(request=request)\n```\n\nFor more comprehensive examples :\n[realtime_poller.py](examples/quotecast/realtime_poller.py) /\n[realtime_one_shot.py](examples/quotecast/realtime_one_shot.py)\n\n## 2.7. How to unsubscribe to a data-stream ?\n\nTo remove metrics from the data-stream you need to setup a Request message.\n\nIf you try to unsubscribe to a metric to which you didn\'t subscribed previously it will most likely have no impact.\n\nA Request has the following parameters :\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|subscriptions|MessageMap|List of products and metrics to subscribe to.|\n|unsubscriptions|MessageMap|List of products and metrics to unsubscribe to.|\n\nHere is an example of request :\n```python\nrequest = Quotecast.Request()\nrequest.unsubscriptions[\'360015751\'].extend([\n    \'LastDate\',\n    \'LastTime\',\n    \'LastPrice\',\n    \'LastVolume\',\n    \'LastPrice\',\n    \'AskPrice\',\n    \'BidPrice\',\n])\nrequest.unsubscriptions[\'AAPL.BATS,E\'].extend([\n    \'LastDate\',\n    \'LastTime\',\n    \'LastPrice\',\n    \'LastVolume\',\n    \'LastPrice\',\n    \'AskPrice\',\n    \'BidPrice\',\n])\n```\n\nOnce you have built this Request object you can send it to Degiro\'s API like this :\n```python\nquotecast_api.subscribe(request=request)\n```\n\nFor more comprehensive examples :\n[realtime_poller.py](examples/quotecast/realtime_poller.py) /\n[realtime_one_shot.py](examples/quotecast/realtime_one_shot.py)\n\n## 2.8. How to fetch the data ?\n\nYou can use the following code :\n```python\nquotecast = quotecast_api.fetch_data()\n```\n\nFor a more comprehensive example :\n[realtime_poller.py](examples/quotecast/realtime_poller.py)\n\n## 2.9. How to use this data ?\n\nReceived data is a Quotecast object with the following properties :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|json_data|dict|Dictionnary representation of what Degiro\'s API has sent.|\n|metadata|Metadata|Containing the "response_datetime" and "request_duration".|\n\nHere is how to access these properties :\n```python\njson_data = quotecast.json_data\nresponse_datetime = quotecast.metadata.response_datetime\nrequest_duration= quotecast.metadata.request_duration\n```\n\n## 2.10. Which are the available data types ?\n\nThis library provides the tools to convert Degiro\'s JSON data into something more programmer-friendly.\n\nHere is the list of available data types :\n\n|**Type**|**Description**|\n|:-|:-|\n|Ticker|Protobuf message (for GRPC).|\n|Dictionnaries|Standard Python Dictionaries : **dict**.|\n|DataFrame|DataFrame from the library Pandas.|\n\nHere is how to build each type :\n\n```python\n# UPDATE PARSER\nquotecast_parser.put_quotecast(quotecast=quotecast)\n\n# BUILD TICKER\nticker = quotecast_parser.ticker\n\n# BUILD DICT\nticker_dict = quotecast_parser.ticker_dict\n\n# BUILD PANDAS.DATAFRAME\nticker_df = quotecast_parser.ticker_df\n```\n\n## 2.11. What is a Ticker ?\n\nThe generated Ticker contains :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|metadata|Metadata|Containing the "response_datetime" and "request_duration".|\n|products|MessageMap|Dictionnary like object containing the metrics group by "vwd_id".|\n|product_list|RepeatedScalarFieldContainer|List of available "vwd_id".|\n\nHere are some operations available :\n\n```python\nproduct = \'360015751\'\nmetric_name = \'LastPrice\'\n\n# ACCESS SPECIFIC PRODUCT\nproduct = ticker.products[product]\n\n# ACCESS SPECIFIC METRIC\nmetric = product[metric_name]\n\n# LOOP OVER PRODUCTS\nfor product in ticker.products:\n    product = ticker.products[product]\n\n# LOOP OVER METRICS\nfor metric_name in product.metrics:\n    metric = product.metrics[metric_name]\n```\n\nA Ticker is a custom Protocol Buffer Message built for this library.\n\nIt can be transmitted over GRPC framework.\n\n## 2.12. What is inside the Dictionnary ?\n\nThe dictionnary representation of a ticker contains the metrics grouped by "vwd_id" (product id), with :\n* keys : vwd_id\n* values : another dictionnary with the metrics concerning this specific product.\n\nExample - Dictionnary :\n\n```python\n{\n    \'360114899\': {\n        \'vwd_id\': 360114899,\n        \'response_datetime\': \'2020-11-08 12:00:27\',\n        \'request_duration\': 1.0224891666870117,\n        \'LastDate\': \'2020-11-06\',\n        \'LastTime\': \'17:36:17\',\n        \'LastPrice\': \'70.0\',\n        \'LastVolume\': \'100\'\n    },\n    \'360015751\': {\n        \'vwd_id\': 360015751,\n        \'response_datetime\': \'2020-11-08 12:00:27\',\n        \'request_duration\': 1.0224891666870117,\n        \'LastDate\': \'2020-11-06\',\n        \'LastTime\': \'17:36:17\',\n        \'LastPrice\': \'22.99\',\n        \'LastVolume\': \'470\'\n    }\n}\n```\n\n## 2.13. What is inside the DataFrame ?\n\nIn addition to whatever metrics you have chosen to subscribe to, the DataFrame will contain the following columns :\n|**Column**|**Description**|\n|:-|:-|\n|vwd_id|Product identifier, for instance "AAPL.BATS,E" for APPLE stock.|\n|response_datetime|Datetime at which the data was received.|\n|request_duration|Duration of the request used to fetch the data.|\n\nExample - DataFrame :\n\n           vwd_id    response_datetime  request_duration    LastDate  LastTime LastPrice LastVolume\n    0   360114899  2020-11-08 12:00:27          1.022489  2020-11-06  17:39:57      70.0        100\n    1   360015751  2020-11-08 12:00:27          1.022489  2020-11-06  17:36:17     22.99        470\n\n## 2.14. How to get chart data ?\nYou can fetch an object containing the same data than in Degiro\'s website graph.\n\nFor that you need to prepare a Chart.Request object.\n\nHere is a table with the available attributes for Chart.Request.\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|requestid|str|It sends you back whatever string you put here, you can set it to : "1".|\n|resolution|Chart.Resolution|Resolution of the chart like : Chart.Resolution.PT1M.|\n|culture|str|Country code like : "en-US" or "fr-FR".|\n|period|Chart.Period|Period of the chart, like : Chart.Period.P1D.|\n|series|repeated string|Data to get like : [\'issueid:36014897\', \'price:issueid:360148977\'].|\n|tz|str|Timezone like : "Europe/Paris"|\n\nExample of code :\n\n```python\nrequest = Chart.Request()\nrequest.culture = "fr-FR"\nrequest.period = Chart.Interval.PT1H\nrequest.requestid = "1"\nrequest.resolution = Chart.Interval.P1D\n# request.series.append("issueid:360148977")\n# request.series.append("price:issueid:360148977")\nrequest.series.append("ohlc:issueid:360148977")\n# request.series.append("volume:issueid:360148977")\n# request.series.append("vwdkey:AAPL.BATS,E")\n# request.series.append("price:vwdkey:AAPL.BATS,E")\n# request.series.append("ohlc:vwdkey:AAPL.BATS,E")\n# request.series.append("volume:vwdkey:AAPL.BATS,E")\nrequest.tz = "Europe/Paris"\n\n# FETCH DATA\nchart = quotecast_api.get_chart(\n    request=request,\n    override={\n        "resolution": "P1D",\n        "period": "P1W",\n    },\n    raw=True,\n)\n```\n\nThe `issueid` parameter is the `vwd_id` of the product from which you want the `Chart` data.\n\nSee the section related to `vwd_id` for more information.\n\nAll the options for the enumerations are available in this file :\n[quotecast.proto](protos/degiro_connector/quotecast/models/quotecast.proto)\n\nFor a more comprehensive examples :\n - [chart.py](examples/quotecast/chart.py)\n - [chart_format.py](examples/quotecast/chart_format.py)\n\n## 2.15. How to find a : vwd_id ?\n\nIn operations related to `Quotecast`, Degiro uses the `vwd_id` to identify a product.\n\nWhich means that if you want a `Chart` or `Real-time data` for a specific product : you first need to find this product\'s `vwd_id`.\n\nThis two identifiers are not the same :\n\n|**Identifier**|**API name(s)**|**Description**|\n|:-|:-|:-|\n|id|str|Id used identify a product in `Trading` related endpoints.|\n|vwd_id|issueid <br /> vwdId <br /> vwdIdSecondary <br />|Id used identify a product in `Quotecast` (`Chart` and `Real-time data`) related endpoint.|\n\nHere are some methods you can use to fetch a product\'s `vwd_id` :\n- `product_search`\n- `get_products_info`\n\nThe method `product_search` let you use the name or other attributes of a product to fetch it\'s `vwd_id`.\n\nThe method `get_products_info` let you use a product\'s `id` to fetch it\'s `vwd_id`.\n\n\n# 3. Trading connection\n\nThis library is divided into two modules :\n- quotecast : to consume real-time financial data.\n- trading : to manage your Degiro\'s account.\n\nThe module **quotecast** is described in the section related to real-time data.\n\nThe rest of this document will only refer to the module : **trading**.\n\n## 3.1. How to login ?\nIn order to use the module **trading.api** you need to establish a connection.\n\nCheck the section related to **int_account** to understand how to get yours.\n\nHere is how to connect :\n```python\n# SETUP CREDENTIALS\ncredentials = Credentials(\n    username = YOUR_USERNAME,\n    password = YOUR_PASSWORD,\n    int_account = YOUR_INT_ACCOUNT,  # OPTIONAL FOR LOGIN\n)\n\n# SETUP TRADING API\ntrading_api = API(credentials=credentials)\n\n# ESTABLISH CONNECTION\ntrading_api.connect()\n```\n\nFor a more comprehensive example :\n[connection.py](examples/trading/connection.py)\n\n## 3.2. How to logout ?\n\nOnce you no longer need to use the API you can destroy your connection.\n\nYou can use the following code to disconnect :\n\n```python\n# DESTROY CONNECTION\nquotecast_api.logout()\n```\n\nFor a more comprehensive example :\n[logout.py](examples/trading/logout.py)\n\n## 3.3. What are the credentials ?\n\nSome credentials are required to use Degiro\'s trading API.\n\nHere are these credentials :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|username|str|Username used to log into Degiro\'s website.|\n|password|str|Password used to log into Degiro\'s website.|\n|int_account|int|OPTIONAL : unique identifier of the account : used by Degiro\'s server.|\n|totp_secret_key|str|OPTIONAL : used for Two-factor Authentication (2FA).|\n|one_time_password|str|OPTIONAL : used for Two-factor Authentication (2FA).|\n\nCheck the section related to **int_account** to understand how to get yours.\n\nCheck the section related to **2FA** if you want to know more about these two parameters :\n- **totp_secret_key**\n- **one_time_password**\n\n## 3.4. How to find your : int_account ?\n\nTo get your **int_acccount** you can run this example :\n[client_details_table.py](examples/trading/client_details_table.py)\n\nSee section related to **ClientDetails** table for more details.\n\nThis **int_acccount** is required to do most of the trading operations available in this connector.\n\nHere are some operations for which your **int_acccount** is not required :\n- Connection\n- Fetch table : ClientDetails\n- Fetch table : Config\n\nBeware, these two identifiers are not the same thing :\n- user_token : used to fetch real-time data and charts.\n- int_account : used for some trading operations.\n\n## 3.5. How to use 2FA ?\n\nFirst I will briefly explain what is : **Two-Factor Authentication (2FA)**.\n\nI recommend to skip a few paragraphs if you already know what is **2FA**.\n\nIn a standard connection you are providing two parameters :\n- username\n- password\n\nIf you use **Two-Factor Authentication (2FA)** you need an extra parameter :\n- one_time_password\n\nThis **one_time_password** has a validity of 30 secondes and is generated using a **totp_secret_key** code.\n\nThis **totp_secret_key** code is provided in Degiro\'s website when you enable **2FA** : it is the QRCode.\n\nUsually you put this QRCode inside an app like **\u200eGoogle Authenticator**.\n\n**\u200eGoogle Authenticator** generates a **one_time_password** that you can to log in. \n\nTo use **2FA** with this library you have two solution.\n\n**SOLUTION 1**\n\nProvide your **totp_secret_key** : the library will use it to generate a new **one_time_password** at each connection.\n\nSo you won\'t have to type your **one_time_password** manually at each connection.\n\nThis is the proper way.\n\nSee the section about **totp_secret_key** to understand how to get yours.\n\nHere is an example of connection with the **totp_secret_key** :\n```python\n# SETUP CREDENTIALS\ncredentials = Credentials(\n    username=YOUR_USERNAME,\n    password=YOUR_PASSWORD,\n    int_account=YOUR_INT_ACCOUNT,  # OPTIONAL FOR LOGIN\n    totp_secret_key=YOUR_2FA_SECRET_KEY,  # ONLY IF 2FA IS ENABLED\n)\n\n# SETUP TRADING API\ntrading_api = API(credentials=credentials)\n\n# ESTABLISH CONNECTION\ntrading_api.connect()\n```\n\nA complete example here :\n[connection_2fa.py](examples/trading/connection_2fa.py)\n\n**SOLUTION 2**\n\nProvide a new **one_time_password** at each connection.\n\nHere is an example of connection with the **one_time_password** :\n```python\n# SETUP CREDENTIALS\ncredentials = Credentials(\n    username=YOUR_USERNAME,\n    password=YOUR_PASSWORD,\n    int_account=YOUR_INT_ACCOUNT,  # OPTIONAL FOR LOGIN\n    one_time_password=YOUR_2FA_OTP,  # ONLY IF 2FA IS ENABLED\n)\n\n# SETUP TRADING API\ntrading_api = API(credentials=credentials)\n\n# ESTABLISH CONNECTION\ntrading_api.connect()\n```\n\nA complete example here :\n[connection_otp.py](examples/trading/connection_otp.py)\n\n## 3.6. How to find your : totp_secret_key ?\n\nThe parameter **totp_secret_key** is only required if you have enabled `2FA` on Degiro\'s website.\n\nWhen you try to activate `2FA` on Degiro\'s website, it displays a `QRCode`.\n\nThis `QRCode` changes at each activation.\n\nA `QRCode` is a picture which can be converted into a text.\n\nYou can download this `QRCode` and use a tool to extract the text from it.\n\nThis extracted text will look like this :\n\n    otpauth://totp/DEGIRO:YOUR_USERNAME?algorithm=SHA1&issuer=DEGIRO&secret=YOUR_TOPT_SECRET_KEY&digits=6&period=30\n\nHas you can guess the "totp_secret_key" is in this part :\n\n    secret=YOUR_TOPT_SECRET_KEY\n\nHere is an example of script that extracts the text from a `QRCode` :\n[qrcode.py](examples/trading/qrcode.py)\n\n## 3.7. How to find your : one_time_password ?\n\nThe parameter **one_time_password** is the password you type when you log in the website using **2FA**.\n\nUsually you get it through an app like **Google Authenticator**.\n\nIt is preferable to use the parameter **totp_secret_key** instead of **one_time_password**.\n\n## 3.8. Is there a timeout ?\nA connection for trading operations seems to have a timeout of : around 30 minutes.\n\nIf a connection is left unused for this amount of time it will cease to work.\n\nEvery time you do an operation using a connection, Degiro\'s API seems to reset the timeout for this connection.\n\n# 4. Order\n\nHere are the main parameters of an Order.\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|action|Order.Action|Whether you want to : `BUY` or `SELL`.|\n|order_type|Order.OrderType|Type of order : `LIMIT`, `STOP_LIMIT`, `MARKET` or `STOP_LOSS`.|\n|price|float|Price of the order. <br /> Only used for the following `order_type` : `LIMIT` and `STOPLIMIT`.|\n|product_id|int|Identifier of the product concerned by the order.|\n|size|float|Size of the order.|\n|stop_price|float|Stop price of the order. <br /> Only used for the following `order_type` : `STOPLIMIT` and `STOPLOSS`|\n|time_type|Order.TimeType|Duration of the order : GOOD_TILL_DAY or GOOD_TILL_CANCELED|\n\nThe full description of an Order is available here :\n[trading.proto](protos/degiro_connector/trading/models/trading.proto)\n\n## 4.1. How to create an Order ?\n\nThe Order creation is done in two step :\n* Checking : send the Order to the API to check if it is valid.\n* Confirmation : confirm the creation of the Order.\n\nKeeping these two steps (instead of reducing to one single "create" function) provides more options.\n\nHere are the parameters of a CheckingResponse :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|confirmation_id|str|Id necessary to confirm the creation of the Order.|\n|free_space_new|float|New free space (balance) if the Order is confirmed.|\n|response_datetime|Timestamp|Timestamp can be converted to date string using : ToJsonString().|\n|transaction_fees|repeated Struct|Transaction fees that will be applied to the Order.|\n|transaction_opposite_fees|repeated Struct|Other kind of fees that will be applied to the Order.|\n|transaction_taxes|repeated Struct|Taxes that will be applied to the Order.|\n\nHere are the parameters of a ConfirmationResponse :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|order_id|str|Id of the created Order.|\n|response_datetime|Timestamp|Timestamp can be converted to date string using : ToJsonString().|\n\nHere is an example :\n\n```python\n# SETUP ORDER\norder = Order(\n    action=Order.Action.BUY,\n    order_type=Order.OrderType.LIMIT,\n    price=10,\n    product_id=71981,\n    size=1,\n    time_type=Order.TimeType.GOOD_TILL_DAY,\n)\n\n# FETCH CHECKING_RESPONSE\nchecking_response = trading_api.check_order(order=order)\n\n# EXTRACT CONFIRMATION_ID\nconfirmation_id = checking_response.confirmation_id\n\n# SEND CONFIRMATION\nconfirmation_response = trading_api.confirm_order(\n    confirmation_id=confirmation_id,\n    order=order,\n)\n```\n\nFor a more comprehensive example :\n[order.py](examples/trading/order.py)\n\n## 4.2. How to update an Order ?\n\nTo modify a specific Order you need to setup it\'s "id".\n\nHere is an example :\n\n```python\n# ORDER SETUP\norder = Order(\n    id=YOUR_ORDER_ID,\n    action=Order.Action.BUY,\n    order_type=Order.OrderType.LIMIT,\n    price=10.60,\n    product_id=71981,\n    size=1,\n    time_type=Order.TimeType.GOOD_TILL_DAY,\n)\n\n# UPDATE ORDER\nsucccess = trading_api.update_order(order=order)\n```\n\n## 4.3. How to delete an Order ?\n\nTo delete a specific Order you just need it\'s "id".\n\nHere is an example :\n\n```python\n# DELETE ORDER\nsucccess = trading_api.delete_order(order_id=YOUR_ORDER_ID)\n```\n\n# 5. Portfolio\n\n## 5.1. How to retrieve pending Orders ?\n\nThis is how to get the list of Orders currently created but not yet executed or deleted :\n```python\nrequest_list = Update.RequestList()\nrequest_list.values.extend([\n    Update.Request(option=Update.Option.ORDERS, last_updated=0),\n])\n\nupdate = trading_api.get_update(request_list=request_list)\nupdate_dict = pb_handler.message_to_dict(message=update)\norders_df = pd.DataFrame(update_dict[\'orders\'][\'values\'])\n```\n\nExample : Orders\n\n       product_id      time_type  price  size                                    id  ...  action  order_type stop_price retained_order  sent_to_exchange\n    0           0  GOOD_TILL_DAY      2     3  202cb962-ac59-075b-964b-07152d234b70  ...     BUY       LIMIT         16             17                18\n\nFor a more comprehensive example :\n[update.py](examples/trading/update.py)\n\n## 5.2. How to get the Portfolio ?\n\nThis is how to list the stocks/products currently in the portfolio :\n```python\nrequest_list = Update.RequestList()\nrequest_list.values.extend([\n    Update.Request(option=Update.Option.PORTFOLIO, last_updated=0),\n])\n\nupdate = trading_api.get_update(request_list=request_list)\nupdate_dict = pb_handler.message_to_dict(message=update)\nportfolio_df = pd.DataFrame(update_dict[\'portfolio\'][\'values\'])\n```\n\nFor a more comprehensive example :\n[update.py](examples/trading/update.py)\n\n## 5.3. How to get the TotalPortfolio ?\n\nThis is how to get aggregated data about the portfolio :\n```python\nrequest_list = Update.RequestList()\nrequest_list.values.extend([\n    Update.Request(option=Update.Option.TOTALPORTFOLIO, last_updated=0),\n])\n\nupdate = trading_api.get_update(request_list=request_list)\nupdate_dict = pb_handler.message_to_dict(message=update)\ntotal_portfolio_df = pd.DataFrame(update_dict[\'total_portfolio\'][\'values\'])\n```\n\nExample : DataFrame\n\n       degiroCash  flatexCash  totalCash  totalDepositWithdrawal  todayDepositWithdrawal  ...  reportNetliq  reportOverallMargin  reportTotalLongVal  reportDeficit  marginCallStatus\n    0           0           1          2                       3                       4  ...            16                   17                  18             19    NO_MARGIN_CALL\n\nFor a more comprehensive example :\n[update.py](examples/trading/update.py)\n\n## 5.4. How to retrieve the OrdersHistory ?\n\nThis method returns data about passed orders between two dates.\n\nThe result contains a list of "Orders" objects with the following attributes :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|created|str|RFC 3339 Datetime, example : "2020-10-06T20:07:18+02:00".|\n|orderId|str|MD5 HASH, example : "098f6bcd-4621-d373-cade-4e832627b4f6"|\n|productId|int|Id of the product example : 65156|\n|size|float|Size of the order, example : 10.0000|\n|price|float|Price of the order, example : 8.6800|\n|buysell|str|"B" or "S"|\n|orderTypeId|int|see 3.Order|\n|orderTimeTypeId|int|see 3.Order|\n|stopPrice|float|Price like : 0.0000|\n|totalTradedSize|int|-|\n|type|str|"CREATE", "DELETE" or "MODIFY"|\n|status|str|"CONFIRMED"|\n|last|str|RFC 3339 Datetime, example : "2020-10-06T20:07:18+02:00".|\n|isActive|bool|-|\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nfrom_date = OrdersHistory.Request.Date(year=2020,month=11,day=15)\nto_date = OrdersHistory.Request.Date(year=2020,month=10,day=15)\nrequest = OrdersHistory.Request(from_date=from_date, to_date=to_date)\n\n# FETCH DATA\norders_history = trading_api.get_orders_history(request=request)\n```\n\nFor a more comprehensive example :\n[orders_history.py](examples/trading/orders_history.py)\n\n## 5.5. How to retrieve the TransactionsHistory ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nfrom_date = TransactionsHistory.Request.Date(year=2020,month=11,day=15)\nto_date = TransactionsHistory.Request.Date(year=2020,month=10,day=15)\nrequest = TransactionsHistory.Request(from_date=from_date, to_date=to_date)\n\n# FETCH DATA\ntransactions_history = trading_api.get_transactions_history(request=request)\n```\n\nFor a more comprehensive example :\n[transactions_history.py](examples/trading/transactions_history.py)\n\n# 6. Account\n\n## 6.1. How to retrieve the table : Config ?\n\nThe config table contains the following informations :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|sessionId|str|Current session id.|\n|clientId|int|Unique Degiro\'s Account identifier also called "userToken"|\n|tradingUrl|str|-|\n|paUrl|str|-|\n|reportingUrl|str|-|\n|paymentServiceUrl|str|-|\n|productSearchUrl|str|-|\n|dictionaryUrl|str|-|\n|productTypesUrl|str|-|\n|companiesServiceUrl|str|-|\n|i18nUrl|str|-|\n|vwdQuotecastServiceUrl|str|-|\n|vwdNewsUrl|str|-|\n|vwdGossipsUrl|str|-|\n|taskManagerUrl|str|-|\n|refinitivNewsUrl|str|-|\n|refinitivAgendaUrl|str|-|\n|refinitivCompanyProfileUrl|str|-|\n|refinitivCompanyRatiosUrl|str|-|\n|refinitivFinancialStatementsUrl|str|-|\n|refinitivClipsUrl|str|-|\n|landingPath|str|-|\n|betaLandingPath|str|-|\n|mobileLandingPath|str|-|\n|loginUrl|str|-|\n\nHere is how to get this table :\n\n```python\n# FETCH DATA\nconfig_table = trading_api.get_config()\n\n# EXTRACT SOME DATA\nuser_token = config_table[\'clientId\']\nsession_id = config_table[\'sessionId\']\n```\n\nFor a more comprehensive example :\n[config_table.py](examples/trading/config_table.py)\n\n## 6.2. How to retrieve the table : ClientDetails ?\n\nThe ClientDetails table contains information about the current Degiro Account.\n\n|**Parameter**|**Type**|\n|:-|:-|\n|id|int|\n|intAccount|int|\n|loggedInPersonId|int|\n|clientRole|str|\n|effectiveClientRole|str|\n|contractType|str|\n|username|str|\n|displayName|str|\n|email|str|\n|firstContact.firstName|str|\n|firstContact.lastName|str|\n|firstContact.displayName|str|\n|firstContact.nationality|str|\n|firstContact.gender|str|\n|firstContact.dateOfBirth|str|\n|firstContact.placeOfBirth|str|\n|firstContact.countryOfBirth|str|\n|address.streetAddress|str|\n|address.streetAddressNumber|str|\n|address.zip|str|\n|address.city|str|\n|address.country|str|\n|cellphoneNumber|str|\n|locale|str|\n|language|str|\n|culture|str|\n|bankAccount.bankAccountId|int|\n|bankAccount.bic|str|\n|bankAccount.iban|str|\n|bankAccount.status|str|\n|flatexBankAccount.bic|str|\n|flatexBankAccount.iban|str|\n|memberCode|str|\n|isWithdrawalAvailable|bool|\n|isAllocationAvailable|bool|\n|isIskClient|bool|\n|isCollectivePortfolio|bool|\n|isAmClientActive|bool|\n|canUpgrade|bool|\n\nHere is how to get this table :\n\n```python\n# FETCH DATA\nclient_details_table = trading_api.get_client_details()\n\n# EXTRACT SOME DATA\nint_account = client_details_table[\'data\'][\'intAccount\']\nuser_token = client_details_table[\'data\'][\'id\']\n```\n\nFor a more comprehensive example :\n[client_details_table.py](examples/trading/client_details_table.py)\n\n## 6.3. How to retrieve the table : AccountInfo ?\n\nThe AccountInfo table contains the following information about currencies.\n\n|**Parameter**|**Type**|\n|:-|:-|\n|clientId|int|\n|baseCurrency|str|\n|currencyPairs|dict|\n|marginType|str|\n|cashFunds|dict|\n|compensationCapping|float|-|\n\nHere is how to get this table :\n\n```python\naccount_info_table = trading_api.get_account_info()\n```\n\nFor a more comprehensive example :\n[account_info_table.py](examples/trading/account_info_table.py)\n\n## 6.4. How to get the table : AccountOverview ?\n\nIt will provide a list of cash movements.\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nfrom_date = AccountOverview.Request.Date(year=2020,month=11,day=15)\nto_date = AccountOverview.Request.Date(year=2020,month=10,day=15)\nrequest = AccountOverview.Request(from_date=from_date, to_date=to_date)\n\n# FETCH DATA\naccount_overview = trading_api.get_account_overview(request=request)\n```\n\nFor a more comprehensive example :\n[account_overview.py](examples/trading/account_overview.py)\n\nEach cash movement contains this kind of parameters :\n\n|**Parameter**|**Type**|\n|:-|:-|\n|date|str|\n|valueDate|str| \n|id|int| \n|orderId|str| \n|description|str| \n|productId|int|\n|currency|str|\n|change|float|\n|balance|dict| \n|unsettledCash|float|\n|total|float|\n\n\n## 6.5. How to export the table : CashAccountReport ?\n\nIt will export a list of cash movements in a specific format.\n\nAvailable formats :\n- CSV\n- HTML\n- PDF\n- XLS\n\nHere is how to get this content in `CSV` format :\n\n```python\n# SETUP REQUEST\nfrom_date = CashAccountReport.Request.Date(year=2020,month=11,day=15)\nto_date = CashAccountReport.Request.Date(year=2020,month=10,day=15)\nrequest = CashAccountReport.Request(\n    format=CashAccountReport.Format.CSV,\n    country=\'FR\',\n    lang=\'fr\',\n    from_date=from_date,\n    to_date=to_date,\n)\n\n# FETCH DATA\ncash_account_report = trading_api.get_cash_account_report(\n    request=request,\n    raw=False,\n)\n```\n\nHere are the available parameters for `CashAccountReport.Request` :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|format|CashAccountReport.Format|Wanted format : <br>`CSV`<br>`HTML`<br>`PDF`<br>`XLS`|\n|country|str|Country name, like : `FR`|\n|lang|int|Language, like : `fr`|\n|from_date|CashAccountReport.Request.Date|Events starting after this date.|\n|to_date|CashAccountReport.Request.Date|Events before this date.|\n\nExact definitions of `CashAccountReport` and `CashAccountReport.Request` are in this file :\n[trading.proto](protos/degiro_connector/trading/models/trading.proto)\n\nFor a more comprehensive example :\n[cash_account_report.py](examples/trading/cash_account_report.py)\n\n# 7. Products\n\n## 7.1. How to get the table : ProductsConfig ?\n\nThis table contains useful parameters to filter products.\n\nHere are the parameters which are inside this table :\n\n|**Parameter**|**Type**|\n|:-|:-|\n|stockCountries|list|\n|bondExchanges|list|\n|bondIssuerTypes|list|\n|eurexCountries|list|\n|futureExchanges|list|\n|optionExchanges|list|\n|combinationExchanges|list|\n|cfdExchanges|list|\n|exchanges|list|\n|indices|list|\n|regions|list|\n|countries|list|\n|productTypes|list|\n|etfFeeTypes|list|\n|investmentFundFeeTypes|list|\n|optionAggregateTypes|list|\n|leveragedAggregateTypes|list|\n|etfAggregateTypes|list|\n|investmentFundAggregateTypes|list|\n|lookupSortColumns|list|\n|stockSortColumns|list|\n|bondSortColumns|list|\n|cfdSortColumns|list|\n|etfSortColumns|list|\n|futureSortColumns|list|\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\nproducts_config = trading_api.get_products_config()\n```\n\nFor a more comprehensive example :\n[products_config.py](examples/trading/products_config.py)\n\n## 7.2. How to get my favourite products ?\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\nfavourites_list = trading_api.get_favourites_list()\n```\n\nFor a more comprehensive example :\n[favourites_list.py](examples/trading/favourites_list.py)\n\n## 7.3. How to lookup products (search by name) ?\n\nText research on a financial product.\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestLookup(\n    search_text=\'APPLE\',\n    limit=10,\n    offset=0,\n    product_type_id=1,\n)\n\n# FETCH DATA\nproducts_lookup = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_lookup.py](examples/trading/product_lookup.py)\n\n## 7.4. How to search bonds ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestBonds(\n    bond_issuer_type_id=0,\n    bond_exchange_id=710,\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\nbond_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.5. How to search etfs ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestETFs(\n    popular_only=False,\n    input_aggregate_types=\'\',\n    input_aggregate_values=\'\',\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\netf_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.6. How to search funds ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestFunds(\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\nfund_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.7. How to search futures ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestFutures(\n    future_exchange_id=1,\n    underlying_isin=\'FR0003500008\',\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\nfund_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.8. How to search leverageds ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestLeverageds(\n    popular_only=False,\n    input_aggregate_types=\'\',\n    input_aggregate_values=\'\',\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\netf_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.9. How to search options ?\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestOptions(\n    input_aggregate_types=\'\',\n    input_aggregate_values=\'\',\n    option_exchange_id=3,\n    underlying_isin=\'FR0003500008\',\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'expirationDate,strike\',\n    sort_types=\'asc,asc\',\n)\n\n# FETCH DATA\noption_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.10. How to search stocks ?\n\nIt contains information about available stocks.\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestStocks(\n    index_id=122001,    # NASDAQ 100\n    exchange_id=663,    # NASDAQ\n                        # You can either use `index_id` or `exchange id`\n                        # See which one to use in the `ProductsConfig` table\n    is_in_us_green_list=True,\n    stock_country_id=846, # US\n\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\nstock_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.11. How to search warrants ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductSearch.RequestWarrants(\n    search_text=\'\',\n    offset=0,\n    limit=100,\n    require_total=True,\n    sort_columns=\'name\',\n    sort_types=\'asc\',\n)\n\n# FETCH DATA\nwarrant_list = trading_api.product_search(request=request)\n```\n\nFor a more comprehensive example :\n[product_search.py](examples/trading/product_search.py)\n\n## 7.12. How to search products from ids ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = ProductsInfo.Request()\nrequest.products.extend([96008, 1153605, 5462588])\n\n# FETCH DATA\nproducts_info = trading_api.get_products_info(\n    request=request,\n    raw=True,\n)\n```\n\nFor a more comprehensive example :\n[products_info.py](examples/trading/products_info.py)\n\n# 8. Companies\n\n## 8.1. How to get : CompanyProfile ?\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\ncompany_profile = trading_api.get_company_profile(\n    product_isin=\'FR0000131906\',\n)\n```\n\nFor a more comprehensive example :\n[company_profile.py](examples/trading/company_profile.py)\n\n## 8.2. How to get : CompanyRatios ?\n\nThis table contains information about the company.\n\nHere are the parameters which are inside this table :\n\n|**Parameter**|**Type**|\n|:-|:-|\n|totalFloat|str|\n|sharesOut|str|\n|consRecommendationTrend|dict|\n|forecastData|dict|\n|currentRatios|dict|\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\ncompany_ratios = trading_api.get_company_ratios(\n    product_isin=\'FR0000131906\',\n)\n```\n\nFor a more comprehensive example :\n[company_ratios.py](examples/trading/company_ratios.py)\n\n## 8.3. How to get : FinancialStatements ?\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\nfinancials_statements = trading_api.get_financials_statements(\n    product_isin=\'FR0000131906\',\n)\n```\n\nFor a more comprehensive example :\n[financial_statements.py](examples/trading/financial_statements.py)\n\n## 8.4. How to get : LatestNews ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = LatestNews.Request(\n    offset=0,\n    languages=\'en,fr\',\n    limit=20,\n)\n\n# FETCH DATA\nlatest_news = trading_api.get_latest_news(\n    request=request,\n    raw=True,\n)\n```\n\nFor a more comprehensive example :\n[latest_news.py](examples/trading/latest_news.py)\n\n## 8.5. How to get : TopNewsPreview ?\n\nHere is how to get this data :\n\n```python\n# FETCH DATA\ntop_news_preview = trading_api.get_top_news_preview(raw=True)\n```\n\nFor a more comprehensive example :\n[top_news_preview.py](examples/trading/top_news_preview.py)\n\n## 8.6. How to get : NewsByCompany ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = NewsByCompany.Request(\n    isin=\'NL0000235190\',\n    limit=10,\n    offset=0,\n    languages=\'en,fr\',\n)\n\n# FETCH DATA\nnews_by_company = trading_api.get_news_by_company(\n    request=request,\n    raw=True,\n)\n```\n\nFor a more comprehensive example :\n[news_by_company.py](examples/trading/news_by_company.py)\n\n\n## 8.7. How to get : Agenda ?\n\nHere is how to get this data :\n\n```python\n# SETUP REQUEST\nrequest = Agenda.Request()\nrequest.start_date.FromJsonString(\'2021-06-21T22:00:00Z\')\nrequest.end_date.FromJsonString(\'2021-11-28T23:00:00Z\')\nrequest.calendar_type = Agenda.CalendarType.DIVIDEND_CALENDAR\nrequest.offset = 0\nrequest.limit = 25\n\n# FETCH DATA\nagenda = trading_api.get_agenda(\n    request=request,\n    raw=False,\n)\n```\n\nHere are the available parameters for `Agenda.Request` :\n\n|**Parameter**|**Type**|**Description**|\n|:-|:-|:-|\n|calendar_type|Agenda.CalendarType|Type of agenda : <br>`DIVIDEND_CALENDAR`<br>`ECONOMIC_CALENDAR`<br>`EARNINGS_CALENDAR`<br>`HOLIDAY_CALENDAR`<br>`IPO_CALENDAR`<br>`SPLIT_CALENDAR`|\n|offset|int|-|\n|limit|int|-|\n|order_by_desc|bool|-|\n|start_date|Timestamp|Events starting after this date.|\n|end_date|Timestamp|Events before this date.|\n|company_name|str|Filter used on the events description.|\n|countries|str|Comma separated list of countries like : `FR,US`|\n|classifications|str|Comma separated list of sectors like : `GovernmentSector,ExternalSector`|\n|units|str|Comma separated list of units like : `Acre,Barrel`|\n\nExact definitions of `Agenda` and `Agenda.Request` are in this file :\n[trading.proto](protos/degiro_connector/trading/models/trading.proto)\n\nFor a more comprehensive example :\n[agenda.py](examples/trading/agenda.py)\n\n# 9. Contributing\nPull requests are welcome.\n\nFeel free to open an issue or send me a message if you have a question.\n\n# 10. License\n[BSD-3-Clause License](https://raw.githubusercontent.com/Chavithra/degiro-connector/main/LICENSE)\n',
    'author': 'Chavithra PARANA',
    'author_email': 'chavithra@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/chavithra/degiro-connector',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
