import sys
import os
import argparse
import json
from tqdm import tqdm
import pickle
import random
import re
import multiprocessing
import cProfile, pstats
import numpy as np

import eval_utils
from eval_utils import essential_keys_no_lex_pos


parser = argparse.ArgumentParser()
parser.add_argument("-output_dir", default='eval_files',
                    type=str, help="Path of the directory to output evaluation results.")
parser.add_argument("-config_file", default='configs/config_default.json',
                    type=str, help="Config file specifying which sheets to use.")
parser.add_argument("-db_system", required=True,
                        type=str, help="Path of the system DB file we will be evaluating against the baseline.")
parser.add_argument("-pos", required=True,
                    type=str, help="Part-of-speech to perform the evaluation on.")
parser.add_argument("-baseline_db", default='eval_files/calima-msa-s31_0.4.2.utf8.db',
                    type=str, help="Path of the baseline DB file we will be comparing against.")
parser.add_argument("-multiprocessing", default=False, action='store_true',
                    help="Whether or not to use multiprocessing.")
parser.add_argument("-report_dir", default='eval_files/report_default',
                    type=str, help="Path of the directory containing partial reports (matrices) generated by the full generative evaluation.")
parser.add_argument("-possible_feat_combs", default='',
                    type=str, help="Paths of the file containing the union of possible feature combinations between the baseline and the system.")
parser.add_argument("-oblig_specs", default=False, action='store_true',
                    help="Whether to specify the obligatory features for the generator to generate or only optional (clitics).")
parser.add_argument("-feats_sets", required=True, action="extend", nargs="+",
                    type=str, help="Which feature set(s) to compute.")
parser.add_argument("-test_mode", default=False, action='store_true',
                    help="Only test mode.")
parser.add_argument("-profiling", default=False, action='store_true',
                    help="Run profiling.")
parser.add_argument("-n_cpu", default=8,
                    type=int, help="Number of cores to use.")
parser.add_argument("-n", default=1000000,
                    type=int, help="Number of inputs to the two compared systems.")
parser.add_argument("-camel_tools", default='local', choices=['local', 'official'],
                    type=str, help="Path of the directory containing the camel_tools modules.")
args = parser.parse_args()

random.seed(42)

with open(args.config_file) as f:
    config = json.load(f)

if args.camel_tools == 'local':
    camel_tools_dir = config['global']['camel_tools']
    sys.path.insert(0, camel_tools_dir)

from camel_tools.morphology.database import MorphologyDB
from camel_tools.morphology.generator import Generator
from camel_tools.morphology.utils import merge_features
from camel_tools.utils.charmap import CharMapper

bw2ar = CharMapper.builtin_mapper('bw2ar')
ar2bw = CharMapper.builtin_mapper('ar2bw')
    

def _process_generations(generations, defaults):
    generations_ = {}
    for g in generations:
        eval_utils.preprocess_lex_features(g)
        key = tuple([g.get(k, defaults[k]) for k in essential_keys_no_lex_pos])
        generations_.setdefault(key, []).append((g['lex'], g['diac'], g['pos']))
    return generations_


def _produce_generations(lemma_ar, oblig_feats, clitic_feats, feats_set):
    generations_baseline, generations_system = [], []
    failed = {}
    if oblig_feats:
        #FIXME: currently broken
        for feats_oblig in tqdm(oblig_feats):
            for feats_clitic in clitic_feats:
                feats_all = {**feats_oblig, **feats_clitic}
                try:
                    generations_baseline += generator_baseline.generate(
                        lemma_ar, feats_all, legacy=True)
                except:
                    failed.setdefault('baseline', []).append((lemma_ar, feats_all))
                if feats_all.get('prc0') not in ['mA_neg', 'lA_neg']:
                    try:
                        generations_system += generator_system.generate(
                            lemma_ar, feats_all)
                    except:
                        failed.setdefault('system', []).append((lemma_ar, feats_all))
    else:
        for feats_clitic in clitic_feats:
            if feats_set in ['baseline_only', 'intersection']:
                try:
                    generations_ = generator_baseline.generate(
                        lemma_ar, feats_clitic, legacy=True)
                    generations_baseline += generations_
                except:
                    failed.setdefault('baseline', []).append((lemma_ar, feats_clitic))
            if feats_set in ['system_only', 'intersection']:
                try:
                    generations_ = generator_system.generate(
                        lemma_ar, feats_clitic)
                    generations_system += generations_
                except:
                    failed.setdefault('system', []).append((lemma_ar, feats_clitic))
    
    return generations_baseline, generations_system, failed


def generate_all_possible_words_from_lemma(info):
    lemma_id, lemma_ar = info['lemma_id'], info['lemma']
    analysis2index = info['analysis2index']
    feats_set = info['feats_set']
    defaults = info['defaults']
    feats_size = len(analysis2index)
    oblig_feats, clitic_feats = info['oblig_feats'], info['clitic_feats']

    if not args.multiprocessing:
        #FIXME: currently broken
        global failed
        global diac_mat_system, diac_mat_baseline
        global system_only_mat, baseline_only_mat, no_intersect_mat
    else:
        failed = {}
        result = dict(
            diac_mat_system=np.zeros((1, feats_size), dtype='uint8')
                if feats_set in ['system_only', 'intersection'] else None,
            diac_mat_baseline=np.zeros((1, feats_size), dtype='uint8')
                if feats_set in ['baseline_only', 'intersection'] else None,
            system_only_mat=np.zeros((1, feats_size), dtype='uint8')
                if feats_set == 'intersection' else None,
            baseline_only_mat=np.zeros((1, feats_size), dtype='uint8')
                if feats_set == 'intersection' else None,
            no_intersect_mat=np.zeros((1, feats_size), dtype='bool')
                if feats_set == 'intersection' else None,
        )

    generations_baseline, generations_system, failed_ = _produce_generations(
        lemma_ar, oblig_feats, clitic_feats, feats_set)
    if not args.multiprocessing:
        for system, fails_ in failed_.items():
            fails = failed.setdefault(system, [])
            fails += fails_
    else:
        failed = failed_

    generations_baseline_ = _process_generations(generations_baseline, defaults)
    generations_system_ = _process_generations(generations_system, defaults)

    generations_system_set, generations_baseline_set = set(generations_system_), set(generations_baseline_)
    
    system_only = generations_system_set - generations_baseline_set
    for k in system_only:
        lemma_diac_pos_system_set = set(generations_system_[k])
        if not args.multiprocessing:
            result['diac_mat_system'][lemma_id][analysis2index[k]] += len(
                    lemma_diac_pos_system_set)
        else:
            analysis_index = analysis2index.get(k)
            if analysis_index is not None:
                result['diac_mat_system'][0][analysis_index] += len(
                    lemma_diac_pos_system_set)
    
    baseline_only = generations_baseline_set - generations_system_set
    for k in baseline_only:
        lemma_diac_pos_baseline_set = set(generations_baseline_[k])
        if not args.multiprocessing:
            result['diac_mat_baseline'][lemma_id][analysis2index[k]] += len(
                    lemma_diac_pos_baseline_set)
        else:
            analysis_index = analysis2index.get(k)
            if analysis_index is not None:
                result['diac_mat_baseline'][0][analysis_index] += len(
                    lemma_diac_pos_baseline_set)
    
    if feats_set == 'intersection':
        intersection = generations_system_set & generations_baseline_set
        for k in intersection:
            lemma_diac_pos_baseline_set = set(generations_baseline_[k])
            lemma_diac_pos_system_set = set(generations_system_[k])

            if not args.multiprocessing:
                result['diac_mat_system'][lemma_id][analysis2index[k]] += len(
                    lemma_diac_pos_system_set)
                result['diac_mat_baseline'][lemma_id][analysis2index[k]] += len(
                    lemma_diac_pos_baseline_set)
            else:
                result['diac_mat_system'][0][analysis2index[k]] += len(lemma_diac_pos_system_set)
                result['diac_mat_baseline'][0][analysis2index[k]] += len(lemma_diac_pos_baseline_set)
            
            system_baseline_lex_intersect = lemma_diac_pos_system_set & lemma_diac_pos_baseline_set
            baseline_only_lex = lemma_diac_pos_baseline_set - lemma_diac_pos_system_set
            system_only_lex = lemma_diac_pos_system_set - lemma_diac_pos_baseline_set
            if system_baseline_lex_intersect:
                if lemma_diac_pos_system_set != lemma_diac_pos_baseline_set:
                    if system_only_lex:
                        if not args.multiprocessing:
                            result['system_only_mat'][lemma_id][analysis2index[k]] += len(
                                system_baseline_lex_intersect)
                        else:
                            result['system_only_mat'][0][analysis2index[k]] += len(
                                system_baseline_lex_intersect)
                    if baseline_only_lex:
                        if not args.multiprocessing:
                            result['baseline_only_mat'][lemma_id][analysis2index[k]] += len(
                                system_baseline_lex_intersect)
                        else:
                            result['baseline_only_mat'][0][analysis2index[k]] += len(
                                system_baseline_lex_intersect)
            else:
                if not args.multiprocessing:
                    result['no_intersect_mat'][lemma_id][analysis2index[k]] = True
                else:
                    result['no_intersect_mat'][0][analysis2index[k]] = True

    result = {**{'lemma_id': lemma_id, 'failed': failed}, **result}

    return result


path_db_baseline = args.baseline_db
db_baseline = MorphologyDB(path_db_baseline)

db_baseline_gen = MorphologyDB(path_db_baseline, flags='g')
generator_baseline = Generator(db_baseline_gen)

path_db_system = os.path.join(args.db_system)
db_system = MorphologyDB(path_db_system)
db_system_gen = MorphologyDB(path_db_system, flags='g')
generator_system = Generator(db_system_gen)

POS = eval_utils.get_pos(args.pos, db_baseline, db_system)

eval_utils.harmonize_defaults(db_baseline, db_system, POS)
eval_utils.harmonize_defaults(generator_baseline._db, generator_system._db, POS)


if __name__ == "__main__":
    if args.profiling:
        profiler = cProfile.Profile()
        profiler.enable()

    if os.path.isdir(args.report_dir):
        for file_name in os.listdir(args.report_dir):
            if 'feat_combs' not in file_name:
                os.remove(os.path.join(args.report_dir, file_name))
    else:
        os.makedirs(args.report_dir)

    #FIXME: currently broken
    if args.oblig_specs:
        pos2obligfeats_baseline = eval_utils.get_pos2obligfeats(db_baseline)
        pos2obligfeats_system = eval_utils.get_pos2obligfeats(db_system)
        pos2obligfeats = pos2obligfeats_system
   
    #FIXME: currently broken
    POS2OBLIGFEATS = {pos: dict(intersection=None, baseline_only=None, system_only=None)
                      for pos in POS}

    lemmas_pos_baseline = eval_utils.get_all_lemmas_from_db(db_baseline)
    lemmas_pos_baseline = set([lemma_pos for lemma_pos in lemmas_pos_baseline
                            if lemma_pos[1] in POS])
    lemmas_pos_system = eval_utils.get_all_lemmas_from_db(MorphologyDB(path_db_system))
    lemmas_pos_system = set([lemma_pos for lemma_pos in lemmas_pos_system
                            if lemma_pos[1] in POS])
    lemmas_intersect = lemmas_pos_system & lemmas_pos_baseline
    lemmas_pos_baseline_only = lemmas_pos_baseline - lemmas_pos_system
    lemmas_pos_system_only = lemmas_pos_system - lemmas_pos_baseline

    LEMMAS_POS = dict(
        baseline_only=lemmas_pos_baseline_only,
        system_only=lemmas_pos_system_only,
        intersection=lemmas_intersect,
    )

    if args.possible_feat_combs:
        with open(args.possible_feat_combs, 'rb') as f:
            possible_feat_combs = pickle.load(f)
    else:
        if not args.test_mode:
            pos2possible_feat_combs_baseline = eval_utils.get_pos2possible_feat_combs(
                db_baseline, POS, merge_features)
            pos2possible_feat_combs_system = eval_utils.get_pos2possible_feat_combs(
                db_system, POS, merge_features)
            with open(os.path.join(args.report_dir, 'possible_feat_combs_system.pkl'), 'wb') as f:
                pickle.dump(pos2possible_feat_combs_system, f)
            with open(os.path.join(args.report_dir, 'possible_feat_combs_baseline.pkl'), 'wb') as f:
                pickle.dump(pos2possible_feat_combs_baseline, f)
            
        else:
            with open(os.path.join(args.report_dir, 'possible_feat_combs_system.pkl'), 'rb') as f:
                pos2possible_feat_combs_system = pickle.load(f)
            with open(os.path.join(args.report_dir, 'possible_feat_combs_baseline.pkl'), 'rb') as f:
                pos2possible_feat_combs_baseline = pickle.load(f)
                
        pos2possible_feat_combs_baseline_only = {}
        pos2possible_feat_combs_system_only = {}
        po2possible_feat_combs_intersection = {}
        for pos in POS:
            pos2possible_feat_combs_baseline_only[pos] = set(pos2possible_feat_combs_baseline[pos]) - \
                                                         set(pos2possible_feat_combs_system[pos])
            pos2possible_feat_combs_system_only[pos] = set(pos2possible_feat_combs_system[pos]) - \
                                                       set(pos2possible_feat_combs_baseline[pos])
            po2possible_feat_combs_intersection[pos] = set(pos2possible_feat_combs_system[pos]) & \
                                                       set(pos2possible_feat_combs_baseline[pos])
            
        pos2feat_value_pairs_baseline = eval_utils.get_pos2feat_value_pairs(pos2possible_feat_combs_baseline)
        pos2feat_value_pairs_system = eval_utils.get_pos2feat_value_pairs(pos2possible_feat_combs_system)
        with open(os.path.join(args.report_dir, 'pos2feat_value_pairs_baseline.pkl'), 'wb') as f:
            pickle.dump(pos2feat_value_pairs_baseline, f)
        with open(os.path.join(args.report_dir, 'pos2feat_value_pairs_system.pkl'), 'wb') as f:
            pickle.dump(pos2feat_value_pairs_system, f)
        
        POSSIBLE_FEAT_COMBS = dict(
            baseline_only=pos2possible_feat_combs_baseline_only,
            system_only=pos2possible_feat_combs_system_only,
            intersection=po2possible_feat_combs_intersection
        )
    
    defaults = db_baseline.defaults
    del db_baseline, db_system
    
    MATRICES, POS2CLITICFEATS = {}, {}
    for feats_set, pos2possible_feat_combs in POSSIBLE_FEAT_COMBS.items():
        possible_feat_combs = set.union(*[pos2possible_feat_combs[pos] for pos in POS])
        if feats_set not in args.feats_sets:
            continue
        lemmas_pos = LEMMAS_POS[feats_set] | (LEMMAS_POS['intersection']
                                            if feats_set != 'intersection' else set()) 
        lemmas_pos = list(lemmas_pos)[:args.n]

        lemmas_pos = [(i, lemma_pos) for i, lemma_pos in enumerate(lemmas_pos)]
        index2analysis = list(possible_feat_combs)
        analysis2index = {analysis: i for i, analysis in enumerate(index2analysis)}
        diac_mat_baseline = np.zeros((len(lemmas_pos), len(index2analysis)), dtype='uint8')
        diac_mat_system = np.zeros((len(lemmas_pos), len(index2analysis)), dtype='uint8')
        system_only_mat = np.zeros((len(lemmas_pos), len(index2analysis)), dtype='uint8')
        baseline_only_mat = np.zeros((len(lemmas_pos), len(index2analysis)), dtype='uint8')
        no_intersect_mat = np.zeros((len(lemmas_pos), len(index2analysis)), dtype='bool')
        MATRICES[feats_set] = dict(
            lemmas_pos=lemmas_pos,
            index2analysis=index2analysis,
            analysis2index=analysis2index,
            diac_mat_baseline=diac_mat_baseline if feats_set in ['baseline_only', 'intersection'] else None,
            diac_mat_system=diac_mat_system if feats_set in ['system_only', 'intersection'] else None,
            system_only_mat=system_only_mat if feats_set == 'intersection' else None,
            baseline_only_mat=baseline_only_mat if feats_set == 'intersection' else None,
            no_intersect_mat=no_intersect_mat if feats_set == 'intersection' else None
        )
        POS2CLITICFEATS[feats_set] = eval_utils.get_pos2cliticfeats(pos2possible_feat_combs, POS)
    
    for feats_set, info in MATRICES.items():
        print(f'Computing {feats_set}...')
        iterable = []
        for lemma_id, (lemma, pos) in info['lemmas_pos']:
            inputs = dict(
                lemma=lemma,
                lemma_id=lemma_id,
                analysis2index=info['analysis2index'],
                clitic_feats=POS2CLITICFEATS[feats_set][pos],
                oblig_feats=POS2OBLIGFEATS[pos][feats_set],
                defaults=defaults[pos],
                feats_set=feats_set
            )
            iterable.append(inputs)

        if args.multiprocessing:
            with multiprocessing.Pool(args.n_cpu) as p:
                results = list(tqdm(p.imap(generate_all_possible_words_from_lemma, iterable),
                                    total=len(iterable), smoothing=0.2))
            for result in results:
                lemma_id = result['lemma_id']
                for mat_name in eval_utils.mat_names:
                    if info[mat_name] is not None:
                        info[mat_name][lemma_id] = result[mat_name]
                
                for system, failed_ in result['failed'].items():
                    failed = info.setdefault('failed', {}).setdefault(system, [])
                    failed += failed_
        else:
            #FIXME: currently broken
            results = []
            for id_lemma_pos, oblig_specs in tqdm(lemmas_pos):
                generate_all_possible_words_from_lemma(id_lemma_pos, oblig_specs)

    with open(os.path.join(args.report_dir, 'matrices.pkl'), 'wb') as f:
        pickle.dump(MATRICES, f)
    
    print('Done.')