# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyebus', 'pyebus.cli']

package_data = \
{'': ['*']}

install_requires = \
['anytree>=2.8.0,<3.0.0']

entry_points = \
{'console_scripts': ['ebt = pyebus.cli:main']}

setup_kwargs = {
    'name': 'pyebus',
    'version': '1.0.0',
    'description': 'Pythonic Interface to EBUS Daemon (ebusd)',
    'long_description': '.. image:: https://badge.fury.io/py/pyebus.svg\n    :target: https://badge.fury.io/py/pyebus\n\n.. image:: https://img.shields.io/pypi/dm/pyebus.svg?label=pypi%20downloads\n   :target: https://pypi.python.org/pypi/pyebus\n\n.. image:: https://travis-ci.org/c0fec0de/pyebus.svg?branch=main\n    :target: https://travis-ci.org/c0fec0de/pyebus\n\n.. image:: https://readthedocs.org/projects/pyebus/badge/?version=latest\n    :target: https://pyebus.readthedocs.io/en/latest/?badge=latest\n\n.. image:: https://codeclimate.com/github/c0fec0de/pyebus.png\n    :target: https://codeclimate.com/github/c0fec0de/pyebus\n\n.. image:: https://img.shields.io/pypi/pyversions/pyebus.svg\n   :target: https://pypi.python.org/pypi/pyebus\n\n.. image:: https://img.shields.io/badge/code%20style-pep8-brightgreen.svg\n   :target: https://www.python.org/dev/peps/pep-0008/\n\n.. image:: https://img.shields.io/badge/code%20style-pep257-brightgreen.svg\n   :target: https://www.python.org/dev/peps/pep-0257/\n\nPythonic interface to EBUSD_.\n\n\nInstallation\n============\n\nTo install the `pyebus` module run::\n\n    pip install pyebus\n\nIf you do not have write-permissions to the python installation, try::\n\n    pip install pyebus --user\n\nCommand-Line-Interface\n======================\n\nUsage::\n\n\tusage: ebustool [-h] [--host HOST] [--port PORT] [--timeout TIMEOUT] [--version] [--debug] {cmd,listen,ls,observe,read,state,write,info} ...\n\n\tpositional arguments:\n\t  {cmd,listen,ls,observe,read,state,write,info}\n\t                        Sub Commands\n\t    cmd                 Issue TCP Command on EBUSD. See https://github.com/john30/ebusd/wiki/3.1.-TCP-client-commands for reference.\n\t    listen              Listen on the bus, decode messages and and print\n\t    ls                  List all messages\n\t    observe             Read all known messages once and continue listening so that ALL EBUS values are available, decode every message and print.\n\t    read                Read values from the bus, decode and print\n\t    state               Show EBUSD state\n\t    write               Write value to the bus\n\t    info                Show EBUSD meta information\n\n\toptional arguments:\n\t  -h, --help            show this help message and exit\n\t  --host HOST, -H HOST  EBUSD address. Default is \'127.0.0.1\'.\n\t  --port PORT, -P PORT  EBUSD port. Default is 8888.\n\t  --timeout TIMEOUT, -T TIMEOUT\n\t                        EBUSD connection timeout. Default is 10.\n\t  --version             show program\'s version number and exit\n\t  --debug\n\nList all messages and fields\n----------------------------\n\nThe `ls` command lists all messages `CIRCUIT/MESSAGENAME/FIELDNAME     rwuSP INFO`::\n\n\t$ ebt -H 192.168.1.4 ls\n\tmc/DateTime/dcfstate                     r---1 nosignal, ok, sync, valid [DCF Empfängerstatus]\n\tmc/DateTime/btime                        r---1 HOUR:MINUTE:SECOND [Uhrzeit]\n\tmc/DateTime/bdate                        r---1 DAY.MONTH.YEAR [Datum]\n\tmc/DateTime/temp2                        r---1 Float within [-127.99:127.99] with 0.00390625 fraction\n\tmc/DateTime/bdate+btime+dcfstate         r---1 DAY.MONTH.YEAR HOUR:MINUTE:SECOND\n\tmc/FlowTemp/temp                         r---1 Float within [-2047.9:2047.9] with 0.0625 fraction [Temperatur]\n\tmc/FlowTemp/sensor                       r---1 ok, circuit, cutoff\n\tmc/FlowTemp/temp+sensor                  r---1 Float within [-2047.9:2047.9] with 0.0625 fraction\n\tmc/FlowTempDesired/temp1                 r---1 Float within [0:100] with 0.5 fraction\n\tmc/FlowTempMax/temp0                     rw--2 Integer within [0:254]\n\tmc/FlowTempMin/temp0                     rw--2 Integer within [0:254]\n\tmc/OperatingMode/mcmode                  rw--2 disabled, on, off, auto, eco, low\n\tmc/TempDesired/temp1                     rw--2 Float within [0:100] with 0.5 fraction\n\tmc/TempDesiredLow/temp1                  rw--2 Float within [0:100] with 0.5 fraction\n\n`CIRCUIT/MESSAGENAME/FIELDNAME` is a unique message field identifier.\n`CIRCUIT` names the device which contains the information.\n`MESSAGENAME` is the name of the message on the bus.\nEach message consists of fields.\n`FIELDNAME` identifies the specific information within the message.\nEBUSD fieldnames are **NOT** unique. `pyebus` appends a suffix in case of naming collisions.\nField names with a `+` are virtual and just the concatenation of existing fields.\n\nThe access rights have the following meaning:\n\n* `r`: The message is explicitly readable.\n* `w`: The message is explicitly writable.\n* `u`: The message is **NOT** readable, but emitted by the sender on change.\n* `S`: Is the poll priority to be set. **This applies to readable messages only.** Values `1-9` can be used. `A` is a placeholder for automatic. This will choose an appropriate priority.\n* `P`: Actual polling priority.\n\nList some messages and fields\n-----------------------------\n\nThe `ls` command accepts explicit names with wildcards and placeholders. **Case-Insensitive**.\n\n\t$ ebt -H 192.168.1.4 ls "mc/FlowTemp*"\n\tmc/FlowTemp/temp                         r---1 Float within [-2047.9:2047.9] with 0.0625 fraction [Temperatur]\n\tmc/FlowTemp/sensor                       r---1 ok, circuit, cutoff\n\tmc/FlowTemp/temp+sensor                  r---1 Float within [-2047.9:2047.9] with 0.0625 fraction\n\tmc/FlowTempDesired/temp1                 r---1 Float within [0:100] with 0.5 fraction\n\tmc/FlowTempMax/temp0                     rw--2 Integer within [0:254]\n\tmc/FlowTempMin/temp0                     rw--2 Integer within [0:254]\n\n\t$ ebt -H 192.168.1.4 ls "*/*mode*"\n\tmc/OperatingMode/mcmode                  rw--2 disabled, on, off, auto, eco, low\n\nRead messages and fields\n------------------------\n\n`read` behaves identical to `ls` (with or without patterns), but returns the actual value::\n\n\t$ ebt -H 192.168.1.4 read "*/*mode*"\n\tmc/OperatingMode/mcmode                  rw--2 eco                 disabled, on, off, auto, eco, low\n\nNon-readable messages are filtered automatically.\n\nPlease note, EBUS is slow. EBUSD only reads values which are older than 300s or not cached.\n`--ttl` explicitly specifies the maximum age in seconds.\n\nWrite Message Field\n-------------------\n\nEach writable field can be set by::\n\n\t$ ebt -H 192.168.1.4 write mc/OperatingMode/mcmode auto\n\nEBUS Status\n-----------\n\nThe EBUS status can be retrieved by::\n\n\t$ ebt -H 192.168.1.4 state\n\tok\n\nor more detailled::\n\n\t$ ebt -H 192.168.1.4 info\n\tversion                ebusd 21.1.v21.1-12-gccfc025\n\tupdate check           version 3.4 available\n\tsignal                 acquired\n\tsymbol rate            114\n\tmax symbol rate        217\n\tmin arbitration micros 317\n\tmax arbitration micros 4751\n\tmin symbol latency     0\n\tmax symbol latency     10\n\treconnects             0\n\tmasters                7\n\tmessages               1006\n\tconditional            14\n\tpoll                   597\n\tupdate                 10\n\taddress 03             master #11\n\taddress 08             slave #11, scanned "MF=Vaillant;ID=BAI00;SW=0204;HW=9602", loaded "vaillant/bai.0010015600.inc" ([HW=9602]), "vaillant/08.bai.csv"\n\taddress 10             master #2\n\taddress 15             slave #2, scanned "MF=Vaillant;ID=UI   ;SW=0508;HW=6201", loaded "vaillant/15.ui.csv"\n\taddress 17             master #17\n\n\n\nProgramming API\n===============\n\nComplete API-Documentation_\n\nOverview\n--------\n\n`Ebus` represents one connection to a EBUSD instance.\n\n>>> from pyebus import Ebus\n>>> ebus = Ebus(\'127.0.0.1\')\n\nThe instance needs to know the messages definitions handled by the EBUSD instance.\nThe EBUSD message definitions are based on a EBUSD scan algorithm and the EBUSD-configuration_.\n\nWait for the EBUSD scan to be completed:\n\n>>> await ebus.async_wait_scancompleted()\n\nNow, we can load the message definition codes and convert them to message definitions `MsgDef`.\n\n>>> await ebus.async_load_msgdefs()\n\nThe message definitions are stored in a `MsgDefs` instance at\n\n>>> ebus.msgdefs\n\nA single message can be read via:\n\n>>> msgdef = ebus.msgdefs.get(\'circuit\', \'name\')\n>>> print(await ebus.async_read(msgdef))\n\nAll messages can be read via:\n\n>>> for msgdef in ebus.msgdefs:\n>>>   if msgdef.read:\n>>>     print(await ebus.async_read(msgdef))\n\nMessage can be filtered via patterns `CIRCUIT/MESSAGENAME` or `CIRCUIT/MESSAGENAME/FIELDNAME`.\nWildcards (`*`) and Placeholder (`?`) are supported.\nThe following example will read all temperature fields and all message of the circuit named `mc`.\nThe read value must not be older than 1000s.\n\n>>> for msgdef in ebus.msgdefs.resolve([\'*/*/*temp*\', \'mc/*\']):\n>>>   if msgdef.read:\n>>>     print(await ebus.async_read(msgdef, ttl=1000))\n\nWriting is also possible\n\n>>> await ebus.async_write(msgdef, value)\n\nEBUS meta informations are available at:\n\n>>> await ebus.async_get_state()\n>>> await ebus.async_get_info()\n\n.. _EBUSD: https://github.com/john30/ebusd\n.. _EBUSD-Configuration: https://github.com/john30/ebusd-configuration\n.. _API-Documentation: https://pyebus.readthedocs.io/en/latest/api/pyebus.html\n',
    'author': 'c0fec0de',
    'author_email': 'c0fec0de@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/c0fec0de/pyebus',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
