# AUTOGENERATED! DO NOT EDIT! File to edit: 09_optoceutics.ipynb (unless otherwise specified).

__all__ = ['get_raw', 'correct_time', 'ms_to_datetime', 'get_time_range', 'decompose', 'API', 'login', 'query_tasks',
           'split_test', 'parse_test', 'parse_block', 'parse_task', 'parse_tasks', 'get_session_time_range',
           'select_sessions']

# Cell
import matplotlib
import matplotlib.pyplot as plt
import pandas as pd
import numpy as np
import time
import mne
from pylab import rcParams
from mne.preprocessing import (ICA, create_eog_epochs, create_ecg_epochs,corrmap)
from gql import gql, Client
from gql.transport.aiohttp import AIOHTTPTransport
import csv
import datetime

# Cell
def get_raw(fpath):
    raw = mne.io.read_raw_edf(fpath, preload=True)

    # set channel types
    n_eeg_channels = len([c for c in raw.ch_names if 'EEG' in c])
    n_stim_channels = len([c for c in raw.ch_names if 'Photic' in c])
    channel_types = dict(zip(raw.info['ch_names'], ['eeg']*n_eeg_channels + ['stim']*n_stim_channels))
    raw.set_channel_types(channel_types)

    # rename channels and add montage
    ch_names = [ch[4:] for ch in raw.ch_names if 'EEG' in ch]
    raw.rename_channels(dict(zip(raw.ch_names[:-1], ch_names)))
    raw.set_montage('standard_1020')

    return raw

# Cell
def correct_time(raw, fname):
    contents = []
    with open(fname, newline='') as csvfile:
        csvreader = csv.reader(csvfile, delimiter=';', quotechar='|')
        for row in csvreader:
            print(', '.join(row))
            contents.append(row)
    print("\nEnd of CSV contents\nExtracted time correction:",contents[-2][-1])
    corrected_time = datetime.datetime.strptime(contents[-2][-1], '%Y-%m-%d %H:%M:%S.%f')

    df_raw = raw.to_data_frame()
    stimulus = df_raw[df_raw['BIN Photic'] != 0.0]
    stimulus_trigger = stimulus['BIN Photic'].index[0]
    df_raw['adjusted_time'] = pd.to_timedelta(df_raw['time'] - df_raw['time'][stimulus_trigger],unit='ms') + corrected_time - np.timedelta64(1, 'h')
    df_raw_adjusted = df_raw.set_index('adjusted_time')

    # TODO: adjust time on raw object

    return df_raw_adjusted

# Cell
def ms_to_datetime(times): return pd.to_datetime(times, unit='ms').values

# Cell
def get_time_range(task): return ms_to_datetime([task['startTime'], task['endTime']])

# Cell
def decompose(data, keys): return dict(zip(keys,[data[k] for k in keys]))

# Cell
class API:
    url="https://prod.insaicognition.com/graphql"
    token = None
    org = None
    transport = AIOHTTPTransport(url=url)
    client = Client(transport=transport, fetch_schema_from_transport=True)
    def __init__(self, url=None):
        if (url): self.url = url

# Cell
async def login(self, user, password):
    mutation = f"""
    mutation login {{
      login(email: "{user}", password: "{password}") {{
        accessToken
        user {{
            id
            email
            name
            organization {{
                id
            }}
        }}
        }}
      }}
    """
    login = gql(mutation)
    result = await self.client.execute_async(login)
    self.token = result['login']['accessToken']
    self.org = result['login']['user']['organization'][0]['id']
    self.transport = AIOHTTPTransport(url="https://prod.insaicognition.com/graphql",
                                      headers={'authorization': self.token, 'organization': self.org})
    self.client = Client(transport=self.transport,
                         fetch_schema_from_transport=True)

# Cell
API.login = login

# Cell
async def query_tasks(self, email):
    query = f"""query tasks {{
      tasks(where: {{user: {{email: {{equals: "{email}"}}}}}},
        orderBy: {{createdAt: asc}}) {{
        type
        startTime
        endTime
        completed
        createdAt
        battery {{
          template {{
            id
          }}
        }}
        config {{
          window
          trials
          duration
          time
          timed
        }}
        trials {{
          id
          stimulus
          index
          expected
          critical
          response {{
            id
            response
            result
            presentedAt
            respondedAt
          }}
        }}
     }}
    }}
    """
    tasks_query = gql(query)
    result = await self.client.execute_async(tasks_query)
    tasks = result['tasks']
    return tasks

# Cell
API.query_tasks = query_tasks

# Cell
def split_test(test):
    config = test['config']
    config['type'] = test['type']
    trials = test['trials']
    responses = [t['response'] for t in trials]
    # strip trials from response fields
    keys = ['id', 'stimulus', 'index', 'expected', 'critical']
    trials = [decompose(t, keys) for t in trials]
    assert len(trials) == len(responses)
    return config, trials, responses

# Cell
def parse_test(test):
    config, trials, responses = split_test(test)
    trial_df = pd.DataFrame(trials)
    resp_df = pd.DataFrame(responses).rename(columns={'presentedAt': 'presented_at', 'respondedAt': 'responded_at'})
    resp_df.presented_at = ms_to_datetime(resp_df.presented_at)
    resp_df.responded_at = ms_to_datetime(resp_df.responded_at)
    session = pd.concat([trial_df, resp_df], axis=1)
    return config, session

# Cell
def parse_block(block):
    config = block['config']
    config['type'] = block['type']
    start, end = get_time_range(block)
    session = pd.DataFrame.from_dict(dict(start=[start], end=[end]))
    return config, session

# Cell
def parse_task(task):
    if task['type'] == 'NBACK':
        # check for valid
        if (task['trials'][0]['response'] is None): return False
        if (task['trials'][0]['response']['presentedAt'] is None): return False
        return parse_test(task)
    if task['type'] == 'REST':
        if (task['config'] is None): return False
        return parse_block(task)

# Cell
def parse_tasks(tasks, exclude=True):
    sessions = [parse_task(t) for t in tasks]
    [print(f'Bad session: {i}') for i,s in enumerate(sessions) if s==False]
    if exclude:
        print('Excluding bad sessions')
        return [s for s in sessions if s != False]
    else: return sessions

# Cell
def get_session_time_range(session):
    config, session = session
    task_type = config['type']
    if task_type == 'NBACK':
        start = session['presented_at'][0]
        end = session['responded_at'][len(session)-1]
        return task_type, start, end
    if task_type == 'REST':
        start = session.start[0]
        end = session.end[0]
        return task_type, start, end

# Cell
def select_sessions(raw_df, sessions):
    start, end = raw_df.index[0], raw_df.index[-1]
    time_ranges = pd.DataFrame([get_session_time_range(s) for s in sessions], columns=['type', 'start', 'end'])
    time_mask = (time_ranges['start'] >= start) & (time_ranges['end'] <= end)
    return [s for m,s in zip(time_mask, sessions) if m]