"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const eks = require("../lib");
const util_1 = require("./util");
module.exports = {
    'create nodegroup correctly'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new eks.Nodegroup(stack, 'Nodegroup', { cluster });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            NodeRole: {
                'Fn::GetAtt': [
                    'NodegroupNodeGroupRole038A128B',
                    'Arn',
                ],
            },
            Subnets: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
            ForceUpdateEnabled: true,
            ScalingConfig: {
                DesiredSize: 2,
                MaxSize: 2,
                MinSize: 1,
            },
        }));
        test.done();
    },
    'create nodegroup correctly with security groups provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            remoteAccess: {
                sshKeyName: 'foo',
                sourceSecurityGroups: [new ec2.SecurityGroup(stack, 'SG', { vpc })],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            RemoteAccess: {
                Ec2SshKey: 'foo',
                SourceSecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'SGADB53937',
                            'GroupId',
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    'create nodegroup with forceUpdate disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new eks.Nodegroup(stack, 'Nodegroup', { cluster, forceUpdate: false });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ForceUpdateEnabled: false,
        }));
        test.done();
    },
    'create nodegroup with instanceType provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            instanceType: new ec2.InstanceType('m5.large'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            InstanceTypes: [
                'm5.large',
            ],
        }));
        test.done();
    },
    'remoteAccess without security group provided'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            remoteAccess: {
                sshKeyName: 'foo',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            RemoteAccess: {
                Ec2SshKey: 'foo',
            },
        }));
        test.done();
    },
    'import nodegroup correctly'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        // const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        const ng = new eks.Nodegroup(stack1, 'Nodegroup', { cluster });
        const imported = eks.Nodegroup.fromNodegroupName(stack2, 'ImportedNg', ng.nodegroupName);
        new cdk.CfnOutput(stack2, 'NodegroupName', { value: imported.nodegroupName });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                NodegroupName: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputRefNodegroup62B4B2C1EF8AB7C1',
                    },
                },
            },
        });
        test.done();
    },
    'addNodegroup correctly'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        // WHEN
        cluster.addNodegroup('ng');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            NodeRole: {
                'Fn::GetAtt': [
                    'ClusterNodegroupngNodeGroupRoleDA0D35DA',
                    'Arn',
                ],
            },
            Subnets: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
            ForceUpdateEnabled: true,
            ScalingConfig: {
                DesiredSize: 2,
                MaxSize: 2,
                MinSize: 1,
            },
        }));
        test.done();
    },
    'throws when desiredSize > maxSize'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        // THEN
        test.throws(() => cluster.addNodegroup('ng', { desiredSize: 3, maxSize: 2 }), /Desired capacity 3 can't be greater than max size 2/);
        test.done();
    },
    'throws when desiredSize < minSize'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        // THEN
        test.throws(() => cluster.addNodegroup('ng', { desiredSize: 2, minSize: 3 }), /Minimum capacity 3 can't be greater than desired size 2/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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