"use strict";
const sinon = require("sinon");
const consts_1 = require("../lib/cluster-resource-handler/consts");
const fargate_1 = require("../lib/cluster-resource-handler/fargate");
function newRequestMock(props = {}) {
    return {
        RequestType: 'Create',
        ServiceToken: 'ServiceTokenMock',
        LogicalResourceId: 'LogicalResourceIdMock',
        RequestId: 'RequestIdMock',
        ResourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
        ResponseURL: 'ResponseURLMock',
        StackId: 'StackIdMock',
        PhysicalResourceId: 'PhysicalResourceIdMock',
        ResourceProperties: {
            ServiceToken: 'ServiceTokenMock',
            AssumeRoleArn: 'AssumeRoleArn',
            Config: {
                clusterName: 'MockClusterName',
            },
        },
        ...props,
    };
}
function newEksClientMock() {
    return {
        createCluster: sinon.fake.throws('not implemented'),
        deleteCluster: sinon.fake.throws('not implemented'),
        describeCluster: sinon.fake.throws('not implemented'),
        updateClusterConfig: sinon.fake.throws('not implemented'),
        updateClusterVersion: sinon.fake.throws('not implemented'),
        configureAssumeRole: sinon.fake(),
        createFargateProfile: sinon.fake.returns({
            fargateProfile: {
                fargateProfileName: 'MockProfileName',
                fargateProfileArn: 'MockProfileArn',
            },
        }),
        deleteFargateProfile: sinon.fake(),
        describeFargateProfile: sinon.fake.throws('not implemented'),
    };
}
module.exports = {
    create: {
        async 'calls createFargateProfile with a generated name'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Create.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'LogicalResourceIdMock-RequestIdMock',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
        async 'respects physical name if provided'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                ResourceProperties: {
                    AssumeRoleArn: 'AssumeRoleArnMock',
                    Config: {
                        fargateProfileName: 'MyProfileNameBoom',
                        clusterName: 'MockClusterName',
                    },
                },
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                fargateProfileName: 'MyProfileNameBoom',
                clusterName: 'MockClusterName',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
        async 'isComplete returns true if fargate profile is ACTIVE'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'ACTIVE',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(resp.IsComplete, true);
            test.done();
        },
        async 'isComplete returns false as long as fargate profile is CREATING'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'CREATING',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(resp.IsComplete, false);
            test.done();
        },
        async 'isComplete throws an exception if the status is CREATE_FAILED'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'CREATE_FAILED',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(error.message, 'CREATE_FAILED');
            test.done();
        },
    },
    update: {
        async 'calls createFargateProfile with a new name'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Update',
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Update.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.createFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'LogicalResourceIdMock-RequestIdMock',
            });
            test.deepEqual(onEventResponse, {
                PhysicalResourceId: 'MockProfileName',
                Data: { fargateProfileArn: 'MockProfileArn' },
            });
            test.done();
        },
    },
    delete: {
        async 'calls deleteFargateProfile'(test) {
            // GIVEN
            const client = newEksClientMock();
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Delete',
            }));
            // WHEN
            const onEventResponse = await handler.onEvent();
            // THEN
            sinon.assert.calledWithExactly(client.configureAssumeRole, {
                RoleArn: 'AssumeRoleArn',
                RoleSessionName: 'AWSCDK.EKSCluster.Delete.RequestIdMock',
            });
            sinon.assert.calledWithExactly(client.deleteFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(onEventResponse, undefined);
            test.done();
        },
        async 'isComplete returns true when describeFargateProfile throws ResourceNotFoundException'(test) {
            // GIVEN
            const client = newEksClientMock();
            const resourceNotFoundError = new Error();
            resourceNotFoundError.code = 'ResourceNotFoundException';
            client.describeFargateProfile = sinon.fake.throws(resourceNotFoundError);
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock({
                RequestType: 'Delete',
            }));
            const resp = await handler.isComplete();
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.deepEqual(resp, {
                IsComplete: true,
            });
            test.done();
        },
        async 'isComplete throws an exception if the status is DELETE_FAILED'(test) {
            // GIVEN
            const client = newEksClientMock();
            client.describeFargateProfile = sinon.fake.returns({
                fargateProfile: {
                    status: 'DELETE_FAILED',
                },
            });
            // WHEN
            const handler = new fargate_1.FargateProfileResourceHandler(client, newRequestMock());
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            // THEN
            sinon.assert.calledWithExactly(client.describeFargateProfile, {
                clusterName: 'MockClusterName',
                fargateProfileName: 'PhysicalResourceIdMock',
            });
            test.equal(error.message, 'DELETE_FAILED');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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