"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOrgObj = exports.OrgObjectTypes = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The supported OrgObject types.
 *
 * @stability stable
 */
var OrgObjectTypes;
(function (OrgObjectTypes) {
    OrgObjectTypes["OU"] = "ou";
    OrgObjectTypes["ACCOUNT"] = "account";
})(OrgObjectTypes = exports.OrgObjectTypes || (exports.OrgObjectTypes = {}));
;
;
/**
 * @function processOrgObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} provider The custom resource provider the custom resources will utilized to create the resources.
 * @param {OrgObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOrgObj(provider, obj, parent) {
    var _c;
    if (obj.type === OrgObjectTypes.OU) {
        const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
        let props = obj.properties;
        let id = (_c = obj.id) !== null && _c !== void 0 ? _c : obj.properties.name;
        const ou = new OrganizationOU(this, id, {
            provider,
            parent: parentStr,
            name: props.name,
            importOnDuplicate: props.importOnDuplicate,
            allowMergeOnMove: props.allowMergeOnMove,
            allowRecreateOnUpdate: props.allowRecreateOnUpdate,
        });
        obj.children.forEach(child => {
            processOrgObj.call(this, provider, child, ou);
        });
    }
}
exports.processOrgObj = processOrgObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.2.1" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id);
        const importOnDuplicate = (_c = props.importOnDuplicate) !== null && _c !== void 0 ? _c : false;
        const allowMergeOnMove = (_d = props.allowMergeOnMove) !== null && _d !== void 0 ? _d : false;
        const allowRecreateOnUpdate = (_e = props.allowRecreateOnUpdate) !== null && _e !== void 0 ? _e : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowMergeOnMove: allowMergeOnMove,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.2.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IsNkNBT3FCO0FBQ3JCLDJDQUF1Qzs7Ozs7O0FBR3ZDLElBQVksY0FHWDtBQUhELFdBQVksY0FBYztJQUN4QiwyQkFBUyxDQUFBO0lBQ1QscUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUhXLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBR3pCO0FBWUEsQ0FBQztBQVFELENBQUM7QUFjRjs7Ozs7Ozs7R0FRRztBQUNILFNBQWdCLGFBQWEsQ0FBa0IsUUFBbUMsRUFBRSxHQUFjLEVBQUUsTUFBK0I7O0lBQ2pJLElBQUksR0FBRyxDQUFDLElBQUksS0FBSyxjQUFjLENBQUMsRUFBRSxFQUFFO1FBQ2xDLE1BQU0sU0FBUyxHQUFHLE1BQU0sWUFBWSxjQUFjLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFbEYsSUFBSSxLQUFLLEdBQVksR0FBRyxDQUFDLFVBQVUsQ0FBQztRQUNwQyxJQUFJLEVBQUUsU0FBRyxHQUFHLENBQUMsRUFBRSxtQ0FBSSxHQUFHLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztRQUV2QyxNQUFNLEVBQUUsR0FBRyxJQUFJLGNBQWMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3RDLFFBQVE7WUFDUixNQUFNLEVBQUUsU0FBUztZQUNqQixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxxQkFBcUI7U0FDbkQsQ0FBQyxDQUFDO1FBRUgsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDM0IsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNoRCxDQUFDLENBQUMsQ0FBQztLQUNKO0FBQ0gsQ0FBQztBQXBCRCxzQ0FvQkM7Ozs7Ozs7O0FBU0QsTUFBYSxzQkFBdUIsU0FBUSxzQkFBUzs7OztJQUluRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtDO1FBQzFFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxJQUFlLENBQUM7UUFFcEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDZixJQUFJLEdBQUcsSUFBSSxxQkFBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO2dCQUNoQyxTQUFTLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILElBQUksTUFBTSxHQUFHLElBQUkscUJBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtnQkFDakQsVUFBVSxFQUFFO29CQUNWLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQ3RCLE9BQU8sRUFBRTs0QkFDUCx3Q0FBd0M7NEJBQ3hDLHdDQUF3Qzs0QkFDeEMsZ0RBQWdEOzRCQUNoRCx3Q0FBd0M7eUJBQ3pDO3dCQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztxQkFDakIsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUM5QixJQUFJLENBQUMsZ0JBQWdCLENBQUMscUJBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUMsQ0FBQyxDQUFDO1NBQy9HO2FBQU07WUFDTCxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztTQUNuQjtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRXpELE1BQU0sT0FBTyxHQUFHLElBQUksd0JBQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNuRCxPQUFPLEVBQUUsd0JBQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxJQUFJLEVBQUUsd0JBQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUM7WUFDakQsT0FBTyxFQUFFLGdCQUFnQjtZQUN6QixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUk7U0FDTCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksOEJBQWdCLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUQsY0FBYyxFQUFFLE9BQU87WUFDdkIsWUFBWSxFQUFFLHNCQUFJLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUFoREgsd0RBaURDOzs7Ozs7Ozs7O0FBV0QsTUFBYSxjQUFlLFNBQVEsc0JBQVM7Ozs7SUFJM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjs7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGlCQUFpQixTQUFHLEtBQUssQ0FBQyxpQkFBaUIsbUNBQUksS0FBSyxDQUFDO1FBQzNELE1BQU0sZ0JBQWdCLFNBQUcsS0FBSyxDQUFDLGdCQUFnQixtQ0FBSSxLQUFLLENBQUM7UUFDekQsTUFBTSxxQkFBcUIsU0FBRyxLQUFLLENBQUMscUJBQXFCLG1DQUFJLEtBQUssQ0FBQztRQUVuRSxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsTUFBTSxZQUFZLGNBQWMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRXBHLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSw0QkFBYyxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUU7WUFDN0MsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWTtZQUN6QyxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLFNBQVM7Z0JBQ2pCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtnQkFDaEIsaUJBQWlCLEVBQUUsaUJBQWlCO2dCQUNwQyxnQkFBZ0IsRUFBRSxnQkFBZ0I7Z0JBQ2xDLHFCQUFxQixFQUFFLHFCQUFxQjthQUM3QztTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakQsSUFBSSxLQUFLLENBQUMsTUFBTSxZQUFZLGNBQWMsRUFBRTtZQUMxQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2hEO1FBQUEsQ0FBQztJQUNKLENBQUM7O0FBNUJILHdDQTZCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQge1xuICBjdXN0b21fcmVzb3VyY2VzLFxuICBhd3NfbGFtYmRhIGFzIGxhbWJkYSxcbiAgYXdzX2lhbSBhcyBpYW0sXG4gIEN1c3RvbVJlc291cmNlLFxuICBhd3NfbG9ncyBhcyBsb2dzLFxuICBEdXJhdGlvbixcbn0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIE9yZ09iamVjdFR5cGVzIHtcbiAgT1UgPSAnb3UnLFxuICBBQ0NPVU5UID0gJ2FjY291bnQnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE9VUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW1wb3J0T25EdXBsaWNhdGU/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93TWVyZ2VPbk1vdmU/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGxvd1JlY3JlYXRlT25VcGRhdGU/OiBib29sZWFuO1xufTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFjY291bnRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbWFpbDogc3RyaW5nO1xufTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcmdPYmplY3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaWQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb3BlcnRpZXM6IE9VUHJvcHMgfCBBY2NvdW50UHJvcHM7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0eXBlOiBPcmdPYmplY3RUeXBlcztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjaGlsZHJlbjogT3JnT2JqZWN0W107XG59XG5cbi8qKlxuICogQGZ1bmN0aW9uIHByb2Nlc3NPcmdPYmpcbiAqIEZ1bmN0aW9uIHRvIHByb2Nlc3MgYW4gT3JnT2JqZWN0IGFuZCBjcmVhdGUgdGhlIGNvcnJlc3BvbmRpbmcgQVdTIHJlc291cmNlc1xuICpcbiAqIEBwYXJhbSB7Q29uc3RydWN0fSB0aGlzIFRoZSBjb25zdHJ1Y3QgcmVzb3VyY2VzIHdpbGwgYmUgYWRkZWQgdG8uXG4gKiBAcGFyYW0ge2N1c3RvbV9yZXNvdXJjZXMuUHJvdmlkZXJ9IHByb3ZpZGVyIFRoZSBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIgdGhlIGN1c3RvbSByZXNvdXJjZXMgd2lsbCB1dGlsaXplZCB0byBjcmVhdGUgdGhlIHJlc291cmNlcy5cbiAqIEBwYXJhbSB7T3JnT2JqZWN0fSBvYmogVGhlIE9yZ09iamVjdCB0byBwcm9jZXNzLlxuICogQHBhcmFtIHtzdHJpbmcgfCBPcmdhbml6YXRpb25PVX0gcGFyZW50IFRoZSBwYXJlbnQgb2YgdGhlIE9yZ09iamVjdC4gVGhpcyBpcyBlaXRoZXIgYSBzdHJpbmcsIGxpa2UgZm9yIHRoZSBvcmcgcm9vdCwgb3IgYW4gT3JnYW5pemF0aW9uT1Ugb2JqZWN0IGZyb20gdGhlIHNhbWUgc3RhY2suXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBwcm9jZXNzT3JnT2JqKHRoaXM6IENvbnN0cnVjdCwgcHJvdmlkZXI6IGN1c3RvbV9yZXNvdXJjZXMuUHJvdmlkZXIsIG9iajogT3JnT2JqZWN0LCBwYXJlbnQ6IHN0cmluZyB8IE9yZ2FuaXphdGlvbk9VKSB7XG4gIGlmIChvYmoudHlwZSA9PT0gT3JnT2JqZWN0VHlwZXMuT1UpIHtcbiAgICBjb25zdCBwYXJlbnRTdHIgPSBwYXJlbnQgaW5zdGFuY2VvZiBPcmdhbml6YXRpb25PVSA/IHBhcmVudC5yZXNvdXJjZS5yZWYgOiBwYXJlbnQ7XG5cbiAgICBsZXQgcHJvcHM6IE9VUHJvcHMgPSBvYmoucHJvcGVydGllcztcbiAgICBsZXQgaWQgPSBvYmouaWQgPz8gb2JqLnByb3BlcnRpZXMubmFtZTtcblxuICAgIGNvbnN0IG91ID0gbmV3IE9yZ2FuaXphdGlvbk9VKHRoaXMsIGlkLCB7XG4gICAgICBwcm92aWRlcixcbiAgICAgIHBhcmVudDogcGFyZW50U3RyLFxuICAgICAgbmFtZTogcHJvcHMubmFtZSxcbiAgICAgIGltcG9ydE9uRHVwbGljYXRlOiBwcm9wcy5pbXBvcnRPbkR1cGxpY2F0ZSxcbiAgICAgIGFsbG93TWVyZ2VPbk1vdmU6IHByb3BzLmFsbG93TWVyZ2VPbk1vdmUsXG4gICAgICBhbGxvd1JlY3JlYXRlT25VcGRhdGU6IHByb3BzLmFsbG93UmVjcmVhdGVPblVwZGF0ZSxcbiAgICB9KTtcblxuICAgIG9iai5jaGlsZHJlbi5mb3JFYWNoKGNoaWxkID0+IHtcbiAgICAgIHByb2Nlc3NPcmdPYmouY2FsbCh0aGlzLCBwcm92aWRlciwgY2hpbGQsIG91KTtcbiAgICB9KTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcmdhbml6YXRpb25PVVByb3ZpZGVyUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9yZ2FuaXphdGlvbk9VUHJvdmlkZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIHB1YmxpYyByZWFkb25seSBwcm92aWRlcjogY3VzdG9tX3Jlc291cmNlcy5Qcm92aWRlcjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogT3JnYW5pemF0aW9uT1VQcm92aWRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGxldCByb2xlOiBpYW0uSVJvbGU7XG5cbiAgICBpZiAoIXByb3BzLnJvbGUpIHtcbiAgICAgIHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ3JvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgICAgfSk7XG5cbiAgICAgIGxldCBwb2xpY3kgPSBuZXcgaWFtLk1hbmFnZWRQb2xpY3kodGhpcywgJ3BvbGljeScsIHtcbiAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgJ29yZ2FuaXphdGlvbnM6VXBkYXRlT3JnYW5pemF0aW9uYWxVbml0JyxcbiAgICAgICAgICAgICAgJ29yZ2FuaXphdGlvbnM6RGVsZXRlT3JnYW5pemF0aW9uYWxVbml0JyxcbiAgICAgICAgICAgICAgJ29yZ2FuaXphdGlvbnM6TGlzdE9yZ2FuaXphdGlvbmFsVW5pdHNGb3JQYXJlbnQnLFxuICAgICAgICAgICAgICAnb3JnYW5pemF0aW9uczpDcmVhdGVPcmdhbml6YXRpb25hbFVuaXQnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgcm9sZS5hZGRNYW5hZ2VkUG9saWN5KHBvbGljeSk7XG4gICAgICByb2xlLmFkZE1hbmFnZWRQb2xpY3koaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJykpO1xuICAgIH0gZWxzZSB7XG4gICAgICByb2xlID0gcHJvcHMucm9sZTtcbiAgICB9XG5cbiAgICBjb25zdCBoYW5kbGVyc1BhdGggPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vaGFuZGxlcnMnKTtcblxuICAgIGNvbnN0IG9uRXZlbnQgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdoYW5kbGVyJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOSxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChoYW5kbGVyc1BhdGggKyAnL291JyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXgub25fZXZlbnQnLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygxMCksXG4gICAgICByb2xlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wcm92aWRlciA9IG5ldyBjdXN0b21fcmVzb3VyY2VzLlByb3ZpZGVyKHRoaXMsICdwcm92aWRlcicsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBvbkV2ZW50LFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgT1VSZXNvdXJjZVByb3BzIGV4dGVuZHMgT1VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGFyZW50OiBzdHJpbmcgfCBPcmdhbml6YXRpb25PVTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb3ZpZGVyOiBjdXN0b21fcmVzb3VyY2VzLlByb3ZpZGVyO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JnYW5pemF0aW9uT1UgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIHB1YmxpYyByZWFkb25seSByZXNvdXJjZTogQ3VzdG9tUmVzb3VyY2U7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE9VUmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBpbXBvcnRPbkR1cGxpY2F0ZSA9IHByb3BzLmltcG9ydE9uRHVwbGljYXRlID8/IGZhbHNlO1xuICAgIGNvbnN0IGFsbG93TWVyZ2VPbk1vdmUgPSBwcm9wcy5hbGxvd01lcmdlT25Nb3ZlID8/IGZhbHNlO1xuICAgIGNvbnN0IGFsbG93UmVjcmVhdGVPblVwZGF0ZSA9IHByb3BzLmFsbG93UmVjcmVhdGVPblVwZGF0ZSA/PyBmYWxzZTtcblxuICAgIGNvbnN0IHBhcmVudFN0ciA9IHByb3BzLnBhcmVudCBpbnN0YW5jZW9mIE9yZ2FuaXphdGlvbk9VID8gcHJvcHMucGFyZW50LnJlc291cmNlLnJlZiA6IHByb3BzLnBhcmVudDtcblxuICAgIHRoaXMucmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ291Jywge1xuICAgICAgc2VydmljZVRva2VuOiBwcm9wcy5wcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIFBhcmVudDogcGFyZW50U3RyLFxuICAgICAgICBOYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgICBJbXBvcnRPbkR1cGxpY2F0ZTogaW1wb3J0T25EdXBsaWNhdGUsXG4gICAgICAgIEFsbG93TWVyZ2VPbk1vdmU6IGFsbG93TWVyZ2VPbk1vdmUsXG4gICAgICAgIEFsbG93UmVjcmVhdGVPblVwZGF0ZTogYWxsb3dSZWNyZWF0ZU9uVXBkYXRlLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMucmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLnByb3ZpZGVyKTtcbiAgICBpZiAocHJvcHMucGFyZW50IGluc3RhbmNlb2YgT3JnYW5pemF0aW9uT1UpIHtcbiAgICAgIHRoaXMucmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLnBhcmVudCk7XG4gICAgfTtcbiAgfVxufVxuXG4vLyBleHBvcnQgaW50ZXJmYWNlIE9yZ2FuaXphdGlvbkFjY291bnRQcm9wcyB7XG4vLyAgIC8qKlxuLy8gICAgKiBUaGUgbmFtZSBvZiB0aGUgYWNjb3VudFxuLy8gICAgKi9cbi8vICAgIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbi8vICAgLyoqXG4vLyAgICAqIFRoZSBlbWFpbCBmb3IgdGhlIGFjY291bnRcbi8vICAgICovXG4vLyAgICByZWFkb25seSBlbWFpbDogc3RyaW5nO1xuLy8gICAvKipcbi8vICAgKiBUaGUgT1UgaWQgZm9yIHRoZSBhY2NvdW50LiBUaGlzIHByb3BlcnR5IGlzIGFsd2F5cyBwcmVmZXJyZWQgaWYgaWQgYW5kIG5hbWUgYXJlIGJvdGggZ2l2ZW4uIERlZmF1bHRzIHRvIHRoZSByb290XG4vLyAgICovXG4vLyAgICByZWFkb25seSBvdV9pZD86IHN0cmluZztcbi8vICAgLyoqXG4vLyAgICAqIFRoZSBPVSBuYW1lIGZvciB0aGUgYWNjb3VudC4gVGhpcyBwcm9wZXJ0eSBpcyBpZ25vcmVkIGlmIHRoZSBvdSBpZCBpcyBnaXZlbi5cbi8vICAgICovXG4vLyAgICByZWFkb25seSBvdV9uYW1lPzogc3RyaW5nO1xuLy8gfVxuXG4vLyBleHBvcnQgY2xhc3MgT3JnYW5pemF0aW9uQWNjb3VudCBleHRlbmRzIENvbnN0cnVjdCB7XG4vLyAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBPcmdhbml6YXRpb25BY2NvdW50UHJvcHMpIHtcbi8vICAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4vLyAgICAgLy8gY29kZVxuLy8gICB9XG4vLyB9XG4iXX0=