# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['excel_exporter',
 'excel_exporter.configuration',
 'excel_exporter.exporter',
 'excel_exporter.utils']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0', 'openpyxl>=3.1.1,<4.0.0']

entry_points = \
{'console_scripts': ['example = example:example']}

setup_kwargs = {
    'name': 'excel-exporter-bms',
    'version': '0.1.4',
    'description': 'An excel exporter based on YAML configuration files',
    'long_description': '# Excel Exporter\n\n## Motivation\n\nFor many years, I have been creating excel spreadsheets with a similar look and feel. My template consists of data organized into excel tables with similar categories having the same background colors and a merged cell with a category title above them (outside the table). The background colors are dark(ish) to provide contrast with a bold white font. I use blank columns before and after the table as borders and hide grid lines and unused columns.\n\n![Image showing an example excel sheet as described above (check homepage)](images/img01_spreadsheet.png)\n\nI have been manually creating these spreadsheets in Excel, but when I needed to update one of them every week, I decided to create a python script to automate the process. The script worked well, but it was very hard to maintain. If I needed to add or remove a column, I would have to make a lot of changes, and if I needed to adapt the script to another dataset, I would have to almost start from scratch.\n\nI promised myself that I would make a cleaner version of the excel exporter as soon as I had time. The original script was written using the pandas and xlswriter libraries, where I put the data on the sheet using pandas and formatted everything I needed using xlswriter. However, recently I needed to add hyperlinks to some cells in a particular column, and I was unable to do so with my current approach. So, I decided to completely rewrite my script to make it easier to adapt to new uses and allow for the possibility of adding hyperlinks to cells.\n\nI don\'t intend to maintain this repository. However, I believe that the new version makes it very easy to create spreadsheets with my template and may be useful to others. Additionally, I think that the code is organized well enough to be adapted to other templates or used as a reference on how to organize a script to generate excel spreadsheets with complex formatting.\n\n\n## The new approach\n\nThe Excel Exporter was created utilizing openpyxl and has the capability to export multiple sheets into a single Excel file.\n\nThe data being exported is sourced from dictionaries (one for each sheet). In cases where the data contains associated URLs, the exporter will automatically convert them into hyperlinks within the Excel file.\n\nThe following animation shows how the data is organized:\n   \n![Gif animation showing the structure of data (check homepage)](images/img02_data.gif)\n\n Configurations for columns, groups, sheets, and the workbook as a whole are stored in a YAML file.\n - Workbook: File name, Cell formats and Sheets.\n - Cell Formats: Font size, Horizontal alignment, Vertical alignment and Line break\n - Sheets: Sheet Name, Group(s), and Column(s)\n - Groups: Name and Background Color\n - Columns: Order, Title (to be display in Excel), Variable Name (in the python dictionary), Cell format (previouly defined), Group it belongs to (previously defined) and  Column Width\n \n The following animation shows how the YAML file is organized:\n\n![Gif animation showing the structure of the YAML file (check homepage)](images/img03_yaml.gif)\n\n## How to install\n\nJust use pip\n\n```\npip install excel-exporter-bms\n```\n\nor poetry\n\n```\npoestry add excel-exporter-bms\n```\n\n## How to use\n\nTo run this tutorial you should download [example.py](https://github.com/brunomsantiago/excel_exporter/blob/master/example.py) and [example_config](https://github.com/brunomsantiago/excel_exporter/blob/master/example_config.yaml). It is expected to generate a file like [example_output.xlsx](https://github.com/brunomsantiago/excel_exporter/raw/master/example_output.xlsx)\n\n### 1. Prepare the Data\nIt\'s necessary to properly prepare your data before using the exporter.\nThe data must be a list of dictionary of lists, where each dictionary represents a sheet in the excel file.\nInside the dictionary each key represents a column and its corresponding list contains the column data. Each element in the list represents a cell value.\nIf a hyperlink is required, the cell value should be represented as a tuple, with the first element being the cell value and the second element being the URL for the hyperlink.\n```python\ndata = [\n    {\n        "employee_id": [(1001, \'https://employee.my_company.com/id/1001\'),\n                        (1002, \'https://employee.my_company.com/id/1002\')],\n        "employee_name": ["John Doe", "Jane Doe"],\n        "department": ["Sales South Area", "Sales West Area"],\n        "job_title": ["Manager", "Associate"]\n    },\n    {\n        "employee_id": [1001, 1002],\n        "sales_current_year": [100000, 120000],\n        "sales_last_year": [90000, 110000]\n    }\n]\n```\n\n- If your data originally comes from a list of dictionaries, you can use the `utils` module to convert it.\n```python\nfrom excel_exporter.utils.data_conversion import from_list_of_dicts\n\nsheet1_data = from_list_of_dicts(sheet1_list_of_dicts)\ndata = [sheet1_data]\n```\n- If your data is in a pandas dataframe, a similar conversion process applies.\n```python\nfrom excel_exporter.utils.data_conversion import from_pandas_dataframe\n\nsheet1_data = from_pandas_dataframe(sheet1_df)\ndata = [sheet1_data]\n```\n### 2. Prepare the YAML\nConfigure the YAML based on the example described on the previous section\n\n\n### 3. Load the YAML\nBefore using the exporter, you\'ll need to load the YAML configuration file that you\'ll be using to export your data.\n```python\nfrom excel_exporter.configuration.load_config import load_config\n\nconfig = load_config(\'example_config.yaml\')\n```\n\n### 4. Retrieve the Current Date and Time\nThis exporter was created for periodically updated spreadsheets. So it also important to include the time and date of the dataset. These can be retrieved using datetime module.\n```python\nfrom datetime import datetime\n\nupdate_time = datetime.now()\n```\n\n### 5. Run the Exporter\nOnce your data is prepared and the YAML is loaded, you\'re ready to run the exporter. Simply pass your data, configuration and update time as arguments to the exporter, and the export process will begin. It will return a virtual file (BytesIO)\n```python\nfrom excel_exporter.exporter.create_excel import export_excel\n\nvirtual_file = export_excel(data, config, update_time)\n```\n### 6. Save or upload the file\nOnce the export process is complete, you can choose to either save the virtual file as a real file on your local machine, or upload it to a remote server using an API. Here is an example of how to save the file locally:\n```python\nwith open(config.file_name, \'wb\') as file:\n    file.write(virtual_file.getbuffer())\n```',
    'author': 'Bruno Santiago',
    'author_email': 'not_a_real_email_13124123@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/brunomsantiago/excel_exporter',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
