"""
This module creates a Python object DataFrameRowGroup for group by dataframe.
"""
from typing import (
    TYPE_CHECKING,
    Any,
    Callable,
    Iterable,
    List,
    MutableSet,
    Optional,
    Set,
)

from greenplumpython.expr import Expr, serialize

if TYPE_CHECKING:
    from greenplumpython.dataframe import DataFrame
    from greenplumpython.func import FunctionExpr


class DataFrameGroupingSets:
    """
    Represents a group of rows in a GreenplumPython :class:`~dataframe.DataFrame` generated by
    :func:`~dataframe.DataFrame.group_by`.
    """

    def __init__(self, dataframe: "DataFrame", grouping_sets: List[Iterable["Expr"]]) -> None:
        self._dataframe = dataframe
        self._grouping_sets = grouping_sets

    def apply(
        self,
        func: Callable[["DataFrame"], "FunctionExpr"],
        expand: bool = False,
        as_name: Optional[str] = None,
    ) -> "DataFrame":
        """
        Apply a function to the grouping set.

        Args:
            func: An aggregate function to be applied to
            expand: expand field of composite returning type
            as_name: rename returning column

        Returns:
            DataFrame: resulted GreenplumPython DataFrame

        Example:
            .. code-block::  python

                numbers.group_by("is_even").apply(lambda _: count())
        """
        return func(self._dataframe).bind(group_by=self).apply(expand=expand, as_name=as_name)

    def assign(self, **new_columns: Callable[["DataFrame"], Any]) -> "DataFrame":
        """
        Assigns new columns to the current grouping sets. **Existing columns
        cannot be reassigned**.

        Args:
            new_columns: a :class:`dict` whose keys are column names and values
                are :class:`Callable`s returning column data when applied to the
                current :class:`DataFrameGroupingSets`.

        Returns:
            GreenplumPython :class:`DataFrame` with the new columns.


        Example:
            .. highlight:: python
            .. code-block::  python

                >>> rows = [(1,) for _ in range(10)]
                >>> numbers = db.create_dataframe(rows=rows, column_names=["val"])
                >>> count = gp.aggregate_function("count")
                >>> results = numbers.group_by().assign(count=lambda t: count(t["val"]))
                >>> results
                -------
                 count
                -------
                    10
                -------
                (1 row)

        """
        from greenplumpython.dataframe import DataFrame

        targets: List[str] = list(self.flatten())
        for k, f in new_columns.items():
            v: Any = f(self.dataframe).bind(group_by=self)
            if isinstance(v, Expr) and not (v.dataframe is None or v.dataframe == self.dataframe):
                raise Exception("Newly included columns must be based on the current dataframe")
            targets.append(f"{serialize(v)} AS {k}")
        return DataFrame(
            f"SELECT {','.join(targets)} FROM {self.dataframe.name} {self.clause()}",
            parents=[self.dataframe],
        )

    def union(
        self, other: Callable[["DataFrame"], "DataFrameGroupingSets"]
    ) -> "DataFrameGroupingSets":
        """
        Union with another :class:`DataFrameGroupingSets` so that when applying an
        agggregate function to the list, the function will be applied to
        each grouping set individually.

        Args:
            other: a :class:`Callable` returning the result of
                :func:`DataFrame.group_by()`when applied to the current :class:`DataFrame`.
        """
        return DataFrameGroupingSets(
            self._dataframe,
            self._grouping_sets + other(self._dataframe)._grouping_sets,
        )

    def flatten(self) -> Set[str]:
        """:meta private:"""
        item_set: MutableSet[Expr] = set()
        for grouping_set in self._grouping_sets:
            for item in grouping_set:
                assert isinstance(item, str), f"Grouping item {item} is not a column name."
                item_set.add(item)
        return item_set

    @property
    def dataframe(self) -> "DataFrame":
        """
        Returns GreenplumPython :class:`~dataframe.DataFrame` associated for GROUP BY

        Returns:
            GreenplumPython DataFrame
        """
        return self._dataframe

    # FIXME: Make this function package-private
    def clause(self) -> str:
        """:meta private:"""
        grouping_sets_str = [
            f"({','.join([item for item in grouping_set])})" for grouping_set in self._grouping_sets
        ]
        return "GROUP BY GROUPING SETS " + f"({','.join(grouping_sets_str)})"
