"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsEdgeFunction = void 0;
const fs = require("fs");
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const lambda = require("@aws-cdk/aws-lambda");
const bundling = require("@aws-cdk/aws-lambda-nodejs/lib/bundling");
const package_manager_1 = require("@aws-cdk/aws-lambda-nodejs/lib/package-manager");
const util_1 = require("@aws-cdk/aws-lambda-nodejs/lib/util");
class NodejsEdgeFunction extends cloudfront.experimental.EdgeFunction {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : lambda.Runtime.NODEJS_14_X;
        const entry = path.resolve(findEntry(id, props.entry));
        const architecture = (_c = props.architecture) !== null && _c !== void 0 ? _c : lambda.Architecture.X86_64;
        const depsLockFilePath = findLockFile(props.depsLockFilePath);
        const projectRoot = (_d = props.projectRoot) !== null && _d !== void 0 ? _d : path.dirname(depsLockFilePath);
        super(scope, id, {
            ...props,
            runtime,
            stackId: props.stackId,
            code: bundling.Bundling.bundle({
                ...(_e = props.bundling) !== null && _e !== void 0 ? _e : {},
                architecture,
                runtime,
                depsLockFilePath,
                entry,
                projectRoot,
            }),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsEdgeFunction = NodejsEdgeFunction;
/**
 * Checks given lock file or searches for a lock file
 */
function findLockFile(depsLockFilePath) {
    if (depsLockFilePath) {
        if (!fs.existsSync(depsLockFilePath)) {
            throw new Error(`Lock file at ${depsLockFilePath} doesn't exist`);
        }
        if (!fs.statSync(depsLockFilePath).isFile()) {
            throw new Error('`depsLockFilePath` should point to a file');
        }
        return path.resolve(depsLockFilePath);
    }
    const lockFiles = util_1.findUpMultiple([
        package_manager_1.LockFile.PNPM,
        package_manager_1.LockFile.YARN,
        package_manager_1.LockFile.NPM,
    ]);
    if (lockFiles.length === 0) {
        throw new Error('Cannot find a package lock file (`pnpm-lock.yaml`, `yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
    }
    if (lockFiles.length > 1) {
        throw new Error(`Multiple package lock files found: ${lockFiles.join(', ')}. Please specify the desired one with \`depsFileLockPath\`.`);
    }
    return lockFiles[0];
}
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 * 4. A .mjs file name as the defining file with id as suffix (defining-file.id.mjs)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?|mjs)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    const mjsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.mjs`);
    if (fs.existsSync(mjsHandlerFile)) {
        return mjsHandlerFile;
    }
    throw new Error(`Cannot find handler file ${tsHandlerFile}, ${jsHandlerFile} or ${mjsHandlerFile}`);
}
/**
* Finds the name of the file where the `NodejsFunction` is defined
*/
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsEdgeFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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