"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParameterValidationError = exports.JwtBaseError = exports.CognitoJwtVerifier = void 0;
const assert_1 = require("./assert");
const error_1 = require("./error");
const jwt_rsa_1 = require("./jwt-rsa");
/**
 * Class representing a verifier for JWTs signed by Amazon Cognito
 */
class CognitoJwtVerifier extends jwt_rsa_1.JwtRsaVerifierBase {
    /**
   * Parse a User Pool ID, to extract the issuer and JWKS URI
   *
   * @param userPoolId The User Pool ID
   * @returns The issuer and JWKS URI for the User Pool
   */
    static parseUserPoolId(userPoolId) {
        // Disable safe regexp check as userPoolId is provided by developer, i.e. is not user input
        const match = userPoolId.match(/^(?<region>(\w+-)?\w+-\w+-\d)+_\w+$/);
        if (!match) {
            throw new ParameterValidationError(`Invalid Cognito User Pool ID: ${userPoolId}`);
        }
        const region = match.groups.region;
        const issuer = `https://cognito-idp.${region}.amazonaws.com/${userPoolId}`;
        return {
            issuer,
            jwksUri: `${issuer}/.well-known/jwks.json`,
        };
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    static create(verifyProperties, additionalProperties) {
        return new this(verifyProperties, additionalProperties === null || additionalProperties === void 0 ? void 0 : additionalProperties.jwksCache);
    }
    constructor(props, jwksCache) {
        const issuerConfig = Array.isArray(props)
            ? props.map((p) => ({
                ...p,
                ...CognitoJwtVerifier.parseUserPoolId(p.userPoolId),
                audience: null,
            }))
            : {
                ...props,
                ...CognitoJwtVerifier.parseUserPoolId(props.userPoolId),
                audience: null,
            };
        super(issuerConfig, jwksCache);
    }
    /**
     * Verify (synchronously) a JWT that is signed by Amazon Cognito.
     *
     * @param jwt The JWT, as string
     * @param props Verification properties
     * @returns The payload of the JWT––if the JWT is valid, otherwise an error is thrown
     */
    // public verifySync<T extends SpecificVerifyProperties>(
    //   ...[jwt, properties]: CognitoVerifyParameters<SpecificVerifyProperties>
    // ): CognitoIdOrAccessTokenPayload<IssuerConfig, T> {
    //   const { decomposedJwt, jwksUri, verifyProperties } =
    //     this.getVerifyParameters(jwt, properties);
    //   this.verifyDecomposedJwtSync(decomposedJwt, jwksUri, verifyProperties);
    //   try {
    //     validateCognitoJwtFields(decomposedJwt.payload, verifyProperties);
    //   } catch (err) {
    //     if (
    //       verifyProperties.includeRawJwtInErrors &&
    //       err instanceof JwtInvalidClaimError
    //     ) {
    //       throw err.withRawJwt(decomposedJwt);
    //     }
    //     throw err;
    //   }
    //   return decomposedJwt.payload as CognitoIdOrAccessTokenPayload<
    //     IssuerConfig,
    //     T
    //   >;
    // }
    /**
     * Verify (asynchronously) a JWT that is signed by Amazon Cognito.
     * This call is asynchronous, and the JWKS will be fetched from the JWKS uri,
     * in case it is not yet available in the cache.
     *
     * @param jwt The JWT, as string
     * @param props Verification properties
     * @returns Promise that resolves to the payload of the JWT––if the JWT is valid, otherwise the promise rejects
     */
    async verify(
    // ...[jwt, properties]: CognitoVerifyParameters<SpecificVerifyProperties>
    ...[jwt, properties]) {
        const { decomposedJwt, jwksUri, verifyProperties } = this.getVerifyParameters(jwt, properties);
        await this.verifyDecomposedJwt(decomposedJwt, jwksUri, verifyProperties);
        try {
            validateCognitoJwtFields(decomposedJwt.payload, verifyProperties);
        }
        catch (err) {
            if (verifyProperties.includeRawJwtInErrors &&
                err instanceof error_1.JwtInvalidClaimError) {
                throw err.withRawJwt(decomposedJwt);
            }
            throw err;
        }
        return decomposedJwt.payload;
    }
}
exports.CognitoJwtVerifier = CognitoJwtVerifier;
/**
 * Validate claims of a decoded Cognito JWT.
 * This function throws an error in case there's any validation issue.
 *
 * @param payload - The JSON parsed payload of the Cognito JWT
 * @param options - Validation options
 * @param options.groups - The cognito groups, of which at least one must be present in the JWT's cognito:groups claim
 * @param options.tokenUse - The required token use of the JWT: "id" or "access"
 * @param options.clientId - The required clientId of the JWT. May be an array of string, of which at least one must match
 * @returns void
 */
function validateCognitoJwtFields(payload, options) {
    // Check groups
    if (options.groups != null) {
        assert_1.assertStringArraysOverlap('Cognito group', payload['cognito:groups'], options.groups, error_1.CognitoJwtInvalidGroupError);
    }
    // Check token use
    assert_1.assertStringArrayContainsString('Token use', payload.token_use, ['id', 'access'], error_1.CognitoJwtInvalidTokenUseError);
    if (options.tokenUse !== null) {
        if (options.tokenUse === undefined) {
            throw new ParameterValidationError('tokenUse must be provided or set to null explicitly');
        }
        assert_1.assertStringEquals('Token use', payload.token_use, options.tokenUse, error_1.CognitoJwtInvalidTokenUseError);
    }
    // Check clientId aka audience
    if (options.clientId !== null) {
        if (options.clientId === undefined) {
            throw new ParameterValidationError('clientId must be provided or set to null explicitly');
        }
        if (payload.token_use === 'id') {
            assert_1.assertStringArrayContainsString('Client ID ("audience")', payload.aud, options.clientId, error_1.CognitoJwtInvalidClientIdError);
        }
        else {
            assert_1.assertStringArrayContainsString('Client ID', payload.client_id, options.clientId, error_1.CognitoJwtInvalidClientIdError);
        }
    }
}
/**
 * Base Error for all other errors in this file
 */
class JwtBaseError extends Error {
}
exports.JwtBaseError = JwtBaseError;
class ParameterValidationError extends JwtBaseError {
}
exports.ParameterValidationError = ParameterValidationError;
//# sourceMappingURL=data:application/json;base64,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