# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['simple_profile']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'simple-profile',
    'version': '1.1.0',
    'description': 'Simple profile decorators to monitor execution time and memory usage.',
    'long_description': '# simple-profile\n\nSimple decorators to profile the memory usage and execution time.\n\n## Installation\n\n```bash\npip install simple-profile\n```\n\n## Decorators\n\n| Decorator           | Description                                                     |\n|---------------------|-----------------------------------------------------------------|\n| `@simple_profile()` | Profiles the memory usage and the execution time of a function. |\n| `@memory_profile()` | Profiles only the memory usage of a function.                   |\n| `@time_profile()`   | Profiles only the execution time of a function.                 |\n\n## Usage\n\n### 1. Profile a function\n\nThe `@simple_profile()` decorator allows to log the peak memory usage and the average execution time of each function call.\\\nBy default, memory usage is logged in bytes and execution time in nanoseconds, but it is possible to change the units.\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile()\ndef my_function():\n    return [2 * i for i in range(10)]\n\nmy_function()\n```\n\nOutput: \n\n```\nmy_function | 312 B | 46.03 ns\n```\n\n### 2. Profile only the memory usage of a function\n\nThe `@memory_profile()` decorator allows to log the peak memory usage of each function call.\\\nThis is done using the `tracemalloc` module provided by Python.\n\n```python\nfrom simple_profile import memory_profile\n\n@memory_profile()\ndef my_function():\n    return [2 * i for i in range(10)]\n\nmy_function()\n```\n\nOutput: \n\n```\nmy_function | 312 B\n```\n\n### 3. Profile only the execution time of a function\n\nThe `@time_profile()` decorator allows to log the average execution time of each function call.\\\nThis is done using the `timeit` module provided by Python and by repeating each function call a number of times to get a reliable measurement.\\\nBy default, each function call is repeated `1,000,000` times, but it is possible to change this value.\n\n```python\nfrom simple_profile import time_profile\n\n@time_profile()\ndef my_function():\n    return [2 * i for i in range(10)]\n\nmy_function()\n```\n\nOutput: \n\n```\nmy_function | 45.05 ns\n```\n\n### 4. Change the number of iterations\n\nIt is possible to change the number of times a function call is repeated when profiling the execution time.\\\nTo do this, you can set the `iterations` argument of the `simple_profile()` and `time_profile()` decorators.\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile(iterations=100)\ndef pi(n):\n    result = 0\n    d = 1\n    for i in range(1, n):\n        a = 2 * (i % 2) - 1\n        result += 4 * a / d\n        d += 2\n    return result\n\npi(100)\n```\n\nOutput:\n\n```\npi | 80 B | 819.5 ns\n```\n\n### 5. Change the time and memory units\n\nIt is also possible to change the time and memory units used in the logs.\\\nTo do this, you can set the `unit` argument of the `memory_profile()` and `time_profile()` decorators.\\\nIn the case of the `simple_profile()` decorator, you can set the `time_unit` and `memory_unit` arguments.\n\n```python\nfrom simple_profile import simple_profile, MemoryUnit, TimeUnit\n\n@simple_profile(memory_unit=MemoryUnit.KILOBYTES, time_unit=TimeUnit.MILLISECONDS)\ndef exponential(x, n):\n    result = 1.0\n    for i in range(n, 0, -1):\n        result = 1 + x * result / i\n    return result\n\nexponential(8, 100)\n```\n\nOutput:\n\n```\nexponential | 0.008 kB | 0.000549 ms\n```\n\n### 6. Change the time and memory precision\n\nMoreover, it is possible to change the precision of memory and time values.\\\nTo do this, you can define the number of significant digits you want in the `precision` argument of any decorator provided by this package.\\\nIn the case of the `simple_profile()` decorator, you can set the `time_precision` and `memory_precision` arguments for more granular control.\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile(precision=10)\ndef average(lst):\n    return sum(lst) / len(lst)\n\naverage([25, 12, 18, 88, 64, 55, 22])\n```\n\nOutput:\n\n```\naverage | 48 B | 17.27122 ns\n```\n\n### 7. Log the arguments and the result\n\nFurthermore, it is possible to log the arguments and the result of each function call.\\\nIndeed, this can be useful to better profile a function and analyze its behavior.\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile(print_args=True, print_result=True)\ndef greeting_message(name, coins):\n    return "Hello {}! You have {} coins.".format(name, coins)\n\ngreeting_message("John", coins=5)\n```\n\nOutput:\n\n```\ngreeting_message | John, coins=5 | Hello John! You have 5 coins. | 353 B | 31.89 ns\n```\n\n### 8. Set a custom name for a function\n\nAdditionally, it is possible to define a custom descriptive name for each function.\\\nTo do this, you can set the `name` argument of any decorator provided by this package.\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile(name="Naive method")\ndef factorial(n):\n    result = 1\n    for i in range(1, n + 1):\n        result *= i\n    return result\n\nfactorial(10)\n```\n\nOutput:\n\n```\nNaive method | 96 B | 38.03 ns\n```\n\n### 9. Compare multiple functions\n\n```python\nfrom simple_profile import simple_profile\n\n@simple_profile(name="List comprehension")\ndef my_function(n):\n    return [pow(2, i) for i in range(n)]\n\n\n@simple_profile(name="For loop")\ndef my_function_2(n):\n    lst = []\n    for i in range(n):\n        lst.append(pow(2, i))\n    return lst\n\nmy_function(10)\nmy_function_2(10)\n```\n\nOutput:\n\n```\nList comprehension | 344 B | 62.61 ns\nFor loop | 192 B | 65.3 ns\n```\n',
    'author': 'João Brilhante',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/JoaoBrlt/simple-profile',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.11,<4.0',
}


setup(**setup_kwargs)
