"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const api_call_base_1 = require("./api-call-base");
const providers_1 = require("./providers");
const waiter_state_machine_1 = require("./waiter-state-machine");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends api_call_base_1.ApiCallBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider');
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.api = props.api;
        this.service = props.service;
        this._outputPaths = props.outputPaths;
        this.apiCallResource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                expected: aws_cdk_lib_1.Lazy.any({ produce: () => this.expectedResult }),
                actualPath: aws_cdk_lib_1.Lazy.string({ produce: () => this._assertAtPath }),
                stateMachineArn: aws_cdk_lib_1.Lazy.string({ produce: () => this.stateMachineArn }),
                parameters: this.provider.encode(props.parameters),
                flattenResponse: aws_cdk_lib_1.Lazy.string({ produce: () => this.flattenResponse }),
                outputPaths: aws_cdk_lib_1.Lazy.list({ produce: () => this._outputPaths }),
                salt: Date.now().toString(),
            },
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`.substring(0, 60),
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.apiCallResource.node.addDependency(this.provider);
        // if expectedResult has been configured then that means
        // we are making assertions and we should output the results
        aws_cdk_lib_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall) {
                    if (node.expectedResult) {
                        const result = node.apiCallResource.getAttString('assertion');
                        new aws_cdk_lib_1.CfnOutput(node, 'AssertionResults', {
                            value: result,
                        }).overrideLogicalId(`AssertionResults${id}`);
                    }
                }
            },
        });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        this._assertAtPath = path;
        this._outputPaths = [path];
        this.expectedResult = expected.result;
        this.flattenResponse = 'true';
        return this;
    }
    waitForAssertions(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAssertions);
            }
            throw error;
        }
        const waiter = new waiter_state_machine_1.WaiterStateMachine(this, 'WaitFor', {
            ...options,
        });
        this.stateMachineArn = waiter.stateMachineArn;
        this.provider.addPolicyStatementFromSdkCall('states', 'StartExecution');
        waiter.isCompleteProvider.addPolicyStatementFromSdkCall(this.service, this.api);
        this.waiterProvider = waiter.isCompleteProvider;
        return this;
    }
}
exports.AwsApiCall = AwsApiCall;
_a = JSII_RTTI_SYMBOL_1;
AwsApiCall[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.63.0-alpha.0" };
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType = exports.LogType || (exports.LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this istead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = aws_cdk_lib_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new aws_cdk_lib_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
_b = JSII_RTTI_SYMBOL_1;
LambdaInvokeFunction[_b] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.63.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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