# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['maxar_ard_grid']

package_data = \
{'': ['*']}

install_requires = \
['Shapely>=1.6.4', 'pyproj>=2.2.1']

setup_kwargs = {
    'name': 'maxar-ard-grid',
    'version': '1.2.1',
    'description': 'Tools for the Maxar ARD grid',
    'long_description': '## maxar-ard-grid\n\nTools and utils for the Maxar ARD Grid system and COG tiles.\n\nFor more information on the grid and ARD see the [main documentation page](https://ard.maxar.com/docs/about/)\n\n### Installation\n\nUsing pip:\n\n```\npip install maxar-ard-grid \n```\n\n\n### The ARD Grid\n\nMaxar ARD uses a grid system based on UTM zones. Each zone is divided using a quadkey scheme that divides the zone into useful grid **cells**.\n\nAt the 12th zoom level each cell is 5km square. We use these cells as a boundary to subset image strips into **tiles**. The image tiles are Cloud Optimized Geotiffs stored in S3. The tiffs use a normalized resolution of approximately 30cm. The size and resolution of the tiffs maps to 16,384 pixels inside of each cell (the COGS are generated with an additional 512 pixel apron outside of the cell boundary). Because 16,384 is a power of two, it can be further subdivided with quadkeys along pixel boundaries.\n\nAt the 18th zoom level we have a cell that is 256 pixels in size. This is a convenient size for a chip. Each chip therefore has a unique quadkey describing its bounding box. Fetching image data for this cell will alway retrieve the same square on the earth. Because 256 is also a power of two, that means every pixel also has a unique quadkey identifier on the globe.\n\nThis library does not access these tiles. It is only intended to handle the grid and its cells.\n\n\n### The Basics of the ARD Grid\n\n#### Finding grid cells\n\nTo find out which grid cells cover an area, use the `covers()` function:\n\n```pycon\n>>> from maxar_ard_grid import covers\n>>> geom = \'POINT(40 -120)\'\n>>> covers(geom)\n<generator object covers at 0x11d9dfe08>>\n```\n\nThe `covers` function returns a generator of Cell objects. Unless specified the Cells will be generated at grid zoom level 12 to match the COG tiles. These two are equivalent:\n\n```pycon\n>>> covers(geom)\n>>> covers(geom, zoom=12)\n```\n\nThe following are acceptable formats for the input geometry:\n\n- Objects using the Python Geospatial Interface \n- Shapely geometries\n- ARD Cells\n- ARD Cell IDs\n- GeoJSON-like geometry dicts\n- GeoJSON-like feature dicts\n- Well Know Text (WKT)\n\nGeometries are assumed to be in the WGS84 coordinate system. If the geometry is in a different coordinate system or projection, it can be specified using the `src_proj` keyword. Acceptable inputs are PyProj `CRS` objects or EPSG codes (as strings or integers):\n\n```pycon\n>>> geom = \'POINT(243900 4432069)\'\n>>> covers(geom, src_proj=32611) # UTM 11N\n```\n\nWhen finding the cells covered by a given cell, always use the cell object or identifier. Using a cell\'s bounds or geospatial interface as an input will return unwanted cells that share edges. It is also possible to use the `Cell.get_children_at_zoom()` to calculate the same cells.\n\n#### Working with Cells\n\nEach cell in a zone\'s grid has a unique quadkey identifier. When combined with the the zone\'s UTM number the result is a unique cell on Earth. These cell IDs use the following format:\n\n`Z{zone}-{quadkey}`\n\nAn example is `Z08-03201333302`.\n\nNote that the zone number is zero padded to two digits. Quadkeys can start with the digit 0 so they should always be represented as strings. The length of a quadkey is equal to the cell\'s zoom level.\n\nA specific cell can be initialized using either a Cell ID, or a combination of quadkey and zone:\n\n```pycon\n>>> from maxar_ard_grid import Cell\n>>> c = Cell(\'Z08-03201333302\')\n>>> c = Cell(\'03201333302\', zone=8)\n```\n\nA cell initialized with another cell will return an equivalent cell. This is useful for working with an input that can either be a Cell or a cell ID - you can create a Cell using the input instead of having to check the input type\n\n```pycon\n>>> c = Cell(\'Z08-03201333302\')\n>>> d = Cell(c)\n>>> c == d\nTrue\n```\n\nIdentifying attributes:\n\n```pycon\n>>> c.id\n\'Z08-03201333302\'\n\n>>> c.quadkey\n\'03201333302\'\n\n>>> c.zone\n8\n\n>>> c.zoom\n11\n\n>>> c.hemisphere\n\'N\'\n\n>>> c.proj\n\'EPSG:32608\'\n```\n\nGeospatial attributes:\n\n```pycon\n>>> c.bounds\n[ <xmin, ymin, xmax, ymax in UTM> ]\n\n>>> from shapely.geometry import shape\n>>> shape(c) # Python Geospatial Interface support\n<Shapely.geometry.polygon>   \n\n>>> c.geom_WGS84\n<Shapely.geometry.polygon> #in WGS84\n\n\n>>> c.to_feature()\n{\'type\': \'Feature\', \n    \'properties\': {\n        \'id\': <cell id> \n    }, \n    \'geometry\': <geometry in WGS84>\n}\n\n>>> c.to_GeoJSON()\n\'{"type": "Feature", \n    "properties": {\n        "id": <cell id> \n    }, \n    "geometry": <geometry in WGS84>\n}\'\n\n```\n\nFinding related cells in the quadkey structure:\n\n```pycon\n>>> c.parent\n<Cell Z08-0320133330>\n\n>>> c.get_parent_at_zoom(9)\n<Cell Z08-032013333>\n\n>>> c.children\n[<Cell Z08-03201333300>, <Cell Z08-03201333301>, <Cell Z08-03201333302>, <Cell Z08-03201333303>]\n\n>>> c.get_children_at_zoom(18)\n< generator of zoom 18 Cells >\n\n>>> c.neighbor(\'E\') # cell to the east\n<Cell Z08-03201333303>\n\n>>> c.neighbors\n{\'N\': <Cell Z08-03201333300>, \'NE\': <Cell Z08-03201333301>, \'E\': <Cell Z08-03201333303>, \'SE\': <Cell Z08-03201333321>, \'S\': <Cell Z08-03201333320>, \'SW\': <Cell Z08-03201333231>, \'W\': <Cell Z08-03201333213>, \'NW\': <Cell Z08-03201333211>}\n\n```\n\nOverloaded operators include `==` and `in`. Cell objects are equivalent if they represent the same cell on earth. `in` overloads `__contains__` to check if a cell is a descendent of a given cell, which is also equivalent to being spatially contained.\n\n```pycon\n>>> Cell(\'Z08-03201333302\') == Cell(\'03201333302\', zone=8) \nTrue\n\n>>> Cell(\'Z08-03201333302\') in Cell(\'Z08-032013\') \nTrue\n```\n',
    'author': 'Marc Pfister',
    'author_email': 'marc.pfister@maxar.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
