"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Topic = void 0;
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
const topic_base_1 = require("./topic-base");
/**
 * A new SNS topic.
 *
 * @stability stable
 */
class Topic extends topic_base_1.TopicBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.topicName,
        });
        /**
         * Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        const resource = new sns_generated_1.CfnTopic(this, 'Resource', {
            displayName: props.displayName,
            topicName: this.physicalName,
            kmsMasterKeyId: props.masterKey && props.masterKey.keyArn,
        });
        this.topicArn = this.getResourceArnAttribute(resource.ref, {
            service: 'sns',
            resource: this.physicalName,
        });
        this.topicName = this.getResourceNameAttribute(resource.attrTopicName);
    }
    /**
     * Import an existing SNS topic provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param topicArn topic ARN (i.e. arn:aws:sns:us-east-2:444455556666:MyTopic).
     * @stability stable
     */
    static fromTopicArn(scope, id, topicArn) {
        class Import extends topic_base_1.TopicBase {
            constructor() {
                super(...arguments);
                this.topicArn = topicArn;
                this.topicName = core_1.Stack.of(scope).parseArn(topicArn).resource;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.Topic = Topic;
//# sourceMappingURL=data:application/json;base64,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