## Preliminaries

These instructions detail the Python code you need for running an AEA outside the `cli` tool, using the code interface. 

  
This guide assumes you have already followed the Preliminaries and Installation section in the [quick start](quickstart.md) guide and so have the framework installed and the packages and scripts directory downloaded into the directory you are working in.


## Imports

First, import the necessary common Python libraries and classes.

``` python
import os
import time
from threading import Thread
```

Then, import the application specific libraries.

``` python
from aea.aea_builder import AEABuilder
from aea.crypto.fetchai import FETCHAI
from aea.crypto.helpers import FETCHAI_PRIVATE_KEY_FILE, _create_fetchai_private_key
```

Set up a variable pointing to where the packages directory is located - this should be our current directory - and where the input and output files are located.
``` python
ROOT_DIR = "./"
INPUT_FILE = "input_file"
OUTPUT_FILE = "output_file"
```

## Create a private key
We need a private key to populate the AEA's wallet.
``` python
    # Create a private key
    _create_fetchai_private_key()
```

## Clearing the input and output files
We will use the stub connection to pass envelopes in and out of the AEA. Ensure that any input and output text files are removed before we start.
``` python
    # Ensure the input and output files do not exist initially
    if os.path.isfile(INPUT_FILE):
        os.remove(INPUT_FILE)
    if os.path.isfile(OUTPUT_FILE):
        os.remove(OUTPUT_FILE)
```

## Initialise the AEA
We use the AEABuilder to readily build an AEA. By default, the AEABuilder adds the `fetchai/default:0.1.0` protocol, the `fetchai/stub:0.1.0` connection and the `fetchai/error:0.1.0` skill.
``` python
    # Instantiate the builder and build the AEA
    # By default, the default protocol, error skill and stub connection are added
    builder = AEABuilder()
```

We set the name, add the private key for the AEA to use and set the ledger configurations for the AEA to use.
``` python
    builder.set_name("my_aea")

    builder.add_private_key(FETCHAI, FETCHAI_PRIVATE_KEY_FILE)

    builder.add_ledger_api_config(FETCHAI, {"network": "testnet"})
```

Next, we add the echo skill which will bounce our messages back to us. We first need to place the echo skill into a relevant directory (see path), either by downloading the `packages` directory from the AEA repo or by getting the package from the registry.
``` python
    # Add the echo skill (assuming it is present in the local directory 'packages')
    builder.add_skill("./packages/fetchai/skills/echo")
```

Finally, we can build our AEA:
``` python
    # Create our AEA
    my_aea = builder.build()
```

## Start the AEA
We run the AEA from a different thread so that we can still use the main thread to pass it messages.
``` python
    # Set the AEA running in a different thread
    try:
        t = Thread(target=my_aea.start)
        t.start()

        # Wait for everything to start up
        time.sleep(4)
```

## Send and receive an envelope
We use the input and output text files to send an envelope to our AEA and receive a response (from the echo skill)
``` python
        # Create a message inside an envelope and get the stub connection to pass it on to the echo skill
        message_text = (
            "my_aea,other_agent,fetchai/default:0.1.0,\x08\x01*\x07\n\x05hello,"
        )
        with open(INPUT_FILE, "w") as f:
            f.write(message_text)
            print("input message: " + message_text)

        # Wait for the envelope to get processed
        time.sleep(4)

        # Read the output envelope generated by the echo skill
        with open(OUTPUT_FILE, "r") as f:
            print("output message: " + f.readline())
```

## Shutdown
Finally stop our AEA and wait for it to finish
``` python
    finally:
        # Shut down the AEA
        my_aea.stop()
        t.join()
        t = None
```

## Running the AEA
If you now run this python script file, you should see this output:

    input message: my_aea,other_agent,fetchai/default:0.1.0,\x08\x01*\x07\n\x05hello
    output message: other_agent,my_aea,fetchai/default:0.1.0,\x08\x01*\x07\n\x05hello


## Entire code listing
If you just want to copy and past the entire script in you can find it here:

<details><summary>Click here to see full listing</summary>
<p>

```python
import os
import time
from threading import Thread

from aea.aea_builder import AEABuilder
from aea.crypto.fetchai import FETCHAI
from aea.crypto.helpers import FETCHAI_PRIVATE_KEY_FILE, _create_fetchai_private_key

ROOT_DIR = "./"
INPUT_FILE = "input_file"
OUTPUT_FILE = "output_file"


def run():
    # Create a private key
    _create_fetchai_private_key()

    # Ensure the input and output files do not exist initially
    if os.path.isfile(INPUT_FILE):
        os.remove(INPUT_FILE)
    if os.path.isfile(OUTPUT_FILE):
        os.remove(OUTPUT_FILE)

    # Instantiate the builder and build the AEA
    # By default, the default protocol, error skill and stub connection are added
    builder = AEABuilder()

    builder.set_name("my_aea")

    builder.add_private_key(FETCHAI, FETCHAI_PRIVATE_KEY_FILE)

    builder.add_ledger_api_config(FETCHAI, {"network": "testnet"})

    # Add the echo skill (assuming it is present in the local directory 'packages')
    builder.add_skill("./packages/fetchai/skills/echo")

    # Create our AEA
    my_aea = builder.build()

    # Set the AEA running in a different thread
    try:
        t = Thread(target=my_aea.start)
        t.start()

        # Wait for everything to start up
        time.sleep(4)

        # Create a message inside an envelope and get the stub connection to pass it on to the echo skill
        message_text = (
            "my_aea,other_agent,fetchai/default:0.1.0,\x08\x01*\x07\n\x05hello,"
        )
        with open(INPUT_FILE, "w") as f:
            f.write(message_text)
            print("input message: " + message_text)

        # Wait for the envelope to get processed
        time.sleep(4)

        # Read the output envelope generated by the echo skill
        with open(OUTPUT_FILE, "r") as f:
            print("output message: " + f.readline())
    finally:
        # Shut down the AEA
        my_aea.stop()
        t.join()
        t = None


if __name__ == "__main__":
    run()
```
</p>
</details>

<br />
