"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = {};
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = type;
        }
    };
    // read all .jsii manifests from all modules (incl. projen itself) and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        const p = toProjectType(jsii, fqn);
        if (!p) {
            continue;
        }
        result.push(p);
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function resolveProjectType(projectFqn) {
    const manifest = readJsiiManifest(projectFqn);
    const jsii = {};
    for (const [fqn, type] of Object.entries(manifest.types)) {
        jsii[fqn] = type;
    }
    // Read Projen JSII types
    const projenManifest = readJsiiManifest('projen');
    for (const [fqn, type] of Object.entries(projenManifest.types)) {
        jsii[fqn] = type;
    }
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        return undefined;
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/master/API.md#projen-${typename.toLocaleLowerCase()}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : decamelize(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let typeName;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.primitive) {
                typeName = (_c = prop.type) === null || _c === void 0 ? void 0 : _c.primitive; // e.g. 'string', 'boolean', 'number'
            }
            else if ((_d = prop.type) === null || _d === void 0 ? void 0 : _d.fqn) {
                typeName = (_e = prop.type) === null || _e === void 0 ? void 0 : _e.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
            }
            else { // any other types such as collection types
                typeName = 'unknown';
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_f = prop.docs) === null || _f === void 0 ? void 0 : _f.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_g = prop.type) === null || _g === void 0 ? void 0 : _g.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${typeName}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_h = prop.type) === null || _h === void 0 ? void 0 : _h.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                docs: prop.docs.summary,
                type: typeName,
                switch: propPath.map(p => decamelize(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_j = struct.interfaces) !== null && _j !== void 0 ? _j : []) {
            addOptions(ifc);
        }
    }
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52ZW50b3J5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2ludmVudG9yeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsK0JBQStCO0FBRS9CLGlFQUFpRTtBQUNqRSxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7QUFDekMsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN0RCxNQUFNLGdCQUFnQixHQUFHLGdCQUFnQixDQUFDO0FBK0QxQzs7Ozs7O0dBTUc7QUFDSCxTQUFnQixRQUFRLENBQUMsR0FBRyxVQUFvQjtJQUM5QyxNQUFNLElBQUksR0FBYyxFQUFFLENBQUM7SUFFM0IsTUFBTSxZQUFZLEdBQUcsQ0FBQyxHQUFXLEVBQUUsRUFBRTtRQUNuQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUFFLE9BQU87U0FBRSxDQUFDLG1CQUFtQjtRQUU3RCxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFrQixDQUFDLEVBQUU7WUFDckUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQztTQUNsQjtJQUNILENBQUMsQ0FBQztJQUVGLDRFQUE0RTtJQUM1RSwyQ0FBMkM7SUFDM0MsS0FBSyxNQUFNLEdBQUcsSUFBSSxDQUFDLEdBQUcsVUFBVSxFQUFFLGtCQUFrQixDQUFDLEVBQUU7UUFDckQsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWxCLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ3hELE1BQU0sU0FBUyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztZQUN4RSxLQUFLLE1BQU0sS0FBSyxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ3JCO1NBQ0Y7S0FDRjtJQUVELE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFlLENBQUM7SUFFeEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ25DLE1BQU0sQ0FBQyxHQUFHLGFBQWEsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLENBQUMsRUFBRTtZQUNOLFNBQVM7U0FDVjtRQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDaEI7SUFFRCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztBQUNqRSxDQUFDO0FBdENELDRCQXNDQztBQUVELFNBQWdCLGtCQUFrQixDQUFDLFVBQWtCO0lBQ25ELE1BQU0sUUFBUSxHQUFHLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBRTlDLE1BQU0sSUFBSSxHQUFjLEVBQUUsQ0FBQztJQUMzQixLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBa0IsQ0FBQyxFQUFFO1FBQ3JFLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUM7S0FDbEI7SUFFRCx5QkFBeUI7SUFDekIsTUFBTSxjQUFjLEdBQUcsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDbEQsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEtBQWtCLENBQUMsRUFBRTtRQUMzRSxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDO0tBQ2xCO0lBRUQsT0FBTyxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO0FBQ3pDLENBQUM7QUFmRCxnREFlQztBQUVELFNBQVMsYUFBYSxDQUFDLElBQWUsRUFBRSxHQUFXOztJQUNqRCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRTtRQUM3QixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUVELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzQiw4Q0FBOEM7SUFDOUMsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBRXJELE1BQU0sT0FBTyxHQUFHLDhEQUE4RCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxDQUFDO0lBQzdHLElBQUksSUFBSSxxQkFBRyxRQUFRLENBQUMsSUFBSSwwQ0FBRSxNQUFNLDBDQUFFLElBQUksbUNBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDeEYsT0FBTztRQUNMLFVBQVUsRUFBRSxRQUFRLENBQUMsUUFBUTtRQUM3QixRQUFRO1FBQ1IsSUFBSTtRQUNKLEdBQUc7UUFDSCxPQUFPLEVBQUUsZUFBZSxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEYsSUFBSSxRQUFFLFFBQVEsQ0FBQyxJQUFJLDBDQUFFLE9BQU87UUFDNUIsT0FBTztLQUNPLENBQUM7QUFDbkIsQ0FBQztBQUVELFNBQWdCLGdCQUFnQixDQUFDLE9BQWU7SUFDOUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEMsSUFBSSxVQUFVLEtBQUssUUFBUSxFQUFFO1FBQzNCLFVBQVUsR0FBRyxrQkFBa0IsQ0FBQztLQUNqQztJQUVELE1BQU0sZ0JBQWdCLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLFVBQVUsUUFBUSxDQUFDLENBQUM7SUFDaEUsT0FBTyxFQUFFLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUM7QUFDM0MsQ0FBQztBQVJELDRDQVFDO0FBRUQsU0FBUyxlQUFlLENBQUMsSUFBZSxFQUFFLEdBQVc7O0lBQ25ELE1BQU0sT0FBTyxHQUFzQyxFQUFFLENBQUM7SUFDdEQsTUFBTSxNQUFNLHFCQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQUUsV0FBVywwQ0FBRSxVQUFVLG1DQUFJLEVBQUUsQ0FBQztJQUN4RCxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDL0IsTUFBTSxjQUFjLFNBQUcsWUFBWSxhQUFaLFlBQVksdUJBQVosWUFBWSxDQUFFLElBQUksMENBQUUsR0FBRyxDQUFDO0lBRS9DLElBQUksTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxDQUFBLFlBQVksYUFBWixZQUFZLHVCQUFaLFlBQVksQ0FBRSxJQUFJLE1BQUssU0FBUyxDQUFDLEVBQUU7UUFDbEYsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsR0FBRyxnRUFBZ0UsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7S0FDekk7SUFFRCxVQUFVLENBQUMsY0FBYyxDQUFDLENBQUM7SUFFM0IsTUFBTSxJQUFJLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUVwQyxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUU3RCxTQUFTLFVBQVUsQ0FBQyxJQUFhLEVBQUUsV0FBcUIsRUFBRSxFQUFFLFFBQVEsR0FBRyxLQUFLOztRQUMxRSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ1QsT0FBTztTQUNSO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzFCLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixJQUFJLGdCQUFnQixHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQzNFO1FBRUQsS0FBSyxNQUFNLElBQUksVUFBSSxNQUFNLENBQUMsVUFBVSxtQ0FBSSxFQUFFLEVBQUU7WUFDMUMsTUFBTSxRQUFRLEdBQUcsQ0FBQyxHQUFHLFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFMUMsaUNBQWlDO1lBQ2pDLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxPQUFPLEVBQUU7Z0JBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLElBQUksQ0FBQyxJQUFJLFFBQVEsR0FBRyx5QkFBeUIsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO2FBQ2pIO1lBRUQsSUFBSSxRQUFRLENBQUM7WUFDYixVQUFJLElBQUksQ0FBQyxJQUFJLDBDQUFFLFNBQVMsRUFBRTtnQkFDeEIsUUFBUSxTQUFHLElBQUksQ0FBQyxJQUFJLDBDQUFFLFNBQVMsQ0FBQyxDQUFDLHFDQUFxQzthQUN2RTtpQkFBTSxVQUFJLElBQUksQ0FBQyxJQUFJLDBDQUFFLEdBQUcsRUFBRTtnQkFDekIsUUFBUSxTQUFHLElBQUksQ0FBQyxJQUFJLDBDQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsa0RBQWtEO2FBQy9GO2lCQUFNLEVBQUUsMkNBQTJDO2dCQUNsRCxRQUFRLEdBQUcsU0FBUyxDQUFDO2FBQ3RCO1lBRUQsTUFBTSxVQUFVLEdBQUcsUUFBUSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0MsSUFBSSxZQUFZLFNBQUcsSUFBSSxDQUFDLElBQUksMENBQUUsT0FBTyxDQUFDO1lBRXRDLElBQUksWUFBWSxLQUFLLFdBQVcsRUFBRTtnQkFDaEMsWUFBWSxHQUFHLFNBQVMsQ0FBQzthQUMxQjtZQUVELDRHQUE0RztZQUM1RyxJQUFJLENBQUMsVUFBVSxJQUFJLFlBQVksRUFBRTtnQkFDL0IsSUFBSSxRQUFDLElBQUksQ0FBQyxJQUFJLDBDQUFFLFNBQVMsQ0FBQSxFQUFFO29CQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLG9CQUFvQixJQUFJLENBQUMsSUFBSSwwRkFBMEYsUUFBUSxFQUFFLENBQUMsQ0FBQztpQkFDcEo7Z0JBRUQsc0JBQXNCLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxZQUFZLFFBQUUsSUFBSSxDQUFDLElBQUksMENBQUUsU0FBUyxDQUFDLENBQUM7YUFDdkU7WUFFRCxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLGVBQWUsQ0FBQztnQkFDbkMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJO2dCQUNuQixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztnQkFDdkIsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsTUFBTSxFQUFFLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7Z0JBQ3JFLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixRQUFRLEVBQUUsVUFBVTtnQkFDcEIsVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxLQUFLLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ3BFLENBQUMsQ0FBQztTQUNKO1FBRUQsS0FBSyxNQUFNLEdBQUcsVUFBSSxNQUFNLENBQUMsVUFBVSxtQ0FBSSxFQUFFLEVBQUU7WUFDekMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ2pCO0lBQ0gsQ0FBQztBQUNILENBQUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxHQUFRO0lBQy9CLE1BQU0sR0FBRyxHQUFRLEVBQUUsQ0FBQztJQUNwQixLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUN4QyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDbkIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNaO0tBQ0Y7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxJQUFlLEVBQUUsR0FBVzs7SUFDakQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRXZCLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxPQUFPLEVBQUU7UUFDekIsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUNELElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtRQUNqQixPQUFPLEtBQUssQ0FBQztLQUNkO0lBRUQsVUFBSSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxVQUFVLEVBQUU7UUFDekIsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUVELElBQUksSUFBSSxHQUFHLElBQUksQ0FBQztJQUNoQixPQUFPLElBQUksRUFBRTtRQUNYLElBQUksSUFBSSxDQUFDLEdBQUcsS0FBSyxnQkFBZ0IsRUFBRTtZQUNqQyxPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDZCxPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkIsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNULE9BQU8sS0FBSyxDQUFDO1NBQ2Q7S0FDRjtBQUNILENBQUM7QUFFRCxTQUFTLHNCQUFzQixDQUFDLElBQVksRUFBRSxLQUFhLEVBQUUsSUFBWTtJQUN2RSwwQkFBMEI7SUFDMUIsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3pCLE9BQU87S0FDUjtJQUNELElBQUk7UUFDRixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2pDLElBQUksT0FBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLElBQUksRUFBRTtZQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxJQUFJLFNBQVMsSUFBSSxLQUFLLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDckc7S0FFRjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsS0FBSyxpREFBaUQsSUFBSSxNQUFNLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0tBQy9IO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcy1leHRyYSc7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzXG5jb25zdCBkZWNhbWVsaXplID0gcmVxdWlyZSgnZGVjYW1lbGl6ZScpO1xuY29uc3QgUFJPSkVOX01PRFVMRV9ST09UID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJyk7XG5jb25zdCBQUk9KRUNUX0JBU0VfRlFOID0gJ3Byb2plbi5Qcm9qZWN0JztcblxudHlwZSBKc2lpVHlwZXMgPSB7IFtuYW1lOiBzdHJpbmddOiBKc2lpVHlwZSB9O1xuXG5leHBvcnQgaW50ZXJmYWNlIFByb2plY3RPcHRpb24ge1xuICBwYXRoOiBzdHJpbmdbXTtcbiAgbmFtZTogc3RyaW5nO1xuICBzd2l0Y2g6IHN0cmluZztcbiAgdHlwZTogc3RyaW5nO1xuICBwYXJlbnQ6IHN0cmluZztcbiAgZG9jcz86IHN0cmluZztcbiAgZGVmYXVsdD86IHN0cmluZztcbiAgb3B0aW9uYWw/OiBib29sZWFuO1xuICBkZXByZWNhdGVkPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcm9qZWN0VHlwZSB7XG4gIG1vZHVsZU5hbWU6IHN0cmluZztcbiAgcGppZDogc3RyaW5nO1xuICBmcW46IHN0cmluZztcbiAgdHlwZW5hbWU6IHN0cmluZztcbiAgb3B0aW9uczogUHJvamVjdE9wdGlvbltdO1xuICBkb2NzPzogc3RyaW5nO1xuICBkb2NzdXJsOiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBKc2lpVHlwZSB7XG4gIG5hbWU6IHN0cmluZztcbiAgYXNzZW1ibHk6IHN0cmluZztcbiAga2luZDogc3RyaW5nO1xuICBhYnN0cmFjdD86IGJvb2xlYW47XG4gIGJhc2U/OiBzdHJpbmc7XG4gIGZxbjogc3RyaW5nO1xuICBpbnRlcmZhY2VzPzogc3RyaW5nW107XG4gIGluaXRpYWxpemVyPzoge1xuICAgIHBhcmFtZXRlcnM/OiBBcnJheTx7XG4gICAgICBuYW1lOiBzdHJpbmc7XG4gICAgICB0eXBlPzogeyBmcW4/OiBzdHJpbmcgfTtcbiAgICB9PjtcbiAgfTtcbiAgcHJvcGVydGllcz86IEFycmF5PHtcbiAgICBuYW1lOiBzdHJpbmc7XG4gICAgZG9jczoge1xuICAgICAgc3VtbWFyeT86IHN0cmluZztcbiAgICAgIGRlZmF1bHQ/OiBzdHJpbmc7XG4gICAgICBkZXByZWNhdGVkPzogc3RyaW5nO1xuICAgICAgc3RhYmlsaXR5Pzogc3RyaW5nO1xuICAgIH07XG4gICAgb3B0aW9uYWw/OiBib29sZWFuO1xuICAgIHR5cGU/OiB7XG4gICAgICBwcmltaXRpdmU/OiBzdHJpbmc7XG4gICAgICBmcW4/OiBzdHJpbmc7XG4gICAgfTtcbiAgfT47XG4gIGRvY3M/OiB7XG4gICAgc3VtbWFyeT86IHN0cmluZztcbiAgICBkZXByZWNhdGVkPzogc3RyaW5nO1xuICAgIGN1c3RvbT86IHtcbiAgICAgIHBqaWQ/OiBzdHJpbmc7XG4gICAgfTtcbiAgfTtcbn1cblxuLyoqXG4gKiBSZXR1cm5zIGEgbGlzdCBvZiBwcm9qZWN0IHR5cGVzIGV4cG9ydGVkIHRoZSBtb2R1bGVzIGRlZmluZWQgaW4gYG1vZHVsZURpcnNgLlxuICogVGhpcyBsaXN0IHdpbGwgYWx3YXlzIGFsc28gaW5jbHVkZSB0aGUgYnVpbHQtaW4gcHJvamVuIHByb2plY3QgdHlwZXMuXG4gKiBNb2R1bGVzIHdpdGhvdXQgYSAuanNpaSBtYW5pZmVzdCBhcmUgc2tpcHBlZC5cbiAqXG4gKiBAcGFyYW0gbW9kdWxlRGlycyBBIGxpc3Qgb2YgbnBtIG1vZHVsZSBkaXJlY3Rvcmllc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZGlzY292ZXIoLi4ubW9kdWxlRGlyczogc3RyaW5nW10pIHtcbiAgY29uc3QganNpaTogSnNpaVR5cGVzID0ge307XG5cbiAgY29uc3QgZGlzY292ZXJKc2lpID0gKGRpcjogc3RyaW5nKSA9PiB7XG4gICAgY29uc3QganNpaUZpbGUgPSBwYXRoLmpvaW4oZGlyLCAnLmpzaWknKTtcbiAgICBpZiAoIWZzLmV4aXN0c1N5bmMoanNpaUZpbGUpKSB7IHJldHVybjsgfSAvLyBubyBqc2lpIG1hbmlmZXN0XG5cbiAgICBjb25zdCBtYW5pZmVzdCA9IGZzLnJlYWRKc29uU3luYyhqc2lpRmlsZSk7XG4gICAgZm9yIChjb25zdCBbZnFuLCB0eXBlXSBvZiBPYmplY3QuZW50cmllcyhtYW5pZmVzdC50eXBlcyBhcyBKc2lpVHlwZXMpKSB7XG4gICAgICBqc2lpW2Zxbl0gPSB0eXBlO1xuICAgIH1cbiAgfTtcblxuICAvLyByZWFkIGFsbCAuanNpaSBtYW5pZmVzdHMgZnJvbSBhbGwgbW9kdWxlcyAoaW5jbC4gcHJvamVuIGl0c2VsZikgYW5kIG1lcmdlXG4gIC8vIHRoZW0gYWxsIGludG8gYSBzaW5nbGUgbWFwIG9mIGZxbi0+dHlwZS5cbiAgZm9yIChjb25zdCBkaXIgb2YgWy4uLm1vZHVsZURpcnMsIFBST0pFTl9NT0RVTEVfUk9PVF0pIHtcbiAgICBkaXNjb3ZlckpzaWkoZGlyKTtcblxuICAgIGlmIChkaXIuaW5jbHVkZXMoJ0AnKSAmJiBmcy5sc3RhdFN5bmMoZGlyKS5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICBjb25zdCBjaGlsZERpcnMgPSBmcy5yZWFkZGlyU3luYyhkaXIpLm1hcChmaWxlID0+IHBhdGguam9pbihkaXIsIGZpbGUpKTtcbiAgICAgIGZvciAoY29uc3QgY2hpbGQgb2YgY2hpbGREaXJzKSB7XG4gICAgICAgIGRpc2NvdmVySnNpaShjaGlsZCk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgY29uc3QgcmVzdWx0ID0gbmV3IEFycmF5PFByb2plY3RUeXBlPigpO1xuXG4gIGZvciAoY29uc3QgZnFuIG9mIE9iamVjdC5rZXlzKGpzaWkpKSB7XG4gICAgY29uc3QgcCA9IHRvUHJvamVjdFR5cGUoanNpaSwgZnFuKTtcbiAgICBpZiAoIXApIHtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cblxuICAgIHJlc3VsdC5wdXNoKHApO1xuICB9XG5cbiAgcmV0dXJuIHJlc3VsdC5zb3J0KChyMSwgcjIpID0+IHIxLnBqaWQubG9jYWxlQ29tcGFyZShyMi5wamlkKSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiByZXNvbHZlUHJvamVjdFR5cGUocHJvamVjdEZxbjogc3RyaW5nKSB7XG4gIGNvbnN0IG1hbmlmZXN0ID0gcmVhZEpzaWlNYW5pZmVzdChwcm9qZWN0RnFuKTtcblxuICBjb25zdCBqc2lpOiBKc2lpVHlwZXMgPSB7fTtcbiAgZm9yIChjb25zdCBbZnFuLCB0eXBlXSBvZiBPYmplY3QuZW50cmllcyhtYW5pZmVzdC50eXBlcyBhcyBKc2lpVHlwZXMpKSB7XG4gICAganNpaVtmcW5dID0gdHlwZTtcbiAgfVxuXG4gIC8vIFJlYWQgUHJvamVuIEpTSUkgdHlwZXNcbiAgY29uc3QgcHJvamVuTWFuaWZlc3QgPSByZWFkSnNpaU1hbmlmZXN0KCdwcm9qZW4nKTtcbiAgZm9yIChjb25zdCBbZnFuLCB0eXBlXSBvZiBPYmplY3QuZW50cmllcyhwcm9qZW5NYW5pZmVzdC50eXBlcyBhcyBKc2lpVHlwZXMpKSB7XG4gICAganNpaVtmcW5dID0gdHlwZTtcbiAgfVxuXG4gIHJldHVybiB0b1Byb2plY3RUeXBlKGpzaWksIHByb2plY3RGcW4pO1xufVxuXG5mdW5jdGlvbiB0b1Byb2plY3RUeXBlKGpzaWk6IEpzaWlUeXBlcywgZnFuOiBzdHJpbmcpIHtcbiAgaWYgKCFpc1Byb2plY3RUeXBlKGpzaWksIGZxbikpIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgY29uc3QgdHlwZWluZm8gPSBqc2lpW2Zxbl07XG5cbiAgLy8gcHJvamVuLndlYi5SZWFjdFByb2plY3QgLT4gd2ViLlJlYWN0UHJvamVjdFxuICBjb25zdCB0eXBlbmFtZSA9IGZxbi5zdWJzdHJpbmcoZnFuLmluZGV4T2YoJy4nKSArIDEpO1xuXG4gIGNvbnN0IGRvY3N1cmwgPSBgaHR0cHM6Ly9naXRodWIuY29tL3Byb2plbi9wcm9qZW4vYmxvYi9tYXN0ZXIvQVBJLm1kI3Byb2plbi0ke3R5cGVuYW1lLnRvTG9jYWxlTG93ZXJDYXNlKCl9YDtcbiAgbGV0IHBqaWQgPSB0eXBlaW5mby5kb2NzPy5jdXN0b20/LnBqaWQgPz8gZGVjYW1lbGl6ZSh0eXBlbmFtZSkucmVwbGFjZSgvX3Byb2plY3QkLywgJycpO1xuICByZXR1cm4ge1xuICAgIG1vZHVsZU5hbWU6IHR5cGVpbmZvLmFzc2VtYmx5LFxuICAgIHR5cGVuYW1lLFxuICAgIHBqaWQsXG4gICAgZnFuLFxuICAgIG9wdGlvbnM6IGRpc2NvdmVyT3B0aW9ucyhqc2lpLCBmcW4pLnNvcnQoKG8xLCBvMikgPT4gbzEubmFtZS5sb2NhbGVDb21wYXJlKG8yLm5hbWUpKSxcbiAgICBkb2NzOiB0eXBlaW5mby5kb2NzPy5zdW1tYXJ5LFxuICAgIGRvY3N1cmwsXG4gIH0gYXMgUHJvamVjdFR5cGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiByZWFkSnNpaU1hbmlmZXN0KGpzaWlGcW46IHN0cmluZyk6IGFueSB7XG4gIGxldCBbbW9kdWxlTmFtZV0gPSBqc2lpRnFuLnNwbGl0KCcuJyk7XG4gIGlmIChtb2R1bGVOYW1lID09PSAncHJvamVuJykge1xuICAgIG1vZHVsZU5hbWUgPSBQUk9KRU5fTU9EVUxFX1JPT1Q7XG4gIH1cblxuICBjb25zdCBqc2lpTWFuaWZlc3RGaWxlID0gcmVxdWlyZS5yZXNvbHZlKGAke21vZHVsZU5hbWV9Ly5qc2lpYCk7XG4gIHJldHVybiBmcy5yZWFkSnNvblN5bmMoanNpaU1hbmlmZXN0RmlsZSk7XG59XG5cbmZ1bmN0aW9uIGRpc2NvdmVyT3B0aW9ucyhqc2lpOiBKc2lpVHlwZXMsIGZxbjogc3RyaW5nKTogUHJvamVjdE9wdGlvbltdIHtcbiAgY29uc3Qgb3B0aW9uczogeyBbbmFtZTogc3RyaW5nXTogUHJvamVjdE9wdGlvbiB9ID0ge307XG4gIGNvbnN0IHBhcmFtcyA9IGpzaWlbZnFuXT8uaW5pdGlhbGl6ZXI/LnBhcmFtZXRlcnMgPz8gW107XG4gIGNvbnN0IG9wdGlvbnNQYXJhbSA9IHBhcmFtc1swXTtcbiAgY29uc3Qgb3B0aW9uc1R5cGVGcW4gPSBvcHRpb25zUGFyYW0/LnR5cGU/LmZxbjtcblxuICBpZiAocGFyYW1zLmxlbmd0aCA+IDEgfHwgKHBhcmFtcy5sZW5ndGggPT09IDEgJiYgb3B0aW9uc1BhcmFtPy5uYW1lICE9PSAnb3B0aW9ucycpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBjb25zdHJ1Y3RvciBmb3IgcHJvamVjdCAke2Zxbn0gbXVzdCBoYXZlIGEgc2luZ2xlIFwib3B0aW9uc1wiIGFyZ3VtZW50IG9mIGEgc3RydWN0IHR5cGUuIGdvdCAke0pTT04uc3RyaW5naWZ5KHBhcmFtcyl9YCk7XG4gIH1cblxuICBhZGRPcHRpb25zKG9wdGlvbnNUeXBlRnFuKTtcblxuICBjb25zdCBvcHRzID0gT2JqZWN0LnZhbHVlcyhvcHRpb25zKTtcblxuICByZXR1cm4gb3B0cy5zb3J0KChhLCBiKSA9PiBhLnN3aXRjaC5sb2NhbGVDb21wYXJlKGIuc3dpdGNoKSk7XG5cbiAgZnVuY3Rpb24gYWRkT3B0aW9ucyhvZnFuPzogc3RyaW5nLCBiYXNlUGF0aDogc3RyaW5nW10gPSBbXSwgb3B0aW9uYWwgPSBmYWxzZSkge1xuICAgIGlmICghb2Zxbikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IHN0cnVjdCA9IGpzaWlbb2Zxbl07XG4gICAgaWYgKCFzdHJ1Y3QpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIGZpbmQgb3B0aW9ucyB0eXBlICR7b2Zxbn0gZm9yIHByb2plY3QgJHtmcW59YCk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBwcm9wIG9mIHN0cnVjdC5wcm9wZXJ0aWVzID8/IFtdKSB7XG4gICAgICBjb25zdCBwcm9wUGF0aCA9IFsuLi5iYXNlUGF0aCwgcHJvcC5uYW1lXTtcblxuICAgICAgLy8gcHJvdGVjdCBhZ2FpbnN0IGRvdWJsZS1ib29raW5nXG4gICAgICBpZiAocHJvcC5uYW1lIGluIG9wdGlvbnMpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBkdXBsaWNhdGUgb3B0aW9uIFwiJHtwcm9wLm5hbWV9XCIgaW4gJHtmcW59IChhbHJlYWR5IGRlY2xhcmVkIGluICR7b3B0aW9uc1twcm9wLm5hbWVdLnBhcmVudH0pYCk7XG4gICAgICB9XG5cbiAgICAgIGxldCB0eXBlTmFtZTtcbiAgICAgIGlmIChwcm9wLnR5cGU/LnByaW1pdGl2ZSkge1xuICAgICAgICB0eXBlTmFtZSA9IHByb3AudHlwZT8ucHJpbWl0aXZlOyAvLyBlLmcuICdzdHJpbmcnLCAnYm9vbGVhbicsICdudW1iZXInXG4gICAgICB9IGVsc2UgaWYgKHByb3AudHlwZT8uZnFuKSB7XG4gICAgICAgIHR5cGVOYW1lID0gcHJvcC50eXBlPy5mcW4uc3BsaXQoJy4nKS5wb3AoKTsgLy8gcHJvamVuLk5vZGVQcm9qZWN0T3B0aW9ucyAtPiBOb2RlUHJvamVjdE9wdGlvbnNcbiAgICAgIH0gZWxzZSB7IC8vIGFueSBvdGhlciB0eXBlcyBzdWNoIGFzIGNvbGxlY3Rpb24gdHlwZXNcbiAgICAgICAgdHlwZU5hbWUgPSAndW5rbm93bic7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGlzT3B0aW9uYWwgPSBvcHRpb25hbCB8fCBwcm9wLm9wdGlvbmFsO1xuICAgICAgbGV0IGRlZmF1bHRWYWx1ZSA9IHByb3AuZG9jcz8uZGVmYXVsdDtcblxuICAgICAgaWYgKGRlZmF1bHRWYWx1ZSA9PT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgZGVmYXVsdFZhbHVlID0gdW5kZWZpbmVkO1xuICAgICAgfVxuXG4gICAgICAvLyBpZiB0aGlzIGlzIGEgbWFuZGF0b3J5IG9wdGlvbiBhbmQgd2UgaGF2ZSBhIGRlZmF1bHQgdmFsdWUsIGl0IGhhcyB0byBiZSBKU09OLXBhcnNhYmxlIHRvIHRoZSBjb3JyZWN0IHR5cGVcbiAgICAgIGlmICghaXNPcHRpb25hbCAmJiBkZWZhdWx0VmFsdWUpIHtcbiAgICAgICAgaWYgKCFwcm9wLnR5cGU/LnByaW1pdGl2ZSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgcmVxdWlyZWQgb3B0aW9uIFwiJHtwcm9wLm5hbWV9XCIgd2l0aCBhIEBkZWZhdWx0IG11c3QgdXNlIHByaW1pdGl2ZSB0eXBlcyAoc3RyaW5nLCBudW1iZXIgb3IgYm9vbGVhbikuIHR5cGUgZm91bmQgaXM6ICR7dHlwZU5hbWV9YCk7XG4gICAgICAgIH1cblxuICAgICAgICBjaGVja0RlZmF1bHRJc1BhcnNhYmxlKHByb3AubmFtZSwgZGVmYXVsdFZhbHVlLCBwcm9wLnR5cGU/LnByaW1pdGl2ZSk7XG4gICAgICB9XG5cbiAgICAgIG9wdGlvbnNbcHJvcC5uYW1lXSA9IGZpbHRlclVuZGVmaW5lZCh7XG4gICAgICAgIHBhdGg6IHByb3BQYXRoLFxuICAgICAgICBwYXJlbnQ6IHN0cnVjdC5uYW1lLFxuICAgICAgICBuYW1lOiBwcm9wLm5hbWUsXG4gICAgICAgIGRvY3M6IHByb3AuZG9jcy5zdW1tYXJ5LFxuICAgICAgICB0eXBlOiB0eXBlTmFtZSxcbiAgICAgICAgc3dpdGNoOiBwcm9wUGF0aC5tYXAocCA9PiBkZWNhbWVsaXplKHApLnJlcGxhY2UoL18vZywgJy0nKSkuam9pbignLScpLFxuICAgICAgICBkZWZhdWx0OiBkZWZhdWx0VmFsdWUsXG4gICAgICAgIG9wdGlvbmFsOiBpc09wdGlvbmFsLFxuICAgICAgICBkZXByZWNhdGVkOiBwcm9wLmRvY3Muc3RhYmlsaXR5ID09PSAnZGVwcmVjYXRlZCcgPyB0cnVlIDogdW5kZWZpbmVkLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBpZmMgb2Ygc3RydWN0LmludGVyZmFjZXMgPz8gW10pIHtcbiAgICAgIGFkZE9wdGlvbnMoaWZjKTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gZmlsdGVyVW5kZWZpbmVkKG9iajogYW55KSB7XG4gIGNvbnN0IHJldDogYW55ID0ge307XG4gIGZvciAoY29uc3QgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKG9iaikpIHtcbiAgICBpZiAodiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXRba10gPSB2O1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmV0O1xufVxuXG5mdW5jdGlvbiBpc1Byb2plY3RUeXBlKGpzaWk6IEpzaWlUeXBlcywgZnFuOiBzdHJpbmcpIHtcbiAgY29uc3QgdHlwZSA9IGpzaWlbZnFuXTtcblxuICBpZiAodHlwZS5raW5kICE9PSAnY2xhc3MnKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG4gIGlmICh0eXBlLmFic3RyYWN0KSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgaWYgKHR5cGUuZG9jcz8uZGVwcmVjYXRlZCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGxldCBjdXJyID0gdHlwZTtcbiAgd2hpbGUgKHRydWUpIHtcbiAgICBpZiAoY3Vyci5mcW4gPT09IFBST0pFQ1RfQkFTRV9GUU4pIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIGlmICghY3Vyci5iYXNlKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgY3VyciA9IGpzaWlbY3Vyci5iYXNlXTtcbiAgICBpZiAoIWN1cnIpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gIH1cbn1cblxuZnVuY3Rpb24gY2hlY2tEZWZhdWx0SXNQYXJzYWJsZShwcm9wOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcsIHR5cGU6IHN0cmluZykge1xuICAvLyBtYWNyb3MgYXJlIHBhc3MtdGhyb3VnaFxuICBpZiAodmFsdWUuc3RhcnRzV2l0aCgnJCcpKSB7XG4gICAgcmV0dXJuO1xuICB9XG4gIHRyeSB7XG4gICAgY29uc3QgcGFyc2VkID0gSlNPTi5wYXJzZSh2YWx1ZSk7XG4gICAgaWYgKHR5cGVvZihwYXJzZWQpICE9PSB0eXBlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGNhbm5vdCBwYXJzZSBAZGVmYXVsdCB2YWx1ZSBmb3IgbWFuZGF0b3J5IG9wdGlvbiAke3Byb3B9IGFzIGEgJHt0eXBlfTogJHtwYXJzZWR9YCk7XG4gICAgfVxuXG4gIH0gY2F0Y2ggKGUpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYHVuYWJsZSB0byBKU09OLnBhcnNlKCkgdmFsdWUgXCIke3ZhbHVlfVwiIHNwZWNpZmllZCBhcyBAZGVmYXVsdCBmb3IgbWFuZGF0b3J5IG9wdGlvbiBcIiR7cHJvcH1cIjogJHtlLm1lc3NhZ2V9YCk7XG4gIH1cbn0iXX0=