"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.arnForParameterName = exports.AUTOGEN_MARKER = void 0;
const core_1 = require("@aws-cdk/core");
exports.AUTOGEN_MARKER = '$$autogen$$';
/**
 * Renders an ARN for an SSM parameter given a parameter name.
 * @param scope definition scope
 * @param parameterName the parameter name to include in the ARN
 * @param physicalName optional physical name specified by the user (to auto-detect separator)
 */
function arnForParameterName(scope, parameterName, options = {}) {
    const physicalName = options.physicalName;
    const nameToValidate = physicalName || parameterName;
    if (!core_1.Token.isUnresolved(nameToValidate) && nameToValidate.includes('/') && !nameToValidate.startsWith('/')) {
        throw new Error(`Parameter names must be fully qualified (if they include "/" they must also begin with a "/"): ${nameToValidate}`);
    }
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: isSimpleName() ? '/' : '',
        resourceName: parameterName,
    });
    /**
     * Determines the ARN separator for this parameter: if we have a concrete
     * parameter name (or explicitly defined physical name), we will parse them
     * and decide whether a "/" is needed or not. Otherwise, users will have to
     * explicitly specify `simpleName` when they import the ARN.
     */
    function isSimpleName() {
        // look for a concrete name as a hint for determining the separator
        const concreteName = !core_1.Token.isUnresolved(parameterName) ? parameterName : physicalName;
        if (!concreteName || core_1.Token.isUnresolved(concreteName)) {
            if (options.simpleName === undefined) {
                throw new Error('Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly');
            }
            return options.simpleName;
        }
        const result = !concreteName.startsWith('/');
        // if users explicitly specify the separator and it conflicts with the one we need, it's an error.
        if (options.simpleName !== undefined && options.simpleName !== result) {
            if (concreteName === exports.AUTOGEN_MARKER) {
                throw new Error('If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names');
            }
            throw new Error(`Parameter name "${concreteName}" is ${result ? 'a simple name' : 'not a simple name'}, but "simpleName" was explicitly set to ${options.simpleName}. Either omit it or set it to ${result}`);
        }
        return result;
    }
}
exports.arnForParameterName = arnForParameterName;
//# sourceMappingURL=data:application/json;base64,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