"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.allRouteTableIds = exports.range = exports.ImportSubnetGroup = exports.subnetId = exports.subnetGroupNameFromConstructId = exports.defaultSubnetName = exports.slugify = void 0;
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // eslint-disable-next-line max-len
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            /* eslint-disable max-len */
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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