"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id.
     * @param os The operating system type of the AMI.
     * @param userData optional user data for the given image.
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * Look up a shared Machine Image using DescribeImages.
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the image based on a given SSM parameter.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSSMParameterImage {
    /**
     *
     */
    constructor(parameterName, os, userData) {
        this.parameterName = parameterName;
        this.os = os;
        this.userData = userData;
    }
    /**
     * Return the image to use in the given context.
     */
    getImage(scope) {
        var _a;
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: (_a = this.userData) !== null && _a !== void 0 ? _a : (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
/**
 * Select the latest version of the indicated Windows version.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage extends GenericSSMParameterImage {
    /**
     *
     */
    constructor(version, props = {}) {
        super('/aws/service/ami-windows-latest/' + version, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
/**
 * CPU type.
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    /**
     *
     */
    constructor(props = {}) {
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        const storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        super(parameterName, OperatingSystemType.LINUX, props.userData);
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use.
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition.
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    AmazonLinuxEdition["STANDARD"] = "standard";
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux.
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    AmazonLinuxVirt["HVM"] = "hvm";
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
/**
 *
 */
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map.
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    /**
     *
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     */
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map.
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    /**
     *
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     */
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image.
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    /**
     *
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     */
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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