"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const path = require("path");
const cfn = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 */
class DeployCdkStackAction {
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const actionRole = roleFromPlaceholderArn(scope, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, artifact.cloudFormationExecutionRoleArn);
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        const templatePath = path.relative(appAsmRoot, fullTemplatePath);
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    scope = hackyRoleScope(scope, arn);
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = scope.node.tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = scope.node.tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    return iam.Role.fromRoleArn(scope, id, cfnExpressionFromManifestString(arn), { mutable: false });
}
/**
 * MASSIVE HACK
 *
 * We have a bug in the CDK where it's only going to consider Roles that are physically in a
 * different Stack object from the Pipeline "cross-account", and will add the appropriate
 * Bucket/Key policies.
 * https://github.com/aws/aws-cdk/pull/8280 will resolve this, but for now we fake it by hacking
 * up a Stack object to root the role in!
 *
 * Fortunatey, we can just 'new up' an unrooted Stack (unit tests do this all the time) and toss it
 * away. It will never be synthesized, but all the logic happens to work out!
 */
function hackyRoleScope(scope, arn) {
    const parts = core_1.Arn.parse(cxapi.EnvironmentPlaceholders.replace(arn, {
        accountId: '',
        partition: '',
        region: '',
    }));
    return new core_1.Stack(undefined, undefined, {
        env: {
            // Empty string means ARN had a placeholder which means same account as pipeline stack
            account: parts.account || core_1.Stack.of(scope).account,
        },
    });
}
/**
 * Return a CloudFormation expression from a manifest string with placeholders
 */
function cfnExpressionFromManifestString(s) {
    // This implementation relies on the fact that the manifest placeholders are
    // '${AWS::Partition}' etc., and so are the same values as those that are
    // trivially substituable using a `Fn.sub`.
    return core_1.Fn.sub(s);
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
//# sourceMappingURL=data:application/json;base64,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