"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rootPathTo = exports.Stack = void 0;
const fs = require("fs");
const path = require("path");
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const annotations_1 = require("./annotations");
const app_1 = require("./app");
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const context_provider_1 = require("./context-provider");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const MY_STACK_CACHE = Symbol.for('@aws-cdk/core.Stack.myStack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_compat_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually an `App` or a `Stage`, but could be any construct.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope, id, props = {}) {
        var _a;
        // For unit test scope and id are optional for stacks, but we still want an App
        // as the parent because apps implement much of the synthesis logic.
        scope = scope !== null && scope !== void 0 ? scope : new app_1.App({
            autoSynth: false,
            outdir: fs_1.FileSystem.mkdtemp('cdk-test-app-'),
        });
        // "Default" is a "hidden id" from a `node.uniqueId` perspective
        id = id !== null && id !== void 0 ? id : 'Default';
        super(scope, id);
        this._missingContext = new Array();
        this._stackDependencies = {};
        this.templateOptions = {};
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        this.terminationProtection = props.terminationProtection;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.generateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        //
        // Also use the new behavior if we are using the new CI/CD-ready synthesizer; that way
        // people only have to flip one flag.
        // eslint-disable-next-line max-len
        this.artifactId = this.node.tryGetContext(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT) || this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? this.generateStackArtifactId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        this.synthesizer = (_a = props.synthesizer) !== null && _a !== void 0 ? _a : (this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? new stack_synthesizers_1.DefaultStackSynthesizer()
            : new stack_synthesizers_1.LegacyStackSynthesizer());
        this.synthesizer.bind(this);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        // we want this to be as cheap as possible. cache this result by mutating
        // the object. anecdotally, at the time of this writing, @aws-cdk/core unit
        // tests hit this cache 1,112 times, @aws-cdk/aws-cloudformation unit tests
        // hit this 2,435 times).
        const cache = construct[MY_STACK_CACHE];
        if (cache) {
            return cache;
        }
        else {
            const value = _lookup(construct);
            Object.defineProperty(construct, MY_STACK_CACHE, {
                enumerable: false,
                writable: false,
                configurable: false,
                value,
            });
            return value;
        }
        function _lookup(c) {
            if (Stack.isStack(c)) {
                return c;
            }
            if (!c.node.scope) {
                throw new Error(`No stack could be identified for the construct at path ${construct.node.path}`);
            }
            return _lookup(c.node.scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        if (!Object.values(cxschema.ContextProvider).includes(report.provider)) {
            throw new Error(`Unknown context provider requested in: ${JSON.stringify(report)}`);
        }
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `allocateLogicalId` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     */
    addDependency(target, reason) {
        deps_1.addDependency(this, target, reason);
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Object.values(this._stackDependencies).map(x => x.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.nestedStackResource !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returns the list of AZs that are available in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     *
     * To specify a different strategy for selecting availability zones override this method.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs()),
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    /**
     * Register a file asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addFileAsset()` if you are calling,
     * and a different IDeploymentEnvironment class if you are implementing.
     */
    addFileAsset(asset) {
        return this.synthesizer.addFileAsset(asset);
    }
    /**
     * Register a docker image asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addDockerImageAsset()` if you are calling,
     * and a different `IDeploymentEnvironment` class if you are implementing.
     */
    addDockerImageAsset(asset) {
        return this.synthesizer.addDockerImageAsset(asset);
    }
    /**
     * If this is a nested stack, returns it's parent stack.
     */
    get nestedStackParent() {
        return this.nestedStackResource && Stack.of(this.nestedStackResource);
    }
    /**
     * Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack() {
        return this.nestedStackParent;
    }
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS
     * CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @example addTransform('AWS::Serverless-2016-10-31')
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     *
     * @param transform The transform to add
     */
    addTransform(transform) {
        if (!this.templateOptions.transforms) {
            this.templateOptions.transforms = [];
        }
        this.templateOptions.transforms.push(transform);
    }
    /**
     * Called implicitly by the `addDependency` helper function in order to
     * realize a dependency between two top-level stacks at the assembly level.
     *
     * Use `stack.addDependency` to define the dependency between any two stacks,
     * and take into account nested stack relationships.
     *
     * @internal
     */
    _addAssemblyDependency(target, reason) {
        // defensive: we should never get here for nested stacks
        if (this.nested || target.nested) {
            throw new Error('Cannot add assembly-level dependencies for nested stacks');
        }
        reason = reason || 'dependency added using stack.addDependency()';
        const cycle = target.stackDependencyReasons(this);
        if (cycle !== undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`'${target.node.path}' depends on '${this.node.path}' (${cycle.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        let dep = this._stackDependencies[target.node.uniqueId];
        if (!dep) {
            dep = this._stackDependencies[target.node.uniqueId] = {
                stack: target,
                reasons: [],
            };
        }
        dep.reasons.push(reason);
        if (process.env.CDK_DEBUG_DEPS) {
            // eslint-disable-next-line no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${target.node.path}" because: ${reason}`);
        }
    }
    /**
     * Synthesizes the cloudformation template into a cloud assembly.
     * @internal
     */
    _synthesizeTemplate(session) {
        // In principle, stack synthesis is delegated to the
        // StackSynthesis object.
        //
        // However, some parts of synthesis currently use some private
        // methods on Stack, and I don't really see the value in refactoring
        // this right now, so some parts still happen here.
        const builder = session.assembly;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        const text = JSON.stringify(this._toCloudFormation(), undefined, 2);
        fs.writeFileSync(outPath, text);
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
        // Delegate adding artifacts to the Synthesizer
        this.synthesizer.synthesizeStackArtifacts(session);
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        let transform;
        if (this.templateOptions.transform) {
            // eslint-disable-next-line max-len
            annotations_1.Annotations.of(this).addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `addTransform()`.');
            this.addTransform(this.templateOptions.transform);
        }
        if (this.templateOptions.transforms) {
            if (this.templateOptions.transforms.length === 1) { // Extract single value
                transform = this.templateOptions.transforms[0];
            }
            else { // Remove duplicate values
                transform = Array.from(new Set(this.templateOptions.transforms));
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: transform,
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata,
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Deprecated.
     *
     * @see https://github.com/aws/aws-cdk/pull/7187
     * @returns reference itself without any change
     * @deprecated cross reference handling has been moved to `App.prepare()`.
     */
    prepareCrossReference(_sourceStack, reference) {
        return reference;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        var _a, _b, _c, _d;
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region.
        //
        // (They do not need to be anchored to any construct like resource attributes
        // are, because we'll never Export/Fn::ImportValue them -- the only situation
        // in which Export/Fn::ImportValue would work is if the value are the same
        // between producer and consumer anyway, so we can just assume that they are).
        const containingAssembly = stage_1.Stage.of(this);
        const account = (_b = (_a = env.account) !== null && _a !== void 0 ? _a : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.account) !== null && _b !== void 0 ? _b : cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = (_d = (_c = env.region) !== null && _c !== void 0 ? _c : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.region) !== null && _d !== void 0 ? _d : cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account, region,
            environment: cxapi.EnvironmentUtils.format(envAccount, envRegion),
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of Object.values(this._stackDependencies)) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [...dep.reasons, ...ret];
            }
        }
        return undefined;
    }
    /**
     * Calculate the stack name based on the construct path
     *
     * The stack name is the name under which we'll deploy the stack,
     * and incorporates containing Stage names by default.
     *
     * Generally this looks a lot like how logical IDs are calculated.
     * The stack name is calculated based on the construct root path,
     * as follows:
     *
     * - Path is calculated with respect to containing App or Stage (if any)
     * - If the path is one component long just use that component, otherwise
     *   combine them with a hash.
     *
     * Since the hash is quite ugly and we'd like to avoid it if possible -- but
     * we can't anymore in the general case since it has been written into legacy
     * stacks. The introduction of Stages makes it possible to make this nicer however.
     * When a Stack is nested inside a Stage, we use the path components below the
     * Stage, and prefix the path components of the Stage before it.
     */
    generateStackName() {
        const assembly = stage_1.Stage.of(this);
        const prefix = (assembly && assembly.stageName) ? `${assembly.stageName}-` : '';
        return `${prefix}${this.generateStackId(assembly)}`;
    }
    /**
     * The artifact ID for this stack
     *
     * Stack artifact ID is unique within the App's Cloud Assembly.
     */
    generateStackArtifactId() {
        return this.generateStackId(this.node.root);
    }
    /**
     * Generate an ID with respect to the given container construct.
     */
    generateStackId(container) {
        const rootPath = rootPathTo(this, container);
        const ids = rootPath.map(c => c.node.id);
        // In unit tests our Stack (which is the only component) may not have an
        // id, so in that case just pretend it's "Stack".
        if (ids.length === 1 && !ids[0]) {
            throw new Error('unexpected: stack id must always be defined');
        }
        return makeStackName(ids);
    }
}
exports.Stack = Stack;
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case 'Description':
            return `${val1}\n${val2}`;
        case 'AWSTemplateFormatVersion':
            if (val1 != null && val2 != null && val1 !== val2) {
                throw new Error(`Conflicting CloudFormation template versions provided: '${val1}' and '${val2}`);
            }
            return val1 !== null && val1 !== void 0 ? val1 : val2;
        case 'Transform':
            return mergeSets(val1, val2);
        default:
            return mergeObjectsWithoutDuplicates(section, val1, val2);
    }
}
function mergeSets(val1, val2) {
    const array1 = val1 == null ? [] : (Array.isArray(val1) ? val1 : [val1]);
    const array2 = val2 == null ? [] : (Array.isArray(val2) ? val2 : [val2]);
    for (const value of array2) {
        if (!array1.includes(value)) {
            array1.push(value);
        }
    }
    return array1.length === 1 ? array1[0] : array1;
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
/**
 * Collect all CfnElements from a Stack.
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
/**
 * Return the construct root path of the given construct relative to the given ancestor
 *
 * If no ancestor is given or the ancestor is not found, return the entire root path.
 */
function rootPathTo(construct, ancestor) {
    const scopes = construct.node.scopes;
    for (let i = scopes.length - 2; i >= 0; i--) {
        if (scopes[i] === ancestor) {
            return scopes.slice(i + 1);
        }
    }
    return scopes;
}
exports.rootPathTo = rootPathTo;
/**
 * makeUniqueId, specialized for Stack names
 *
 * Stack names may contain '-', so we allow that character if the stack name
 * has only one component. Otherwise we fall back to the regular "makeUniqueId"
 * behavior.
 */
function makeStackName(components) {
    if (components.length === 1) {
        return components[0];
    }
    return uniqueid_1.makeUniqueId(components);
}
// These imports have to be at the end to prevent circular imports
const deps_1 = require("./deps");
const stack_synthesizers_1 = require("./stack-synthesizers");
const stage_1 = require("./stage");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
const fs_1 = require("./fs");
//# sourceMappingURL=data:application/json;base64,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