"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnJson = void 0;
const construct_compat_1 = require("./construct-compat");
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which
 * resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences
     * this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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