"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const fields_1 = require("../fields");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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