"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/fr_fr/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/fr_fr/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principle = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principle);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principle, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
//# sourceMappingURL=data:application/json;base64,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