"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = void 0;
const os_1 = require("os");
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: options.paths ? options.paths.map(p => this.arnForObjects(p)) : [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${stack.region}.${stack.urlSuffix}/`;
        return this.buildUrl(prefix, key);
    }
    /**
     * The S3 URL of an S3 object. For example:
     * @example s3://onlybucket
     * @example s3://bucket/key
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key) {
        return this.buildUrl('s3://', key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you specify multiple components for keyPattern, they will be concatenated::
     *
     *   arnForObjects('home/', team, '/', user, '/*')
     *
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_WRITE_ACTIONS, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACTIONS, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(perms.BUCKET_WRITE_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.Anyone(),
            resource: this,
        });
    }
    buildUrl(prefix, key) {
        const components = [
            prefix,
            this.bucketName,
        ];
        if (key) {
            // trim prepending '/'
            if (typeof key === 'string' && key.startsWith('/')) {
                key = key.substr(1);
            }
            components.push('/');
            components.push(key);
        }
        return components.join('');
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        let ret;
        if (crossAccountAccess) {
            // if the access is cross-account, we need to trust the accessing principal in the bucket's policy
            ret = iam.Grant.addToPrincipalAndResource({
                grantee,
                actions: bucketActions,
                resourceArns: resources,
                resource: this,
            });
        }
        else {
            // if not, we don't need to modify the resource policy if the grantee is an identity principal
            ret = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: bucketActions,
                resourceArns: resources,
                resource: this,
            });
        }
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
class BlockPublicAccess {
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
exports.BlockPublicAccess = BlockPublicAccess;
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
/**
 * All http request methods
 */
var HttpMethods;
(function (HttpMethods) {
    /**
     * The GET method requests a representation of the specified resource.
     */
    HttpMethods["GET"] = "GET";
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     */
    HttpMethods["PUT"] = "PUT";
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     */
    HttpMethods["HEAD"] = "HEAD";
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     */
    HttpMethods["POST"] = "POST";
    /**
     * The DELETE method deletes the specified resource.
     */
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 */
var InventoryFormat;
(function (InventoryFormat) {
    /**
     * Generate the inventory list as CSV.
     */
    InventoryFormat["CSV"] = "CSV";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["PARQUET"] = "Parquet";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    /**
     * A report is generated every day.
     */
    InventoryFrequency["DAILY"] = "Daily";
    /**
     * A report is generated every Sunday (UTC timezone) after the initial report.
     */
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    /**
     * Includes all versions of each object in the report.
     */
    InventoryObjectVersion["ALL"] = "All";
    /**
     * Includes only the current version of each object in the report.
     */
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 */
class Bucket extends BucketBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.bucketName,
        });
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        this.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.stringValue({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseInventoryConfiguration() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery();
        }
        for (const inventory of (_a = props.inventories) !== null && _a !== void 0 ? _a : []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
    }
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const region = stack.region;
        const urlSuffix = stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteDomain = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                var _a;
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = (_a = attrs.isWebsite) !== null && _a !== void 0 ? _a : false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpiration !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket
     *
     * @param rule The CORS configuration rule to add
     */
    addCorsRule(rule) {
        this.cors.push(rule);
    }
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add
     */
    addInventory(inventory) {
        this.inventories.push(inventory);
    }
    validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            const enabled = rule.enabled !== undefined ? rule.enabled : true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: rule.expiration && rule.expiration.toDays(),
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpiration && rule.noncurrentVersionExpiration.toDays(),
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                tagFilters: self.parseTagFilters(rule.tagFilters),
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        var _a;
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        return {
            destinationBucketName: (_a = props.serverAccessLogsBucket) === null || _a === void 0 ? void 0 : _a.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    allowLogDelivery() {
        if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            var _a, _b, _c, _d, _e, _f;
            const format = (_a = inventory.format) !== null && _a !== void 0 ? _a : InventoryFormat.CSV;
            const frequency = (_b = inventory.frequency) !== null && _b !== void 0 ? _b : InventoryFrequency.WEEKLY;
            const id = (_c = inventory.inventoryId) !== null && _c !== void 0 ? _c : `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${(_d = inventory.destination.prefix) !== null && _d !== void 0 ? _d : ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: (_e = inventory.enabled) !== null && _e !== void 0 ? _e : true,
                includedObjectVersions: (_f = inventory.includeObjectVersions) !== null && _f !== void 0 ? _f : InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
}
exports.Bucket = Bucket;
/**
 * What kind of server-side encryption to apply to this bucket
 */
var BucketEncryption;
(function (BucketEncryption) {
    /**
     * Objects in the bucket are not encrypted.
     */
    BucketEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    BucketEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a master key managed by S3.
     */
    BucketEncryption["S3_MANAGED"] = "S3MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 */
var EventType;
(function (EventType) {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    /**
     * Owner gets FULL_CONTROL. No one else has access rights.
     */
    BucketAccessControl["PRIVATE"] = "Private";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ access.
     */
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     */
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    /**
     * Owner gets FULL_CONTROL. The AuthenticatedUsers group gets READ access.
     */
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     */
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    /**
     * Object owner gets FULL_CONTROL. Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    /**
     * Owner gets FULL_CONTROL. Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     */
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
class ReplaceKey {
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
    /**
     * The specific object key to use in the redirect request
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
}
exports.ReplaceKey = ReplaceKey;
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,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