"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (AWS SAM docker image)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                ...(_b = options.buildArgs) !== null && _b !== void 0 ? _b : {},
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_c = options.parcelVersion) !== null && _c !== void 0 ? _c : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_d = options.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_e = options.nodeModules) !== null && _e !== void 0 ? _e : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_f = options.sourceMaps) !== null && _f !== void 0 ? _f : false,
                    minify: (_g = options.minify) !== null && _g !== void 0 ? _g : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const distFile = path.basename(options.entry).replace(/\.ts$/, '.js');
        const parcelCommand = chain([
            [
                '$(node -p "require.resolve(\'parcel\')")',
                'build', containerEntryPath.replace(/\\/g, '/'),
                '--target', 'cdk-lambda',
                '--dist-dir', cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
                '--no-autoinstall',
                '--no-scope-hoist',
                ...options.cacheDir
                    ? ['--cache-dir', '/parcel-cache']
                    : [],
            ].join(' '),
            // Always rename dist file to index.js because Lambda doesn't support filenames
            // with multiple dots and we can end up with multiple dots when using automatic
            // entry lookup
            `mv ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${distFile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
        ]);
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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