"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Effect = exports.PolicyStatement = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const principals_1 = require("./principals");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== 'string' && !Array.isArray(field)) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== 'string')) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     * This will accept an object created from the `.toJSON()` call
     * @param obj the PolicyStatement in object form.
     */
    static fromJson(obj) {
        return new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined,
        });
    }
    //
    // Actions
    //
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     *
     * @param actions actions that will be allowed.
     */
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error('Cannot add \'Actions\' to policy statement if \'NotActions\' have been added');
        }
        this.action.push(...actions);
    }
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section
     * of the policy document.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     *
     * @param notActions actions that will be denied. All other actions will be permitted.
     */
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error('Cannot add \'NotActions\' to policy statement if \'Actions\' have been added');
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     *
     * @param principals IAM principals that will be added
     */
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error('Cannot add \'Principals\' to policy statement if \'NotPrincipals\' have been added');
        }
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of
     * a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     *
     * @param notPrincipals IAM principals that will be denied access
     */
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error('Cannot add \'NotPrincipals\' to policy statement if \'Principals\' have been added');
        }
        for (const notPrincipal of notPrincipals) {
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     */
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    /**
     * Specify a principal using the ARN  identifier of the principal.
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com')
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    /**
     * Adds an AWS account root user principal to this policy statement
     */
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    /**
     * Adds a canonical user ID principal to this policy document
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account
     */
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    /**
     * Adds all identities in all accounts ("*") to this policy statement
     */
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    /**
     * Specify resources that this policy statement applies into the "Resource" section of
     * this policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to
     */
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error('Cannot add \'Resources\' to policy statement if \'NotResources\' have been added');
        }
        this.resource.push(...arns);
    }
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section
     * of this policy statement. All resources except the specified list will be matched.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to
     */
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error('Cannot add \'NotResources\' to policy statement if \'Resources\' have been added');
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        const existingValue = this.condition[key];
        this.condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    /**
     * JSON-ify the policy statement
     *
     * Used when JSON.stringify() is called
     */
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    /**
     * String representation of this policy statement
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement',
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    addPrincipalConditions(conditions) {
        // Stringifying the conditions is an easy way to do deep equality
        const theseConditions = JSON.stringify(conditions);
        if (this.principalConditionsJson === undefined) {
            // First principal, anything goes
            this.principalConditionsJson = theseConditions;
        }
        else {
            if (this.principalConditionsJson !== theseConditions) {
                throw new Error(`All principals in a PolicyStatement must have the same Conditions (got '${this.principalConditionsJson}' and '${theseConditions}'). Use multiple statements instead.`);
            }
        }
        this.addConditions(conditions);
    }
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     */
    validateForAnyPolicy() {
        const errors = new Array();
        if (this.action.length === 0 && this.notAction.length === 0) {
            errors.push('A PolicyStatement must specify at least one \'action\' or \'notAction\'.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     */
    validateForResourcePolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length === 0 && Object.keys(this.notPrincipal).length === 0) {
            errors.push('A PolicyStatement used in a resource-based policy must specify at least one IAM principal.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     */
    validateForIdentityPolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0) {
            errors.push('A PolicyStatement used in an identity-based policy cannot specify any IAM principals.');
        }
        if (Object.keys(this.resource).length === 0 && Object.keys(this.notResource).length === 0) {
            errors.push('A PolicyStatement used in an identity-based policy must specify at least one resource.');
        }
        return errors;
    }
}
exports.PolicyStatement = PolicyStatement;
/**
 * The Effect element of an IAM policy
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 */
var Effect;
(function (Effect) {
    /**
     * Allows access to a resource in an IAM policy statement. By default, access to resources are denied.
     */
    Effect["ALLOW"] = "Allow";
    /**
     * Explicitly deny access to a resource. By default, all requests are denied implicitly.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html
     */
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into an "AWS" principal
        if (typeof (json) === 'string') {
            json = { AWS: json };
        }
        this.policyFragment = {
            principalJson: json,
            conditions: {},
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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