"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const fs = require("fs");
const path = require("path");
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_patch_1 = require("./k8s-patch");
const k8s_resource_1 = require("./k8s-resource");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Endpoint access characteristics.
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * Restrict public access to specific CIDR blocks.
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     */
    onlyFrom(...cidr) {
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
exports.EndpointAccess = EndpointAccess;
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * The cluster endpoint is only accessible through your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
/**
 * Kubernetes cluster version
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
/**
 * Kubernetes version 1.14
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * Kubernetes version 1.17
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        if (props.kubectlEnabled === false) {
            throw new Error('The "eks.Cluster" class no longer allows disabling kubectl support. ' +
                'As a temporary workaround, you can use the drop-in replacement class `eks.LegacyCluster`, ' +
                'but bear in mind that this class will soon be removed and will no longer receive additional ' +
                'features or bugfixes. See https://github.com/aws/aws-cdk/issues/9332 for more details');
        }
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        this.vpcSubnets = (_a = props.vpcSubnets) !== null && _a !== void 0 ? _a : [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        // Get subnetIds for all selected subnets
        const subnetIds = [...new Set(Array().concat(...this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
        };
        this.endpointAccess = (_b = props.endpointAccess) !== null && _b !== void 0 ? _b : EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlProviderEnv = props.kubectlEnvironment;
        if (this.endpointAccess._config.privateAccess && this.vpc instanceof ec2.Vpc) {
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (!this.vpc.dnsHostnamesEnabled || !this.vpc.dnsSupportEnabled) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
        }
        this.kubctlProviderSecurityGroup = new ec2.SecurityGroup(this, 'KubectlProviderSecurityGroup', {
            vpc: this.vpc,
            description: 'Comminication between KubectlProvider and EKS Control Plane',
        });
        // grant the kubectl provider access to the cluster control plane.
        this.connections.allowFrom(this.kubctlProviderSecurityGroup, this.connections.defaultPort);
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            ...clusterProps,
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
        });
        // the security group and vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
        // this ensures that.
        this._clusterResource.node.addDependency(this.kubctlProviderSecurityGroup, this.vpc);
        // see https://github.com/aws/aws-cdk/issues/9027
        this._clusterResource.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['ec2:DescribeVpcs'],
            resources: [stack.formatArn({
                    service: 'ec2',
                    resource: 'vpc',
                    resourceName: this.vpc.vpcId,
                })],
        }));
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = (_c = props.mastersRole) !== null && _c !== void 0 ? _c : new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType((_d = props.coreDnsComputeType) !== null && _d !== void 0 ? _d : CoreDnsComputeType.EC2);
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new BottleRocketImage() :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tag.add(autoScalingGroup, `kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice) {
            this.addSpotInterruptHandler();
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new iam.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
                clientIds: ['sts.amazonaws.com'],
                /**
                 * For some reason EKS isn't validating the root certificate but a intermediat certificate
                 * which is one level up in the tree. Because of the a constant thumbprint value has to be
                 * stated with this OpenID Connect provider. The certificate thumbprint is the same for all the regions.
                 */
                thumbprints: ['9e99a48a9960b14926bb7f3b02e22da2b0ab7280'],
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Adds a service account to this cluster.
     *
     * @param id the id of this service account
     * @param options service account options
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    get _kubectlCreationRole() {
        return this._clusterResource.creationRole;
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        if (this._kubectlReadyBarrier) {
            this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        }
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @property resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // singleton
        let provider = this.stack.node.tryFindChild(uid);
        if (!provider) {
            // create the provider.
            let providerProps = {
                env: this.kubectlProviderEnv,
            };
            if (!this.endpointAccess._config.publicAccess) {
                const privateSubents = this.selectPrivateSubnets().slice(0, 16);
                if (privateSubents.length === 0) {
                    throw new Error('Vpc must contain private subnets to configure private endpoint access');
                }
                // endpoint access is private only, we need to attach the
                // provider to the VPC so that it can access the cluster.
                providerProps = {
                    ...providerProps,
                    vpc: this.vpc,
                    // lambda can only be accociated with max 16 subnets and they all need to be private.
                    vpcSubnets: { subnets: privateSubents },
                    securityGroups: [this.kubctlProviderSecurityGroup],
                };
            }
            provider = new kubectl_provider_1.KubectlProvider(this.stack, uid, providerProps);
        }
        // allow the kubectl provider to assume the cluster creation role.
        this._clusterResource.addTrustedRole(provider.role);
        if (!this._kubectlReadyBarrier) {
            throw new Error('unexpected: kubectl enabled clusters should have a kubectl-ready barrier resource');
        }
        resourceScope.node.addDependency(this._kubectlReadyBarrier);
        return provider;
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (this.vpc.privateSubnets.includes(subnet)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (this.vpc.publicSubnets.includes(subnet)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.7.3',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    'nodeSelector.lifecycle': user_data_1.LifecycleLabel.SPOT,
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addResource('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                subnet.node.applyAspect(new core_1.Tag(tag, '1'));
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        this.clusterSecurityGroupId = props.clusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = props.clusterEncryptionConfigKeyArn;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        var _a, _b;
        this.nodeType = (_a = props.nodeType) !== null && _a !== void 0 ? _a : NodeType.STANDARD;
        this.kubernetesVersion = (_b = props.kubernetesVersion) !== null && _b !== void 0 ? _b : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor() {
        // only 1.15 is currently available
        this.kubernetesVersion = '1.15';
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/aws-k8s-${this.kubernetesVersion}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
const INFERENTIA_INSTANCETYPES = ['inf1'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        INFERENTIA_INSTANCETYPES.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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