"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn))),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.creationRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    /**
     * Grants `trustedRole` permissions to assume the creation role.
     */
    addTrustedRole(trustedRole) {
        if (this.trustedPrincipals.includes(trustedRole.roleArn)) {
            return;
        }
        if (!this.creationRole.assumeRolePolicy) {
            throw new Error('unexpected: cluster creation role must have trust policy');
        }
        this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ArnPrincipal(trustedRole.roleArn)],
        }));
        this.trustedPrincipals.push(trustedRole.roleArn);
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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