"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a, _b, _c;
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = ((_a = props.signInAliases) === null || _a === void 0 ? void 0 : _a.phone) === true || ((_b = props.autoVerify) === null || _b === void 0 ? void 0 : _b.phone) === true;
        const roleRequired = (mfaConfiguration === null || mfaConfiguration === void 0 ? void 0 : mfaConfiguration.includes('SMS_MFA')) || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = (_c = props.smsRole) !== null && _c !== void 0 ? _c : new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _a;
        const accountRecovery = (_a = props.accountRecovery) !== null && _a !== void 0 ? _a : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1wb29sLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDJDQUErRixDQUFDLG1EQUFtRDtBQUVuSixxQ0FBMEYsQ0FBQyxnREFBZ0Q7QUFDM0ksMkRBQWtEO0FBQ2xELHFEQUE4RDtBQUU5RCx5REFBMkU7QUFDM0UseURBQTJFO0FBcUgzRTs7R0FFRztBQUNILE1BQWEsaUJBQWlCO0lBMEQxQixZQUFvQixhQUFxQjtRQUNyQyxJQUFJLENBQUMsYUFBYSxHQUFHLGFBQWEsQ0FBQztJQUN2QyxDQUFDO0lBVEQsbUNBQW1DO0lBQzVCLE1BQU0sQ0FBQyxFQUFFLENBQUMsSUFBWTtRQUN6QixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsT0FBTyxJQUFJLGlCQUFpQixDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7O0FBdkRMLDhDQTZEQztBQTVERzs7O0dBR0c7QUFDb0IsdUNBQXFCLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0FBQzVGOzs7R0FHRztBQUNvQixnQ0FBYyxHQUFHLElBQUksaUJBQWlCLENBQUMsZUFBZSxDQUFDLENBQUM7QUFDL0U7OztHQUdHO0FBQ29CLHVDQUFxQixHQUFHLElBQUksaUJBQWlCLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQUM1Rjs7O0dBR0c7QUFDb0IscUNBQW1CLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0FBQ3pGOzs7R0FHRztBQUNvQixtQ0FBaUIsR0FBRyxJQUFJLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLENBQUM7QUFDckY7OztHQUdHO0FBQ29CLG9DQUFrQixHQUFHLElBQUksaUJBQWlCLENBQUMsbUJBQW1CLENBQUMsQ0FBQztBQUN2Rjs7O0dBR0c7QUFDb0IsNkJBQVcsR0FBRyxJQUFJLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxDQUFDO0FBQ3hFOzs7R0FHRztBQUNvQixzQ0FBb0IsR0FBRyxJQUFJLGlCQUFpQixDQUFDLG9CQUFvQixDQUFDLENBQUM7QUFDMUY7OztHQUdHO0FBQ29CLGdDQUFjLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsQ0FBQztBQUMvRTs7O0dBR0c7QUFDb0IsZ0RBQThCLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0FBWWpIOztHQUVHO0FBQ0gsSUFBWSxzQkFLWDtBQUxELFdBQVksc0JBQXNCO0lBQzlCLDRCQUE0QjtJQUM1QixvREFBMEIsQ0FBQTtJQUMxQiw0QkFBNEI7SUFDNUIsb0RBQTBCLENBQUE7QUFDOUIsQ0FBQyxFQUxXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBS2pDO0FBeUREOzs7R0FHRztBQUNILElBQVksR0FPWDtBQVBELFdBQVksR0FBRztJQUNYLCtFQUErRTtJQUMvRSxrQkFBVyxDQUFBO0lBQ1gsaUdBQWlHO0lBQ2pHLDRCQUFxQixDQUFBO0lBQ3JCLDZFQUE2RTtJQUM3RSxzQkFBZSxDQUFBO0FBQ25CLENBQUMsRUFQVyxHQUFHLEdBQUgsV0FBRyxLQUFILFdBQUcsUUFPZDtBQXdFRDs7Ozs7Ozs7R0FRRztBQUNILElBQVksZUF5Qlg7QUF6QkQsV0FBWSxlQUFlO0lBQ3ZCOztPQUVHO0lBQ0gsbUdBQTJCLENBQUE7SUFDM0I7O09BRUc7SUFDSCxtR0FBMkIsQ0FBQTtJQUMzQjs7T0FFRztJQUNILGlFQUFVLENBQUE7SUFDVjs7T0FFRztJQUNILHlGQUFzQixDQUFBO0lBQ3RCOztPQUVHO0lBQ0gsMkVBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gscURBQUksQ0FBQTtBQUNSLENBQUMsRUF6QlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUF5QjFCO0FBOEpELE1BQWUsWUFBYSxTQUFRLGVBQVE7SUFBNUM7O1FBR29CLHNCQUFpQixHQUFnQyxFQUFFLENBQUM7SUFnQnhFLENBQUM7SUFmVSxTQUFTLENBQUMsRUFBVSxFQUFFLE9BQStCO1FBQ3hELE9BQU8sSUFBSSxpQ0FBYyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDaEMsUUFBUSxFQUFFLElBQUk7WUFDZCxHQUFHLE9BQU87U0FDYixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ00sU0FBUyxDQUFDLEVBQVUsRUFBRSxPQUE4QjtRQUN2RCxPQUFPLElBQUksaUNBQWMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ2hDLFFBQVEsRUFBRSxJQUFJO1lBQ2QsR0FBRyxPQUFPO1NBQ2IsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNNLHdCQUF3QixDQUFDLFFBQW1DO1FBQy9ELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDMUMsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLFFBQVMsU0FBUSxZQUFZO0lBd0N0QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXVCLEVBQUU7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFGYixhQUFRLEdBQXFDLEVBQUUsQ0FBQztRQUdwRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDL0MsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3RCLEtBQUssTUFBTSxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQy9DLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3hDLElBQUksT0FBTyxLQUFLLFNBQVMsRUFBRTtvQkFDdkIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE9BQTJCLEVBQUUsQ0FBQyxDQUFDLENBQUM7b0JBQ3hELElBQUksQ0FBQyxRQUFnQixDQUFDLENBQUMsQ0FBQyxHQUFJLE9BQTRCLENBQUMsV0FBVyxDQUFDO2lCQUN6RTthQUNKO1NBQ0o7UUFDRCxNQUFNLDJCQUEyQixHQUFHLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNqRixJQUFJLHdCQUF3QixDQUFDO1FBQzdCLElBQUksd0JBQXdCLENBQUM7UUFDN0IsSUFBSSwyQkFBMkIsQ0FBQyxrQkFBa0IsS0FBSyxzQkFBc0IsQ0FBQyxJQUFJLEVBQUU7WUFDaEYsd0JBQXdCLEdBQUcsMkJBQTJCLENBQUMsWUFBWSxDQUFDO1lBQ3BFLHdCQUF3QixHQUFHLDJCQUEyQixDQUFDLFlBQVksQ0FBQztTQUN2RTtRQUNELE1BQU0sc0JBQXNCLEdBQUcsMkJBQTJCLENBQUMsVUFBVSxDQUFDO1FBQ3RFLE1BQU0scUJBQXFCLEdBQThDO1lBQ3JFLFlBQVksUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxTQUFTO1lBQzdDLFlBQVksUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxZQUFZO1lBQ2hELFVBQVUsUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxVQUFVO1NBQy9DLENBQUM7UUFDRixNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBQ2xHLE1BQU0scUJBQXFCLEdBQThDO1lBQ3JFLHdCQUF3QixFQUFFLENBQUMsaUJBQWlCO1lBQzVDLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNoRyxDQUFDO1FBQ0YsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNELE1BQU0sUUFBUSxHQUFHLElBQUksK0JBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQy9DLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxrQkFBa0IsRUFBRSxNQUFNLENBQUMsYUFBYTtZQUN4QyxlQUFlLEVBQUUsTUFBTSxDQUFDLFVBQVU7WUFDbEMsc0JBQXNCLEVBQUUsTUFBTSxDQUFDLGVBQWU7WUFDOUMsWUFBWSxFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUM7WUFDaEYsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQztZQUM5QyxxQkFBcUI7WUFDckIsd0JBQXdCO1lBQ3hCLHdCQUF3QjtZQUN4QixzQkFBc0I7WUFDdEIsMkJBQTJCO1lBQzNCLE1BQU0sRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDO1lBQ3ZDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQzNCLFdBQVcsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDO1lBQ3pDLFFBQVEsRUFBRSxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3ZFLGtCQUFrQixFQUFFLGlCQUFpQixDQUFDO2dCQUNsQyxJQUFJLFFBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsSUFBSTtnQkFDL0IsbUJBQW1CLFFBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsT0FBTzthQUNwRCxDQUFDO1lBQ0YscUJBQXFCLEVBQUUsaUJBQWlCLENBQUM7Z0JBQ3JDLGFBQWEsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2FBQzNDLENBQUM7WUFDRixzQkFBc0IsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQztTQUN0RCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsVUFBVSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFDL0IsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUM7UUFDdEQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxlQUFlLENBQUM7SUFDeEQsQ0FBQztJQW5HRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsVUFBa0I7UUFDekUsTUFBTSxNQUFPLFNBQVEsWUFBWTtZQUFqQzs7Z0JBQ29CLGVBQVUsR0FBRyxVQUFVLENBQUM7Z0JBQ3hCLGdCQUFXLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ25ELE9BQU8sRUFBRSxhQUFhO29CQUN0QixRQUFRLEVBQUUsVUFBVTtvQkFDcEIsWUFBWSxFQUFFLFVBQVU7aUJBQzNCLENBQUMsQ0FBQztZQUNQLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsV0FBbUI7UUFDM0UsT0FBTyxRQUFRLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUMsWUFBYSxDQUFDLENBQUM7SUFDbkcsQ0FBQztJQWlGRDs7O09BR0c7SUFDSSxVQUFVLENBQUMsU0FBNEIsRUFBRSxFQUFvQjtRQUNoRSxJQUFJLFNBQVMsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixTQUFTLGtCQUFrQixDQUFDLENBQUM7U0FDL0U7UUFDRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxFQUFFLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNyRCxJQUFJLENBQUMsUUFBZ0IsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQztJQUNyRSxDQUFDO0lBQ08sbUJBQW1CLENBQUMsRUFBb0IsRUFBRSxJQUFZO1FBQzFELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNoRSxFQUFFLENBQUMsYUFBYSxDQUFDLEdBQUcsVUFBVSxTQUFTLEVBQUU7WUFDckMsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsMkJBQTJCLENBQUM7WUFDNUQsU0FBUyxFQUFFLElBQUksQ0FBQyxXQUFXO1NBQzlCLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDTyxnQ0FBZ0MsQ0FBQyxLQUFvQjs7UUFDekQsTUFBTSxhQUFhLEdBQUcsUUFBUSxDQUFDO1FBQy9CLE1BQU0scUJBQXFCLEdBQUcsb0JBQW9CLENBQUM7UUFDbkQsTUFBTSxVQUFVLGVBQUcsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxVQUFVLG1DQUFJLHNCQUFzQixDQUFDLElBQUksQ0FBQztRQUNyRixNQUFNLFlBQVksZUFBRyxLQUFLLENBQUMsZ0JBQWdCLDBDQUFFLFlBQVksbUNBQUkseUJBQXlCLENBQUM7UUFDdkYsTUFBTSxVQUFVLGVBQUcsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxVQUFVLG1DQUFJLGdEQUFnRCxhQUFhLEVBQUUsQ0FBQztRQUN6SCxJQUFJLFVBQVUsS0FBSyxzQkFBc0IsQ0FBQyxJQUFJLEVBQUU7WUFDNUMsTUFBTSxZQUFZLGVBQUcsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxTQUFTLG1DQUFJLGdEQUFnRCxhQUFhLEVBQUUsQ0FBQztZQUMxSCxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUMsSUFBSSxZQUFZLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDOUUsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsYUFBYSxHQUFHLENBQUMsQ0FBQzthQUNsRztZQUNELElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUMxRSxNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxhQUFhLEdBQUcsQ0FBQyxDQUFDO2FBQ3RGO1lBQ0QsT0FBTztnQkFDSCxrQkFBa0IsRUFBRSxzQkFBc0IsQ0FBQyxJQUFJO2dCQUMvQyxZQUFZO2dCQUNaLFlBQVk7Z0JBQ1osVUFBVTthQUNiLENBQUM7U0FDTDthQUNJO1lBQ0QsTUFBTSxZQUFZLGVBQUcsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxTQUFTLG1DQUNsRCxzQ0FBc0MscUJBQXFCLEVBQUUsQ0FBQztZQUNsRSxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUMsSUFBSSxZQUFZLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUN0RixNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxxQkFBcUIsR0FBRyxDQUFDLENBQUM7YUFDMUc7WUFDRCxPQUFPO2dCQUNILGtCQUFrQixFQUFFLHNCQUFzQixDQUFDLElBQUk7Z0JBQy9DLGtCQUFrQixFQUFFLFlBQVk7Z0JBQ2hDLGtCQUFrQixFQUFFLFlBQVk7Z0JBQ2hDLFVBQVU7YUFDYixDQUFDO1NBQ0w7SUFDTCxDQUFDO0lBQ08sbUJBQW1CLENBQUMsS0FBb0I7O1FBQzVDLElBQUksVUFBZ0MsQ0FBQztRQUNyQyxJQUFJLGFBQW1DLENBQUM7UUFDeEMsSUFBSSxlQUFxQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxTQUFrQixLQUFLLENBQUMsYUFBYSxtQ0FBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQztRQUN4RSxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUU7WUFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3RGO1FBQ0QsSUFBSSxNQUFNLENBQUMsUUFBUSxFQUFFO1lBQ2pCLFVBQVUsR0FBRyxFQUFFLENBQUM7WUFDaEIsSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFO2dCQUNkLFVBQVUsQ0FBQyxJQUFJLENBQUMsbUNBQXNCLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDakQ7WUFDRCxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUU7Z0JBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQyxtQ0FBc0IsQ0FBQyxXQUFXLENBQUMsQ0FBQzthQUN2RDtZQUNELElBQUksTUFBTSxDQUFDLGlCQUFpQixFQUFFO2dCQUMxQixVQUFVLENBQUMsSUFBSSxDQUFDLG1DQUFzQixDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDN0Q7WUFDRCxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUN6QixVQUFVLEdBQUcsU0FBUyxDQUFDO2FBQzFCO1NBQ0o7YUFDSTtZQUNELGFBQWEsR0FBRyxFQUFFLENBQUM7WUFDbkIsSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFO2dCQUNkLGFBQWEsQ0FBQyxJQUFJLENBQUMsbUNBQXNCLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDcEQ7WUFDRCxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUU7Z0JBQ2QsYUFBYSxDQUFDLElBQUksQ0FBQyxtQ0FBc0IsQ0FBQyxXQUFXLENBQUMsQ0FBQzthQUMxRDtTQUNKO1FBQ0QsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ2xCLGVBQWUsR0FBRyxFQUFFLENBQUM7WUFDckIsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRTtnQkFDeEIsZUFBZSxDQUFDLElBQUksQ0FBQyxtQ0FBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUN0RDtZQUNELElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUU7Z0JBQ3hCLGVBQWUsQ0FBQyxJQUFJLENBQUMsbUNBQXNCLENBQUMsV0FBVyxDQUFDLENBQUM7YUFDNUQ7U0FDSjthQUNJLElBQUksTUFBTSxDQUFDLEtBQUssSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFO1lBQ25DLGVBQWUsR0FBRyxFQUFFLENBQUM7WUFDckIsSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFO2dCQUNkLGVBQWUsQ0FBQyxJQUFJLENBQUMsbUNBQXNCLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDdEQ7WUFDRCxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUU7Z0JBQ2QsZUFBZSxDQUFDLElBQUksQ0FBQyxtQ0FBc0IsQ0FBQyxXQUFXLENBQUMsQ0FBQzthQUM1RDtTQUNKO1FBQ0QsT0FBTyxFQUFFLGFBQWEsRUFBRSxVQUFVLEVBQUUsZUFBZSxFQUFFLENBQUM7SUFDMUQsQ0FBQztJQUNPLGdCQUFnQixDQUFDLEtBQW9COztRQUN6QyxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssS0FBSyxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsQ0FBQyxDQUFDO1NBQ2pGO1FBQ0QsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ2YsT0FBTztnQkFDSCxZQUFZLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPO2dCQUNuQyxVQUFVLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjthQUN0QyxDQUFDO1NBQ0w7UUFDRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssS0FBSyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEQsTUFBTSxpQkFBaUIsR0FBRyxPQUFBLEtBQUssQ0FBQyxhQUFhLDBDQUFFLEtBQUssTUFBSyxJQUFJLElBQUksT0FBQSxLQUFLLENBQUMsVUFBVSwwQ0FBRSxLQUFLLE1BQUssSUFBSSxDQUFDO1FBQ2xHLE1BQU0sWUFBWSxHQUFHLENBQUEsZ0JBQWdCLGFBQWhCLGdCQUFnQix1QkFBaEIsZ0JBQWdCLENBQUUsUUFBUSxDQUFDLFNBQVMsTUFBSyxpQkFBaUIsQ0FBQztRQUNoRixJQUFJLENBQUMsWUFBWSxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQ3BELE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsb0NBQW9DO1FBQ2xHLE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxPQUFPLG1DQUFJLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDdkQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsMkJBQTJCLEVBQUU7Z0JBQ3pELFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxpQkFBaUIsRUFBRTtpQkFDeEQ7YUFDSixDQUFDO1lBQ0YsY0FBYyxFQUFFO2dCQUNaOzs7O29CQUlJO2dCQUNKLGFBQWEsRUFBRSxJQUFJLHdCQUFjLENBQUM7b0JBQzlCLFVBQVUsRUFBRTt3QkFDUixJQUFJLHlCQUFlLENBQUM7NEJBQ2hCLE9BQU8sRUFBRSxDQUFDLGFBQWEsQ0FBQzs0QkFDeEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO3lCQUNuQixDQUFDO3FCQUNMO2lCQUNKLENBQUM7YUFDTDtTQUNKLENBQUMsQ0FBQztRQUNILE9BQU87WUFDSCxVQUFVLEVBQUUsaUJBQWlCO1lBQzdCLFlBQVksRUFBRSxPQUFPLENBQUMsT0FBTztTQUNoQyxDQUFDO0lBQ04sQ0FBQztJQUNPLGdCQUFnQixDQUFDLEtBQW9CO1FBQ3pDLElBQUksS0FBSyxDQUFDLEdBQUcsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLEdBQUcsS0FBSyxHQUFHLENBQUMsR0FBRyxFQUFFO1lBQ2xELDZEQUE2RDtZQUM3RCxPQUFPLFNBQVMsQ0FBQztTQUNwQjthQUNJLElBQUksS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTO1lBQ3hDLENBQUMsS0FBSyxDQUFDLEdBQUcsS0FBSyxHQUFHLENBQUMsUUFBUSxJQUFJLEtBQUssQ0FBQyxHQUFHLEtBQUssR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQzVELE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUN0QjthQUNJO1lBQ0QsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1lBQ3ZCLElBQUksS0FBSyxDQUFDLGVBQWdCLENBQUMsR0FBRyxFQUFFO2dCQUM1QixXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2FBQy9CO1lBQ0QsSUFBSSxLQUFLLENBQUMsZUFBZ0IsQ0FBQyxHQUFHLEVBQUU7Z0JBQzVCLFdBQVcsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQzthQUMxQztZQUNELE9BQU8sV0FBVyxDQUFDO1NBQ3RCO0lBQ0wsQ0FBQztJQUNPLHVCQUF1QixDQUFDLEtBQW9COztRQUNoRCxNQUFNLG9CQUFvQixTQUFHLEtBQUssQ0FBQyxjQUFjLDBDQUFFLG9CQUFvQixDQUFDO1FBQ3hFLElBQUksb0JBQW9CLEtBQUssU0FBUyxJQUFJLG9CQUFvQixDQUFDLE1BQU0sRUFBRSxHQUFHLGVBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7WUFDbkcsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsb0JBQW9CLENBQUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBQ3hIO1FBQ0QsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE9BQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxTQUFTLG1DQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ3pGLElBQUksU0FBUyxLQUFLLFNBQVMsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLElBQUksU0FBUyxHQUFHLEVBQUUsQ0FBQyxFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsOERBQThELFNBQVMsR0FBRyxDQUFDLENBQUM7U0FDL0Y7UUFDRCxPQUFPLGlCQUFpQixDQUFDO1lBQ3JCLDZCQUE2QixFQUFFLG9CQUFvQixhQUFwQixvQkFBb0IsdUJBQXBCLG9CQUFvQixDQUFFLE1BQU0sQ0FBQyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQztZQUMvRSxhQUFhLEVBQUUsU0FBUztZQUN4QixnQkFBZ0IsUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxnQkFBZ0I7WUFDeEQsZ0JBQWdCLFFBQUUsS0FBSyxDQUFDLGNBQWMsMENBQUUsZ0JBQWdCO1lBQ3hELGNBQWMsUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxhQUFhO1lBQ25ELGNBQWMsUUFBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxjQUFjO1NBQ3ZELENBQUMsQ0FBQztJQUNQLENBQUM7SUFDTyxtQkFBbUIsQ0FBQyxLQUFvQjtRQUM1QyxNQUFNLE1BQU0sR0FBMEMsRUFBRSxDQUFDO1FBQ3pELElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzFCLE1BQU0sYUFBYSxHQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUEwRDtpQkFDbkgsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO2lCQUM1QixHQUFHLENBQUMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFOztnQkFBQyxPQUFBLENBQUM7b0JBQzVCLElBQUksRUFBRSxtQ0FBc0IsQ0FBQyxRQUFRLENBQUM7b0JBQ3RDLE9BQU8sUUFBRSxJQUFJLENBQUMsT0FBTyxtQ0FBSSxJQUFJO29CQUM3QixRQUFRLFFBQUUsSUFBSSxDQUFDLFFBQVEsbUNBQUksS0FBSztpQkFDbkMsQ0FBQyxDQUFBO2FBQUEsQ0FBQyxDQUFDO1lBQ0osTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLGFBQWEsQ0FBQyxDQUFDO1NBQ2pDO1FBQ0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDeEIsTUFBTSxXQUFXLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFBRTs7Z0JBQ3JFLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxnQkFBaUIsQ0FBQyxRQUFRLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztnQkFDNUQsTUFBTSxpQkFBaUIsR0FBbUQ7b0JBQ3RFLFFBQVEsY0FBRSxVQUFVLENBQUMsaUJBQWlCLDBDQUFFLEdBQUcsMENBQUUsUUFBUSxFQUFFO29CQUN2RCxRQUFRLGNBQUUsVUFBVSxDQUFDLGlCQUFpQiwwQ0FBRSxHQUFHLDBDQUFFLFFBQVEsRUFBRTtpQkFDMUQsQ0FBQztnQkFDRixNQUFNLGlCQUFpQixHQUFtRDtvQkFDdEUsU0FBUyxjQUFFLFVBQVUsQ0FBQyxpQkFBaUIsMENBQUUsTUFBTSwwQ0FBRSxRQUFRLEVBQUU7b0JBQzNELFNBQVMsY0FBRSxVQUFVLENBQUMsaUJBQWlCLDBDQUFFLE1BQU0sMENBQUUsUUFBUSxFQUFFO2lCQUM5RCxDQUFDO2dCQUNGLE9BQU87b0JBQ0gsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsaUJBQWlCLEVBQUUsVUFBVSxDQUFDLFFBQVE7b0JBQ3RDLDBCQUEwQixFQUFFLFVBQVUsQ0FBQyxpQkFBaUI7d0JBQ3BELENBQUMsQ0FBQyxpQkFBaUI7d0JBQ25CLENBQUMsQ0FBQyxTQUFTO29CQUNmLDBCQUEwQixFQUFFLFVBQVUsQ0FBQyxpQkFBaUI7d0JBQ3BELENBQUMsQ0FBQyxpQkFBaUI7d0JBQ25CLENBQUMsQ0FBQyxTQUFTO29CQUNmLE9BQU8sRUFBRSxVQUFVLENBQUMsT0FBTztpQkFDOUIsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLFdBQVcsQ0FBQyxDQUFDO1NBQy9CO1FBQ0QsSUFBSSxNQUFNLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNyQixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2xCLENBQUM7SUFDTyxlQUFlLENBQUMsS0FBb0I7O1FBQ3hDLE1BQU0sZUFBZSxTQUFHLEtBQUssQ0FBQyxlQUFlLG1DQUFJLGVBQWUsQ0FBQywyQkFBMkIsQ0FBQztRQUM3RixRQUFRLGVBQWUsRUFBRTtZQUNyQixLQUFLLGVBQWUsQ0FBQywyQkFBMkI7Z0JBQzVDLE9BQU87b0JBQ0gsa0JBQWtCLEVBQUU7d0JBQ2hCLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRSxDQUFDLEVBQUU7d0JBQ3ZDLEVBQUUsSUFBSSxFQUFFLHVCQUF1QixFQUFFLFFBQVEsRUFBRSxDQUFDLEVBQUU7cUJBQ2pEO2lCQUNKLENBQUM7WUFDTixLQUFLLGVBQWUsQ0FBQywyQkFBMkI7Z0JBQzVDLE9BQU87b0JBQ0gsa0JBQWtCLEVBQUU7d0JBQ2hCLEVBQUUsSUFBSSxFQUFFLHVCQUF1QixFQUFFLFFBQVEsRUFBRSxDQUFDLEVBQUU7d0JBQzlDLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRSxDQUFDLEVBQUU7cUJBQzFDO2lCQUNKLENBQUM7WUFDTixLQUFLLGVBQWUsQ0FBQyxVQUFVO2dCQUMzQixPQUFPO29CQUNILGtCQUFrQixFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsUUFBUSxFQUFFLENBQUMsRUFBRSxDQUFDO2lCQUNoRSxDQUFDO1lBQ04sS0FBSyxlQUFlLENBQUMsc0JBQXNCO2dCQUN2QyxPQUFPO29CQUNILGtCQUFrQixFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsdUJBQXVCLEVBQUUsUUFBUSxFQUFFLENBQUMsRUFBRSxDQUFDO2lCQUN2RSxDQUFDO1lBQ04sS0FBSyxlQUFlLENBQUMsSUFBSTtnQkFDckIsT0FBTztvQkFDSCxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUUsQ0FBQyxFQUFFLENBQUM7aUJBQzVELENBQUM7WUFDTixLQUFLLGVBQWUsQ0FBQyxlQUFlO2dCQUNoQyxPQUFPLFNBQVMsQ0FBQztZQUNyQjtnQkFDSSxNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1NBQ2hGO0lBQ0wsQ0FBQztDQUNKO0FBaFhELDRCQWdYQztBQUNELFNBQVMsaUJBQWlCLENBQUMsTUFBYztJQUNyQyxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUMsS0FBSyxTQUFTLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM5RixPQUFPLFlBQVksQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7QUFDN0MsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSb2xlLCBQb2xpY3lEb2N1bWVudCwgUG9saWN5U3RhdGVtZW50LCBSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSBcIi4uLy4uL2F3cy1sYW1iZGFcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENmblVzZXJQb29sIH0gZnJvbSAnLi9jb2duaXRvLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBTdGFuZGFyZEF0dHJpYnV0ZU5hbWVzIH0gZnJvbSAnLi9wcml2YXRlL2F0dHItbmFtZXMnO1xuaW1wb3J0IHsgSUN1c3RvbUF0dHJpYnV0ZSwgU3RhbmRhcmRBdHRyaWJ1dGUsIFN0YW5kYXJkQXR0cmlidXRlcyB9IGZyb20gJy4vdXNlci1wb29sLWF0dHInO1xuaW1wb3J0IHsgVXNlclBvb2xDbGllbnQsIFVzZXJQb29sQ2xpZW50T3B0aW9ucyB9IGZyb20gJy4vdXNlci1wb29sLWNsaWVudCc7XG5pbXBvcnQgeyBVc2VyUG9vbERvbWFpbiwgVXNlclBvb2xEb21haW5PcHRpb25zIH0gZnJvbSAnLi91c2VyLXBvb2wtZG9tYWluJztcbmltcG9ydCB7IElVc2VyUG9vbElkZW50aXR5UHJvdmlkZXIgfSBmcm9tICcuL3VzZXItcG9vbC1pZHAnO1xuLyoqXG4gKiBUaGUgZGlmZmVyZW50IHdheXMgaW4gd2hpY2ggdXNlcnMgb2YgdGhpcyBwb29sIGNhbiBzaWduIHVwIG9yIHNpZ24gaW4uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2lnbkluQWxpYXNlcyB7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB1c2VyIGlzIGFsbG93ZWQgdG8gc2lnbiB1cCBvciBzaWduIGluIHdpdGggYSB1c2VybmFtZVxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSB1c2VybmFtZT86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogV2hldGhlciBhIHVzZXIgaXMgYWxsb3dlZCB0byBzaWduIHVwIG9yIHNpZ24gaW4gd2l0aCBhbiBlbWFpbCBhZGRyZXNzXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBlbWFpbD86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogV2hldGhlciBhIHVzZXIgaXMgYWxsb3dlZCB0byBzaWduIHVwIG9yIHNpZ24gaW4gd2l0aCBhIHBob25lIG51bWJlclxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGhvbmU/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgYSB1c2VyIGlzIGFsbG93ZWQgdG8gaWduIGluIHdpdGggYSBzZWNvbmRhcnkgdXNlcm5hbWUsIHRoYXQgY2FuIGJlIHNldCBhbmQgbW9kaWZpZWQgYWZ0ZXIgc2lnbiB1cC5cbiAgICAgKiBDYW4gb25seSBiZSB1c2VkIGluIGNvbmp1bmN0aW9uIHdpdGggYFVTRVJOQU1FYC5cbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWZlcnJlZFVzZXJuYW1lPzogYm9vbGVhbjtcbn1cbi8qKlxuICogQXR0cmlidXRlcyB0aGF0IGNhbiBiZSBhdXRvbWF0aWNhbGx5IHZlcmlmaWVkIGZvciB1c2VycyBpbiBhIHVzZXIgcG9vbC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBdXRvVmVyaWZpZWRBdHRycyB7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0aGUgZW1haWwgYWRkcmVzcyBvZiB0aGUgdXNlciBzaG91bGQgYmUgYXV0byB2ZXJpZmllZCBhdCBzaWduIHVwLlxuICAgICAqXG4gICAgICogTm90ZTogSWYgYm90aCBgZW1haWxgIGFuZCBgcGhvbmVgIGlzIHNldCwgQ29nbml0byBvbmx5IHZlcmlmaWVzIHRoZSBwaG9uZSBudW1iZXIuIFRvIGFsc28gdmVyaWZ5IGVtYWlsLCBzZWUgaGVyZSAtXG4gICAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1zZXR0aW5ncy1lbWFpbC1waG9uZS12ZXJpZmljYXRpb24uaHRtbFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSB0cnVlLCBpZiBlbWFpbCBpcyB0dXJuZWQgb24gZm9yIGBzaWduSW5gLiBmYWxzZSwgb3RoZXJ3aXNlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtYWlsPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRoZSBwaG9uZSBudW1iZXIgb2YgdGhlIHVzZXIgc2hvdWxkIGJlIGF1dG8gdmVyaWZpZWQgYXQgc2lnbiB1cC5cbiAgICAgKiBAZGVmYXVsdCAtIHRydWUsIGlmIHBob25lIGlzIHR1cm5lZCBvbiBmb3IgYHNpZ25JbmAuIGZhbHNlLCBvdGhlcndpc2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGhvbmU/OiBib29sZWFuO1xufVxuLyoqXG4gKiBUcmlnZ2VycyBmb3IgYSB1c2VyIHBvb2xcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1pZGVudGl0eS1wb29scy13b3JraW5nLXdpdGgtYXdzLWxhbWJkYS10cmlnZ2Vycy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVXNlclBvb2xUcmlnZ2VycyB7XG4gICAgLyoqXG4gICAgICogQ3JlYXRlcyBhbiBhdXRoZW50aWNhdGlvbiBjaGFsbGVuZ2UuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1jcmVhdGUtYXV0aC1jaGFsbGVuZ2UuaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gbm8gdHJpZ2dlciBjb25maWd1cmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgY3JlYXRlQXV0aENoYWxsZW5nZT86IGxhbWJkYS5JRnVuY3Rpb247XG4gICAgLyoqXG4gICAgICogQSBjdXN0b20gTWVzc2FnZSBBV1MgTGFtYmRhIHRyaWdnZXIuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1jdXN0b20tbWVzc2FnZS5odG1sXG4gICAgICogQGRlZmF1bHQgLSBubyB0cmlnZ2VyIGNvbmZpZ3VyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBjdXN0b21NZXNzYWdlPzogbGFtYmRhLklGdW5jdGlvbjtcbiAgICAvKipcbiAgICAgKiBEZWZpbmVzIHRoZSBhdXRoZW50aWNhdGlvbiBjaGFsbGVuZ2UuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1kZWZpbmUtYXV0aC1jaGFsbGVuZ2UuaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gbm8gdHJpZ2dlciBjb25maWd1cmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVmaW5lQXV0aENoYWxsZW5nZT86IGxhbWJkYS5JRnVuY3Rpb247XG4gICAgLyoqXG4gICAgICogQSBwb3N0LWF1dGhlbnRpY2F0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXBvc3QtYXV0aGVudGljYXRpb24uaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gbm8gdHJpZ2dlciBjb25maWd1cmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcG9zdEF1dGhlbnRpY2F0aW9uPzogbGFtYmRhLklGdW5jdGlvbjtcbiAgICAvKipcbiAgICAgKiBBIHBvc3QtY29uZmlybWF0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXBvc3QtY29uZmlybWF0aW9uLmh0bWxcbiAgICAgKiBAZGVmYXVsdCAtIG5vIHRyaWdnZXIgY29uZmlndXJlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBvc3RDb25maXJtYXRpb24/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIEEgcHJlLWF1dGhlbnRpY2F0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXByZS1hdXRoZW50aWNhdGlvbi5odG1sXG4gICAgICogQGRlZmF1bHQgLSBubyB0cmlnZ2VyIGNvbmZpZ3VyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBwcmVBdXRoZW50aWNhdGlvbj86IGxhbWJkYS5JRnVuY3Rpb247XG4gICAgLyoqXG4gICAgICogQSBwcmUtcmVnaXN0cmF0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXByZS1zaWduLXVwLmh0bWxcbiAgICAgKiBAZGVmYXVsdCAtIG5vIHRyaWdnZXIgY29uZmlndXJlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZVNpZ25VcD86IGxhbWJkYS5JRnVuY3Rpb247XG4gICAgLyoqXG4gICAgICogQSBwcmUtdG9rZW4tZ2VuZXJhdGlvbiBBV1MgTGFtYmRhIHRyaWdnZXIuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wcmUtdG9rZW4tZ2VuZXJhdGlvbi5odG1sXG4gICAgICogQGRlZmF1bHQgLSBubyB0cmlnZ2VyIGNvbmZpZ3VyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBwcmVUb2tlbkdlbmVyYXRpb24/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIEEgdXNlci1taWdyYXRpb24gQVdTIExhbWJkYSB0cmlnZ2VyLlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtbWlncmF0ZS11c2VyLmh0bWxcbiAgICAgKiBAZGVmYXVsdCAtIG5vIHRyaWdnZXIgY29uZmlndXJlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJNaWdyYXRpb24/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuICAgIC8qKlxuICAgICAqIFZlcmlmaWVzIHRoZSBhdXRoZW50aWNhdGlvbiBjaGFsbGVuZ2UgcmVzcG9uc2UuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS12ZXJpZnktYXV0aC1jaGFsbGVuZ2UtcmVzcG9uc2UuaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gbm8gdHJpZ2dlciBjb25maWd1cmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyaWZ5QXV0aENoYWxsZW5nZVJlc3BvbnNlPzogbGFtYmRhLklGdW5jdGlvbjtcbiAgICAvKipcbiAgICAgKiBJbmRleCBzaWduYXR1cmVcbiAgICAgKi9cbiAgICBbdHJpZ2dlcjogc3RyaW5nXTogbGFtYmRhLklGdW5jdGlvbiB8IHVuZGVmaW5lZDtcbn1cbi8qKlxuICogVXNlciBwb29sIG9wZXJhdGlvbnMgdG8gd2hpY2ggbGFtYmRhIHRyaWdnZXJzIGNhbiBiZSBhdHRhY2hlZC5cbiAqL1xuZXhwb3J0IGNsYXNzIFVzZXJQb29sT3BlcmF0aW9uIHtcbiAgICAvKipcbiAgICAgKiBDcmVhdGVzIGEgY2hhbGxlbmdlIGluIGEgY3VzdG9tIGF1dGggZmxvd1xuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtY3JlYXRlLWF1dGgtY2hhbGxlbmdlLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENSRUFURV9BVVRIX0NIQUxMRU5HRSA9IG5ldyBVc2VyUG9vbE9wZXJhdGlvbignY3JlYXRlQXV0aENoYWxsZW5nZScpO1xuICAgIC8qKlxuICAgICAqIEFkdmFuY2VkIGN1c3RvbWl6YXRpb24gYW5kIGxvY2FsaXphdGlvbiBvZiBtZXNzYWdlc1xuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtY3VzdG9tLW1lc3NhZ2UuaHRtbFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ1VTVE9NX01FU1NBR0UgPSBuZXcgVXNlclBvb2xPcGVyYXRpb24oJ2N1c3RvbU1lc3NhZ2UnKTtcbiAgICAvKipcbiAgICAgKiBEZXRlcm1pbmVzIHRoZSBuZXh0IGNoYWxsZW5nZSBpbiBhIGN1c3RvbSBhdXRoIGZsb3dcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLWRlZmluZS1hdXRoLWNoYWxsZW5nZS5odG1sXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZJTkVfQVVUSF9DSEFMTEVOR0UgPSBuZXcgVXNlclBvb2xPcGVyYXRpb24oJ2RlZmluZUF1dGhDaGFsbGVuZ2UnKTtcbiAgICAvKipcbiAgICAgKiBFdmVudCBsb2dnaW5nIGZvciBjdXN0b20gYW5hbHl0aWNzXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wb3N0LWF1dGhlbnRpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBPU1RfQVVUSEVOVElDQVRJT04gPSBuZXcgVXNlclBvb2xPcGVyYXRpb24oJ3Bvc3RBdXRoZW50aWNhdGlvbicpO1xuICAgIC8qKlxuICAgICAqIEN1c3RvbSB3ZWxjb21lIG1lc3NhZ2VzIG9yIGV2ZW50IGxvZ2dpbmcgZm9yIGN1c3RvbSBhbmFseXRpY3NcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXBvc3QtY29uZmlybWF0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBPU1RfQ09ORklSTUFUSU9OID0gbmV3IFVzZXJQb29sT3BlcmF0aW9uKCdwb3N0Q29uZmlybWF0aW9uJyk7XG4gICAgLyoqXG4gICAgICogQ3VzdG9tIHZhbGlkYXRpb24gdG8gYWNjZXB0IG9yIGRlbnkgdGhlIHNpZ24taW4gcmVxdWVzdFxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtcHJlLWF1dGhlbnRpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBSRV9BVVRIRU5USUNBVElPTiA9IG5ldyBVc2VyUG9vbE9wZXJhdGlvbigncHJlQXV0aGVudGljYXRpb24nKTtcbiAgICAvKipcbiAgICAgKiBDdXN0b20gdmFsaWRhdGlvbiB0byBhY2NlcHQgb3IgZGVueSB0aGUgc2lnbi11cCByZXF1ZXN0XG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wcmUtc2lnbi11cC5odG1sXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBQUkVfU0lHTl9VUCA9IG5ldyBVc2VyUG9vbE9wZXJhdGlvbigncHJlU2lnblVwJyk7XG4gICAgLyoqXG4gICAgICogQWRkIG9yIHJlbW92ZSBhdHRyaWJ1dGVzIGluIElkIHRva2Vuc1xuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtcHJlLXRva2VuLWdlbmVyYXRpb24uaHRtbFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUFJFX1RPS0VOX0dFTkVSQVRJT04gPSBuZXcgVXNlclBvb2xPcGVyYXRpb24oJ3ByZVRva2VuR2VuZXJhdGlvbicpO1xuICAgIC8qKlxuICAgICAqIE1pZ3JhdGUgYSB1c2VyIGZyb20gYW4gZXhpc3RpbmcgdXNlciBkaXJlY3RvcnkgdG8gdXNlciBwb29sc1xuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtbWlncmF0ZS11c2VyLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVTRVJfTUlHUkFUSU9OID0gbmV3IFVzZXJQb29sT3BlcmF0aW9uKCd1c2VyTWlncmF0aW9uJyk7XG4gICAgLyoqXG4gICAgICogRGV0ZXJtaW5lcyBpZiBhIHJlc3BvbnNlIGlzIGNvcnJlY3QgaW4gYSBjdXN0b20gYXV0aCBmbG93XG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS12ZXJpZnktYXV0aC1jaGFsbGVuZ2UtcmVzcG9uc2UuaHRtbFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSSUZZX0FVVEhfQ0hBTExFTkdFX1JFU1BPTlNFID0gbmV3IFVzZXJQb29sT3BlcmF0aW9uKCd2ZXJpZnlBdXRoQ2hhbGxlbmdlUmVzcG9uc2UnKTtcbiAgICAvKiogQSBjdXN0b20gdXNlciBwb29sIG9wZXJhdGlvbiAqL1xuICAgIHB1YmxpYyBzdGF0aWMgb2YobmFtZTogc3RyaW5nKTogVXNlclBvb2xPcGVyYXRpb24ge1xuICAgICAgICBjb25zdCBsb3dlckNhbWVsQ2FzZSA9IG5hbWUuY2hhckF0KDApLnRvTG93ZXJDYXNlKCkgKyBuYW1lLnNsaWNlKDEpO1xuICAgICAgICByZXR1cm4gbmV3IFVzZXJQb29sT3BlcmF0aW9uKGxvd2VyQ2FtZWxDYXNlKTtcbiAgICB9XG4gICAgLyoqIFRoZSBrZXkgdG8gdXNlIGluIGBDZm5Vc2VyUG9vbC5MYW1iZGFDb25maWdQcm9wZXJ0eWAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgb3BlcmF0aW9uTmFtZTogc3RyaW5nO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3Iob3BlcmF0aW9uTmFtZTogc3RyaW5nKSB7XG4gICAgICAgIHRoaXMub3BlcmF0aW9uTmFtZSA9IG9wZXJhdGlvbk5hbWU7XG4gICAgfVxufVxuLyoqXG4gKiBUaGUgZW1haWwgdmVyaWZpY2F0aW9uIHN0eWxlXG4gKi9cbmV4cG9ydCBlbnVtIFZlcmlmaWNhdGlvbkVtYWlsU3R5bGUge1xuICAgIC8qKiBWZXJpZnkgZW1haWwgdmlhIGNvZGUgKi9cbiAgICBDT0RFID0gJ0NPTkZJUk1fV0lUSF9DT0RFJyxcbiAgICAvKiogVmVyaWZ5IGVtYWlsIHZpYSBsaW5rICovXG4gICAgTElOSyA9ICdDT05GSVJNX1dJVEhfTElOSydcbn1cbi8qKlxuICogVXNlciBwb29sIGNvbmZpZ3VyYXRpb24gZm9yIHVzZXIgc2VsZiBzaWduIHVwLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJWZXJpZmljYXRpb25Db25maWcge1xuICAgIC8qKlxuICAgICAqIFRoZSBlbWFpbCBzdWJqZWN0IHRlbXBsYXRlIGZvciB0aGUgdmVyaWZpY2F0aW9uIGVtYWlsIHNlbnQgdG8gdGhlIHVzZXIgdXBvbiBzaWduIHVwLlxuICAgICAqIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2wtc2V0dGluZ3MtbWVzc2FnZS10ZW1wbGF0ZXMuaHRtbCB0b1xuICAgICAqIGxlYXJuIG1vcmUgYWJvdXQgbWVzc2FnZSB0ZW1wbGF0ZXMuXG4gICAgICogQGRlZmF1bHQgJ1ZlcmlmeSB5b3VyIG5ldyBhY2NvdW50J1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtYWlsU3ViamVjdD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgZW1haWwgYm9keSB0ZW1wbGF0ZSBmb3IgdGhlIHZlcmlmaWNhdGlvbiBlbWFpbCBzZW50IHRvIHRoZSB1c2VyIHVwb24gc2lnbiB1cC5cbiAgICAgKiBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29sLXNldHRpbmdzLW1lc3NhZ2UtdGVtcGxhdGVzLmh0bWwgdG9cbiAgICAgKiBsZWFybiBtb3JlIGFib3V0IG1lc3NhZ2UgdGVtcGxhdGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAnVGhlIHZlcmlmaWNhdGlvbiBjb2RlIHRvIHlvdXIgbmV3IGFjY291bnQgaXMgeyMjIyN9JyBpZiBWZXJpZmljYXRpb25FbWFpbFN0eWxlLkNPREUgaXMgY2hvc2VuLFxuICAgICAqICdWZXJpZnkgeW91ciBhY2NvdW50IGJ5IGNsaWNraW5nIG9uIHsjI1ZlcmlmeSBFbWFpbCMjfScgaWYgVmVyaWZpY2F0aW9uRW1haWxTdHlsZS5MSU5LIGlzIGNob3Nlbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBlbWFpbEJvZHk/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogRW1haWxzIGNhbiBiZSB2ZXJpZmllZCBlaXRoZXIgdXNpbmcgYSBjb2RlIG9yIGEgbGluay5cbiAgICAgKiBMZWFybiBtb3JlIGF0IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbC1zZXR0aW5ncy1lbWFpbC12ZXJpZmljYXRpb24tbWVzc2FnZS1jdXN0b21pemF0aW9uLmh0bWxcbiAgICAgKiBAZGVmYXVsdCBWZXJpZmljYXRpb25FbWFpbFN0eWxlLkNPREVcbiAgICAgKi9cbiAgICByZWFkb25seSBlbWFpbFN0eWxlPzogVmVyaWZpY2F0aW9uRW1haWxTdHlsZTtcbiAgICAvKipcbiAgICAgKiBUaGUgbWVzc2FnZSB0ZW1wbGF0ZSBmb3IgdGhlIHZlcmlmaWNhdGlvbiBTTVMgc2VudCB0byB0aGUgdXNlciB1cG9uIHNpZ24gdXAuXG4gICAgICogU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbC1zZXR0aW5ncy1tZXNzYWdlLXRlbXBsYXRlcy5odG1sIHRvXG4gICAgICogbGVhcm4gbW9yZSBhYm91dCBtZXNzYWdlIHRlbXBsYXRlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gJ1RoZSB2ZXJpZmljYXRpb24gY29kZSB0byB5b3VyIG5ldyBhY2NvdW50IGlzIHsjIyMjfScgaWYgVmVyaWZpY2F0aW9uRW1haWxTdHlsZS5DT0RFIGlzIGNob3NlbixcbiAgICAgKiBub3QgY29uZmlndXJlZCBpZiBWZXJpZmljYXRpb25FbWFpbFN0eWxlLkxJTksgaXMgY2hvc2VuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc21zTWVzc2FnZT86IHN0cmluZztcbn1cbi8qKlxuICogVXNlciBwb29sIGNvbmZpZ3VyYXRpb24gd2hlbiBhZG1pbmlzdHJhdG9ycyBzaWduIHVzZXJzIHVwLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJJbnZpdGF0aW9uQ29uZmlnIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGVtcGxhdGUgdG8gdGhlIGVtYWlsIHN1YmplY3QgdGhhdCBpcyBzZW50IHRvIHRoZSB1c2VyIHdoZW4gYW4gYWRtaW5pc3RyYXRvciBzaWducyB0aGVtIHVwIHRvIHRoZSB1c2VyIHBvb2wuXG4gICAgICogQGRlZmF1bHQgJ1lvdXIgdGVtcG9yYXJ5IHBhc3N3b3JkJ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtYWlsU3ViamVjdD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgdGVtcGxhdGUgdG8gdGhlIGVtYWlsIGJvZHkgdGhhdCBpcyBzZW50IHRvIHRoZSB1c2VyIHdoZW4gYW4gYWRtaW5pc3RyYXRvciBzaWducyB0aGVtIHVwIHRvIHRoZSB1c2VyIHBvb2wuXG4gICAgICogQGRlZmF1bHQgJ1lvdXIgdXNlcm5hbWUgaXMge3VzZXJuYW1lfSBhbmQgdGVtcG9yYXJ5IHBhc3N3b3JkIGlzIHsjIyMjfS4nXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW1haWxCb2R5Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSB0ZW1wbGF0ZSB0byB0aGUgU01TIG1lc3NhZ2UgdGhhdCBpcyBzZW50IHRvIHRoZSB1c2VyIHdoZW4gYW4gYWRtaW5pc3RyYXRvciBzaWducyB0aGVtIHVwIHRvIHRoZSB1c2VyIHBvb2wuXG4gICAgICogQGRlZmF1bHQgJ1lvdXIgdXNlcm5hbWUgaXMge3VzZXJuYW1lfSBhbmQgdGVtcG9yYXJ5IHBhc3N3b3JkIGlzIHsjIyMjfSdcbiAgICAgKi9cbiAgICByZWFkb25seSBzbXNNZXNzYWdlPzogc3RyaW5nO1xufVxuLyoqXG4gKiBUaGUgZGlmZmVyZW50IHdheXMgaW4gd2hpY2ggYSB1c2VyIHBvb2wncyBNRkEgZW5mb3JjZW1lbnQgY2FuIGJlIGNvbmZpZ3VyZWQuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtc2V0dGluZ3MtbWZhLmh0bWxcbiAqL1xuZXhwb3J0IGVudW0gTWZhIHtcbiAgICAvKiogVXNlcnMgYXJlIG5vdCByZXF1aXJlZCB0byB1c2UgTUZBIGZvciBzaWduIGluLCBhbmQgY2Fubm90IGNvbmZpZ3VyZSBvbmUuICovXG4gICAgT0ZGID0gJ09GRicsXG4gICAgLyoqIFVzZXJzIGFyZSBub3QgcmVxdWlyZWQgdG8gdXNlIE1GQSBmb3Igc2lnbiBpbiwgYnV0IGNhbiBjb25maWd1cmUgb25lIGlmIHRoZXkgc28gY2hvb3NlIHRvLiAqL1xuICAgIE9QVElPTkFMID0gJ09QVElPTkFMJyxcbiAgICAvKiogVXNlcnMgYXJlIHJlcXVpcmVkIHRvIGNvbmZpZ3VyZSBhbiBNRkEsIGFuZCBoYXZlIHRvIHVzZSBpdCB0byBzaWduIGluLiAqL1xuICAgIFJFUVVJUkVEID0gJ09OJ1xufVxuLyoqXG4gKiBUaGUgZGlmZmVyZW50IHdheXMgaW4gd2hpY2ggYSB1c2VyIHBvb2wgY2FuIG9idGFpbiB0aGVpciBNRkEgdG9rZW4gZm9yIHNpZ24gaW4uXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtc2V0dGluZ3MtbWZhLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNZmFTZWNvbmRGYWN0b3Ige1xuICAgIC8qKlxuICAgICAqIFRoZSBNRkEgdG9rZW4gaXMgc2VudCB0byB0aGUgdXNlciB2aWEgU01TIHRvIHRoZWlyIHZlcmlmaWVkIHBob25lIG51bWJlcnNcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtc2V0dGluZ3MtbWZhLXNtcy10ZXh0LW1lc3NhZ2UuaHRtbFxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSBzbXM6IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIE1GQSB0b2tlbiBpcyBhIHRpbWUtYmFzZWQgb25lIHRpbWUgcGFzc3dvcmQgdGhhdCBpcyBnZW5lcmF0ZWQgYnkgYSBoYXJkd2FyZSBvciBzb2Z0d2FyZSB0b2tlblxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1zZXR0aW5ncy1tZmEtdG90cC5odG1sXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBvdHA6IGJvb2xlYW47XG59XG4vKipcbiAqIFBhc3N3b3JkIHBvbGljeSBmb3IgVXNlciBQb29scy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQYXNzd29yZFBvbGljeSB7XG4gICAgLyoqXG4gICAgICogVGhlIGxlbmd0aCBvZiB0aW1lIHRoZSB0ZW1wb3JhcnkgcGFzc3dvcmQgZ2VuZXJhdGVkIGJ5IGFuIGFkbWluIGlzIHZhbGlkLlxuICAgICAqIFRoaXMgbXVzdCBiZSBwcm92aWRlZCBhcyB3aG9sZSBkYXlzLCBsaWtlIER1cmF0aW9uLmRheXMoMykgb3IgRHVyYXRpb24uaG91cnMoNDgpLlxuICAgICAqIEZyYWN0aW9uYWwgZGF5cywgc3VjaCBhcyBEdXJhdGlvbi5ob3VycygyMCksIHdpbGwgZ2VuZXJhdGUgYW4gZXJyb3IuXG4gICAgICogQGRlZmF1bHQgRHVyYXRpb24uZGF5cyg3KVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRlbXBQYXNzd29yZFZhbGlkaXR5PzogRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogTWluaW11bSBsZW5ndGggcmVxdWlyZWQgZm9yIGEgdXNlcidzIHBhc3N3b3JkLlxuICAgICAqIEBkZWZhdWx0IDhcbiAgICAgKi9cbiAgICByZWFkb25seSBtaW5MZW5ndGg/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0aGUgdXNlciBpcyByZXF1aXJlZCB0byBoYXZlIGxvd2VyY2FzZSBjaGFyYWN0ZXJzIGluIHRoZWlyIHBhc3N3b3JkLlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSByZXF1aXJlTG93ZXJjYXNlPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRoZSB1c2VyIGlzIHJlcXVpcmVkIHRvIGhhdmUgdXBwZXJjYXNlIGNoYXJhY3RlcnMgaW4gdGhlaXIgcGFzc3dvcmQuXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlcXVpcmVVcHBlcmNhc2U/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIHVzZXIgaXMgcmVxdWlyZWQgdG8gaGF2ZSBkaWdpdHMgaW4gdGhlaXIgcGFzc3dvcmQuXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlcXVpcmVEaWdpdHM/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIHVzZXIgaXMgcmVxdWlyZWQgdG8gaGF2ZSBzeW1ib2xzIGluIHRoZWlyIHBhc3N3b3JkLlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSByZXF1aXJlU3ltYm9scz86IGJvb2xlYW47XG59XG4vKipcbiAqIEVtYWlsIHNldHRpbmdzIGZvciB0aGUgdXNlciBwb29sLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVtYWlsU2V0dGluZ3Mge1xuICAgIC8qKlxuICAgICAqIFRoZSAnZnJvbScgYWRkcmVzcyBvbiB0aGUgZW1haWxzIHJlY2VpdmVkIGJ5IHRoZSB1c2VyLlxuICAgICAqIEBkZWZhdWx0IG5vcmVwbHlAdmVyaWZpY2F0aW9uZW1haWwuY29tXG4gICAgICovXG4gICAgcmVhZG9ubHkgZnJvbT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgJ3JlcGx5VG8nIGFkZHJlc3Mgb24gdGhlIGVtYWlscyByZWNlaXZlZCBieSB0aGUgdXNlciBhcyBkZWZpbmVkIGJ5IElFVEYgUkZDLTUzMjIuXG4gICAgICogV2hlbiBzZXQsIG1vc3QgZW1haWwgY2xpZW50cyByZWNvZ25pemUgdG8gY2hhbmdlICd0bycgbGluZSB0byB0aGlzIGFkZHJlc3Mgd2hlbiBhIHJlcGx5IGlzIGRyYWZ0ZWQuXG4gICAgICogQGRlZmF1bHQgLSBOb3Qgc2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlcGx5VG8/OiBzdHJpbmc7XG59XG4vKipcbiAqIEhvdyB3aWxsIGEgdXNlciBiZSBhYmxlIHRvIHJlY292ZXIgdGhlaXIgYWNjb3VudD9cbiAqXG4gKiBXaGVuIGEgdXNlciBmb3JnZXRzIHRoZWlyIHBhc3N3b3JkLCB0aGV5IGNhbiBoYXZlIGEgY29kZSBzZW50IHRvIHRoZWlyIHZlcmlmaWVkIGVtYWlsIG9yIHZlcmlmaWVkIHBob25lIHRvIHJlY292ZXIgdGhlaXIgYWNjb3VudC5cbiAqIFlvdSBjYW4gY2hvb3NlIHRoZSBwcmVmZXJyZWQgd2F5IHRvIHNlbmQgY29kZXMgYmVsb3cuXG4gKiBXZSByZWNvbW1lbmQgbm90IGFsbG93aW5nIHBob25lIHRvIGJlIHVzZWQgZm9yIGJvdGggcGFzc3dvcmQgcmVzZXRzIGFuZCBtdWx0aS1mYWN0b3IgYXV0aGVudGljYXRpb24gKE1GQSkuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvaG93LXRvLXJlY292ZXItYS11c2VyLWFjY291bnQuaHRtbFxuICovXG5leHBvcnQgZW51bSBBY2NvdW50UmVjb3Zlcnkge1xuICAgIC8qKlxuICAgICAqIEVtYWlsIGlmIGF2YWlsYWJsZSwgb3RoZXJ3aXNlIHBob25lLCBidXQgZG9u4oCZdCBhbGxvdyBhIHVzZXIgdG8gcmVzZXQgdGhlaXIgcGFzc3dvcmQgdmlhIHBob25lIGlmIHRoZXkgYXJlIGFsc28gdXNpbmcgaXQgZm9yIE1GQVxuICAgICAqL1xuICAgIEVNQUlMX0FORF9QSE9ORV9XSVRIT1VUX01GQSxcbiAgICAvKipcbiAgICAgKiBQaG9uZSBpZiBhdmFpbGFibGUsIG90aGVyd2lzZSBlbWFpbCwgYnV0IGRvbuKAmXQgYWxsb3cgYSB1c2VyIHRvIHJlc2V0IHRoZWlyIHBhc3N3b3JkIHZpYSBwaG9uZSBpZiB0aGV5IGFyZSBhbHNvIHVzaW5nIGl0IGZvciBNRkFcbiAgICAgKi9cbiAgICBQSE9ORV9XSVRIT1VUX01GQV9BTkRfRU1BSUwsXG4gICAgLyoqXG4gICAgICogRW1haWwgb25seVxuICAgICAqL1xuICAgIEVNQUlMX09OTFksXG4gICAgLyoqXG4gICAgICogUGhvbmUgb25seSwgYnV0IGRvbuKAmXQgYWxsb3cgYSB1c2VyIHRvIHJlc2V0IHRoZWlyIHBhc3N3b3JkIHZpYSBwaG9uZSBpZiB0aGV5IGFyZSBhbHNvIHVzaW5nIGl0IGZvciBNRkFcbiAgICAgKi9cbiAgICBQSE9ORV9PTkxZX1dJVEhPVVRfTUZBLFxuICAgIC8qKlxuICAgICAqIChOb3QgUmVjb21tZW5kZWQpIFBob25lIGlmIGF2YWlsYWJsZSwgb3RoZXJ3aXNlIGVtYWlsLCBhbmQgZG8gYWxsb3cgYSB1c2VyIHRvIHJlc2V0IHRoZWlyIHBhc3N3b3JkIHZpYSBwaG9uZSBpZiB0aGV5IGFyZSBhbHNvIHVzaW5nIGl0IGZvciBNRkEuXG4gICAgICovXG4gICAgUEhPTkVfQU5EX0VNQUlMLFxuICAgIC8qKlxuICAgICAqIE5vbmUg4oCTIHVzZXJzIHdpbGwgaGF2ZSB0byBjb250YWN0IGFuIGFkbWluaXN0cmF0b3IgdG8gcmVzZXQgdGhlaXIgcGFzc3dvcmRzXG4gICAgICovXG4gICAgTk9ORVxufVxuLyoqXG4gKiBQcm9wcyBmb3IgdGhlIFVzZXJQb29sIGNvbnN0cnVjdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sUHJvcHMge1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhlIHVzZXIgcG9vbFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBhdXRvbWF0aWNhbGx5IGdlbmVyYXRlZCBuYW1lIGJ5IENsb3VkRm9ybWF0aW9uIGF0IGRlcGxveSB0aW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgdXNlclBvb2xOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgc2VsZiBzaWduIHVwIHNob3VsZCBiZSBlbmFibGVkLiBUaGlzIGNhbiBiZSBmdXJ0aGVyIGNvbmZpZ3VyZWQgdmlhIHRoZSBgc2VsZlNpZ25VcGAgcHJvcGVydHkuXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBzZWxmU2lnblVwRW5hYmxlZD86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogQ29uZmlndXJhdGlvbiBhcm91bmQgdXNlcnMgc2lnbmluZyB0aGVtc2VsdmVzIHVwIHRvIHRoZSB1c2VyIHBvb2wuXG4gICAgICogRW5hYmxlIG9yIGRpc2FibGUgc2VsZiBzaWduLXVwIHZpYSB0aGUgYHNlbGZTaWduVXBFbmFibGVkYCBwcm9wZXJ0eS5cbiAgICAgKiBAZGVmYXVsdCAtIHNlZSBkZWZhdWx0cyBpbiBVc2VyVmVyaWZpY2F0aW9uQ29uZmlnXG4gICAgICovXG4gICAgcmVhZG9ubHkgdXNlclZlcmlmaWNhdGlvbj86IFVzZXJWZXJpZmljYXRpb25Db25maWc7XG4gICAgLyoqXG4gICAgICogQ29uZmlndXJhdGlvbiBhcm91bmQgYWRtaW5zIHNpZ25pbmcgdXAgdXNlcnMgaW50byBhIHVzZXIgcG9vbC5cbiAgICAgKiBAZGVmYXVsdCAtIHNlZSBkZWZhdWx0cyBpbiBVc2VySW52aXRhdGlvbkNvbmZpZ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJJbnZpdGF0aW9uPzogVXNlckludml0YXRpb25Db25maWc7XG4gICAgLyoqXG4gICAgICogVGhlIElBTSByb2xlIHRoYXQgQ29nbml0byB3aWxsIGFzc3VtZSB3aGlsZSBzZW5kaW5nIFNNUyBtZXNzYWdlcy5cbiAgICAgKiBAZGVmYXVsdCAtIGEgbmV3IElBTSByb2xlIGlzIGNyZWF0ZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBzbXNSb2xlPzogSVJvbGU7XG4gICAgLyoqXG4gICAgICogVGhlICdFeHRlcm5hbElkJyB0aGF0IENvZ25pdG8gc2VydmljZSBtdXN0IHVzaW5nIHdoZW4gYXNzdW1pbmcgdGhlIGBzbXNSb2xlYCwgaWYgdGhlIHJvbGUgaXMgcmVzdHJpY3RlZCB3aXRoIGFuICdzdHM6RXh0ZXJuYWxJZCcgY29uZGl0aW9uYWwuXG4gICAgICogTGVhcm4gbW9yZSBhYm91dCBFeHRlcm5hbElkIGhlcmUgLSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfY3JlYXRlX2Zvci11c2VyX2V4dGVybmFsaWQuaHRtbFxuICAgICAqXG4gICAgICogVGhpcyBwcm9wZXJ0eSB3aWxsIGJlIGlnbm9yZWQgaWYgYHNtc1JvbGVgIGlzIG5vdCBzcGVjaWZpZWQuXG4gICAgICogQGRlZmF1bHQgLSBObyBleHRlcm5hbCBpZCB3aWxsIGJlIGNvbmZpZ3VyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBzbXNSb2xlRXh0ZXJuYWxJZD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTZXR0aW5nIHRoaXMgd291bGQgZXhwbGljaXRseSBlbmFibGUgb3IgZGlzYWJsZSBTTVMgcm9sZSBjcmVhdGlvbi5cbiAgICAgKiBXaGVuIGxlZnQgdW5zcGVjaWZpZWQsIENESyB3aWxsIGRldGVybWluZSBiYXNlZCBvbiBvdGhlciBwcm9wZXJ0aWVzIGlmIGEgcm9sZSBpcyBuZWVkZWQgb3Igbm90LlxuICAgICAqIEBkZWZhdWx0IC0gQ0RLIHdpbGwgZGV0ZXJtaW5lIGJhc2VkIG9uIG90aGVyIHByb3BlcnRpZXMgb2YgdGhlIHVzZXIgcG9vbCBpZiBhbiBTTVMgcm9sZSBzaG91bGQgYmUgY3JlYXRlZCBvciBub3QuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5hYmxlU21zUm9sZT86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogTWV0aG9kcyBpbiB3aGljaCBhIHVzZXIgcmVnaXN0ZXJzIG9yIHNpZ25zIGluIHRvIGEgdXNlciBwb29sLlxuICAgICAqIEFsbG93cyBlaXRoZXIgdXNlcm5hbWUgd2l0aCBhbGlhc2VzIE9SIHNpZ24gaW4gd2l0aCBlbWFpbCwgcGhvbmUsIG9yIGJvdGguXG4gICAgICpcbiAgICAgKiBSZWFkIHRoZSBzZWN0aW9ucyBvbiB1c2VybmFtZXMgYW5kIGFsaWFzZXMgdG8gbGVhcm4gbW9yZSAtXG4gICAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1zZXR0aW5ncy1hdHRyaWJ1dGVzLmh0bWxcbiAgICAgKlxuICAgICAqIFRvIG1hdGNoIHdpdGggJ09wdGlvbiAxJyBpbiB0aGUgYWJvdmUgbGluaywgd2l0aCBhIHZlcmlmaWVkIGVtYWlsLCB0aGlzIHByb3BlcnR5IHNob3VsZCBiZSBzZXQgdG9cbiAgICAgKiBgeyB1c2VybmFtZTogdHJ1ZSwgZW1haWw6IHRydWUgfWAuIFRvIG1hdGNoIHdpdGggJ09wdGlvbiAyJyBpbiB0aGUgYWJvdmUgbGluayB3aXRoIGJvdGggYSB2ZXJpZmllZCBlbWFpbCBhbmQgcGhvbmVcbiAgICAgKiBudW1iZXIsIHRoaXMgcHJvcGVydHkgc2hvdWxkIGJlIHNldCB0byBgeyBlbWFpbDogdHJ1ZSwgcGhvbmU6IHRydWUgfWAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB7IHVzZXJuYW1lOiB0cnVlIH1cbiAgICAgKi9cbiAgICByZWFkb25seSBzaWduSW5BbGlhc2VzPzogU2lnbkluQWxpYXNlcztcbiAgICAvKipcbiAgICAgKiBBdHRyaWJ1dGVzIHdoaWNoIENvZ25pdG8gd2lsbCBsb29rIHRvIHZlcmlmeSBhdXRvbWF0aWNhbGx5IHVwb24gdXNlciBzaWduIHVwLlxuICAgICAqIEVNQUlMIGFuZCBQSE9ORSBhcmUgdGhlIG9ubHkgYXZhaWxhYmxlIG9wdGlvbnMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIElmIGBzaWduSW5BbGlhc2AgaW5jbHVkZXMgZW1haWwgYW5kL29yIHBob25lLCB0aGV5IHdpbGwgYmUgaW5jbHVkZWQgaW4gYGF1dG9WZXJpZmllZEF0dHJpYnV0ZXNgIGJ5IGRlZmF1bHQuXG4gICAgICogSWYgYWJzZW50LCBubyBhdHRyaWJ1dGVzIHdpbGwgYmUgYXV0by12ZXJpZmllZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBhdXRvVmVyaWZ5PzogQXV0b1ZlcmlmaWVkQXR0cnM7XG4gICAgLyoqXG4gICAgICogVGhlIHNldCBvZiBhdHRyaWJ1dGVzIHRoYXQgYXJlIHJlcXVpcmVkIGZvciBldmVyeSB1c2VyIGluIHRoZSB1c2VyIHBvb2wuXG4gICAgICogUmVhZCBtb3JlIG9uIGF0dHJpYnV0ZXMgaGVyZSAtIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtc2V0dGluZ3MtYXR0cmlidXRlcy5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFsbCBzdGFuZGFyZCBhdHRyaWJ1dGVzIGFyZSBvcHRpb25hbCBhbmQgbXV0YWJsZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBzdGFuZGFyZEF0dHJpYnV0ZXM/OiBTdGFuZGFyZEF0dHJpYnV0ZXM7XG4gICAgLyoqXG4gICAgICogRGVmaW5lIGEgc2V0IG9mIGN1c3RvbSBhdHRyaWJ1dGVzIHRoYXQgY2FuIGJlIGNvbmZpZ3VyZWQgZm9yIGVhY2ggdXNlciBpbiB0aGUgdXNlciBwb29sLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjdXN0b20gYXR0cmlidXRlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjdXN0b21BdHRyaWJ1dGVzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBJQ3VzdG9tQXR0cmlidXRlO1xuICAgIH07XG4gICAgLyoqXG4gICAgICogQ29uZmlndXJlIHdoZXRoZXIgdXNlcnMgb2YgdGhpcyB1c2VyIHBvb2wgY2FuIG9yIGFyZSByZXF1aXJlZCB1c2UgTUZBIHRvIHNpZ24gaW4uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBNZmEuT0ZGXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWZhPzogTWZhO1xuICAgIC8qKlxuICAgICAqIENvbmZpZ3VyZSB0aGUgTUZBIHR5cGVzIHRoYXQgdXNlcnMgY2FuIHVzZSBpbiB0aGlzIHVzZXIgcG9vbC4gSWdub3JlZCBpZiBgbWZhYCBpcyBzZXQgdG8gYE9GRmAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIHsgc21zOiB0cnVlLCBvbmVUaW1lUGFzc3dvcmQ6IGZhbHNlIH0sIGlmIGBtZmFgIGlzIHNldCB0byBgT1BUSU9OQUxgIG9yIGBSRVFVSVJFRGAuXG4gICAgICogeyBzbXM6IGZhbHNlLCBvbmVUaW1lUGFzc3dvcmQ6IGZhbHNlIH0sIG90aGVyd2lzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1mYVNlY29uZEZhY3Rvcj86IE1mYVNlY29uZEZhY3RvcjtcbiAgICAvKipcbiAgICAgKiBQYXNzd29yZCBwb2xpY3kgZm9yIHRoaXMgdXNlciBwb29sLlxuICAgICAqIEBkZWZhdWx0IC0gc2VlIGRlZmF1bHRzIG9uIGVhY2ggcHJvcGVydHkgb2YgUGFzc3dvcmRQb2xpY3kuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGFzc3dvcmRQb2xpY3k/OiBQYXNzd29yZFBvbGljeTtcbiAgICAvKipcbiAgICAgKiBFbWFpbCBzZXR0aW5ncyBmb3IgYSB1c2VyIHBvb2wuXG4gICAgICogQGRlZmF1bHQgLSBzZWUgZGVmYXVsdHMgb24gZWFjaCBwcm9wZXJ0eSBvZiBFbWFpbFNldHRpbmdzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtYWlsU2V0dGluZ3M/OiBFbWFpbFNldHRpbmdzO1xuICAgIC8qKlxuICAgICAqIExhbWJkYSBmdW5jdGlvbnMgdG8gdXNlIGZvciBzdXBwb3J0ZWQgQ29nbml0byB0cmlnZ2Vycy5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItaWRlbnRpdHktcG9vbHMtd29ya2luZy13aXRoLWF3cy1sYW1iZGEtdHJpZ2dlcnMuaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gTm8gTGFtYmRhIHRyaWdnZXJzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGxhbWJkYVRyaWdnZXJzPzogVXNlclBvb2xUcmlnZ2VycztcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHNpZ24taW4gYWxpYXNlcyBzaG91bGQgYmUgZXZhbHVhdGVkIHdpdGggY2FzZSBzZW5zaXRpdml0eS5cbiAgICAgKiBGb3IgZXhhbXBsZSwgd2hlbiB0aGlzIG9wdGlvbiBpcyBzZXQgdG8gZmFsc2UsIHVzZXJzIHdpbGwgYmUgYWJsZSB0byBzaWduIGluIHVzaW5nIGVpdGhlciBgTXlVc2VybmFtZWAgb3IgYG15dXNlcm5hbWVgLlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSBzaWduSW5DYXNlU2Vuc2l0aXZlPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBIb3cgd2lsbCBhIHVzZXIgYmUgYWJsZSB0byByZWNvdmVyIHRoZWlyIGFjY291bnQ/XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBBY2NvdW50UmVjb3ZlcnkuUEhPTkVfV0lUSE9VVF9NRkFfQU5EX0VNQUlMXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWNjb3VudFJlY292ZXJ5PzogQWNjb3VudFJlY292ZXJ5O1xufVxuLyoqXG4gKiBSZXByZXNlbnRzIGEgQ29nbml0byBVc2VyUG9vbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElVc2VyUG9vbCBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIHBoeXNpY2FsIElEIG9mIHRoaXMgdXNlciBwb29sIHJlc291cmNlXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJQb29sSWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgQVJOIG9mIHRoaXMgdXNlciBwb29sIHJlc291cmNlXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJQb29sQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogR2V0IGFsbCBpZGVudGl0eSBwcm92aWRlcnMgcmVnaXN0ZXJlZCB3aXRoIHRoaXMgdXNlciBwb29sLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlkZW50aXR5UHJvdmlkZXJzOiBJVXNlclBvb2xJZGVudGl0eVByb3ZpZGVyW107XG4gICAgLyoqXG4gICAgICogQWRkIGEgbmV3IGFwcCBjbGllbnQgdG8gdGhpcyB1c2VyIHBvb2wuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLXNldHRpbmdzLWNsaWVudC1hcHBzLmh0bWxcbiAgICAgKi9cbiAgICBhZGRDbGllbnQoaWQ6IHN0cmluZywgb3B0aW9ucz86IFVzZXJQb29sQ2xpZW50T3B0aW9ucyk6IFVzZXJQb29sQ2xpZW50O1xuICAgIC8qKlxuICAgICAqIEFzc29jaWF0ZSBhIGRvbWFpbiB0byB0aGlzIHVzZXIgcG9vbC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtYXNzaWduLWRvbWFpbi5odG1sXG4gICAgICovXG4gICAgYWRkRG9tYWluKGlkOiBzdHJpbmcsIG9wdGlvbnM6IFVzZXJQb29sRG9tYWluT3B0aW9ucyk6IFVzZXJQb29sRG9tYWluO1xuICAgIC8qKlxuICAgICAqIFJlZ2lzdGVyIGFuIGlkZW50aXR5IHByb3ZpZGVyIHdpdGggdGhpcyB1c2VyIHBvb2wuXG4gICAgICovXG4gICAgcmVnaXN0ZXJJZGVudGl0eVByb3ZpZGVyKHByb3ZpZGVyOiBJVXNlclBvb2xJZGVudGl0eVByb3ZpZGVyKTogdm9pZDtcbn1cbmFic3RyYWN0IGNsYXNzIFVzZXJQb29sQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVVzZXJQb29sIHtcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdXNlclBvb2xJZDogc3RyaW5nO1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB1c2VyUG9vbEFybjogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBpZGVudGl0eVByb3ZpZGVyczogSVVzZXJQb29sSWRlbnRpdHlQcm92aWRlcltdID0gW107XG4gICAgcHVibGljIGFkZENsaWVudChpZDogc3RyaW5nLCBvcHRpb25zPzogVXNlclBvb2xDbGllbnRPcHRpb25zKTogVXNlclBvb2xDbGllbnQge1xuICAgICAgICByZXR1cm4gbmV3IFVzZXJQb29sQ2xpZW50KHRoaXMsIGlkLCB7XG4gICAgICAgICAgICB1c2VyUG9vbDogdGhpcyxcbiAgICAgICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkRG9tYWluKGlkOiBzdHJpbmcsIG9wdGlvbnM6IFVzZXJQb29sRG9tYWluT3B0aW9ucyk6IFVzZXJQb29sRG9tYWluIHtcbiAgICAgICAgcmV0dXJuIG5ldyBVc2VyUG9vbERvbWFpbih0aGlzLCBpZCwge1xuICAgICAgICAgICAgdXNlclBvb2w6IHRoaXMsXG4gICAgICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcHVibGljIHJlZ2lzdGVySWRlbnRpdHlQcm92aWRlcihwcm92aWRlcjogSVVzZXJQb29sSWRlbnRpdHlQcm92aWRlcikge1xuICAgICAgICB0aGlzLmlkZW50aXR5UHJvdmlkZXJzLnB1c2gocHJvdmlkZXIpO1xuICAgIH1cbn1cbi8qKlxuICogRGVmaW5lIGEgQ29nbml0byBVc2VyIFBvb2xcbiAqL1xuZXhwb3J0IGNsYXNzIFVzZXJQb29sIGV4dGVuZHMgVXNlclBvb2xCYXNlIHtcbiAgICAvKipcbiAgICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgdXNlciBwb29sIGJhc2VkIG9uIGl0cyBpZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21Vc2VyUG9vbElkKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHVzZXJQb29sSWQ6IHN0cmluZyk6IElVc2VyUG9vbCB7XG4gICAgICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFVzZXJQb29sQmFzZSB7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xJZCA9IHVzZXJQb29sSWQ7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xBcm4gPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgICAgIHNlcnZpY2U6ICdjb2duaXRvLWlkcCcsXG4gICAgICAgICAgICAgICAgcmVzb3VyY2U6ICd1c2VycG9vbCcsXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiB1c2VyUG9vbElkLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogSW1wb3J0IGFuIGV4aXN0aW5nIHVzZXIgcG9vbCBiYXNlZCBvbiBpdHMgQVJOLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVVzZXJQb29sQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHVzZXJQb29sQXJuOiBzdHJpbmcpOiBJVXNlclBvb2wge1xuICAgICAgICByZXR1cm4gVXNlclBvb2wuZnJvbVVzZXJQb29sSWQoc2NvcGUsIGlkLCBTdGFjay5vZihzY29wZSkucGFyc2VBcm4odXNlclBvb2xBcm4pLnJlc291cmNlTmFtZSEpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgcGh5c2ljYWwgSUQgb2YgdGhpcyB1c2VyIHBvb2wgcmVzb3VyY2VcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xJZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBBUk4gb2YgdGhlIHVzZXIgcG9vbFxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbEFybjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFVzZXIgcG9vbCBwcm92aWRlciBuYW1lXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbFByb3ZpZGVyTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFVzZXIgcG9vbCBwcm92aWRlciBVUkxcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sUHJvdmlkZXJVcmw6IHN0cmluZztcbiAgICBwcml2YXRlIHRyaWdnZXJzOiBDZm5Vc2VyUG9vbC5MYW1iZGFDb25maWdQcm9wZXJ0eSA9IHt9O1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVc2VyUG9vbFByb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgY29uc3Qgc2lnbkluID0gdGhpcy5zaWduSW5Db25maWd1cmF0aW9uKHByb3BzKTtcbiAgICAgICAgaWYgKHByb3BzLmxhbWJkYVRyaWdnZXJzKSB7XG4gICAgICAgICAgICBmb3IgKGNvbnN0IHQgb2YgT2JqZWN0LmtleXMocHJvcHMubGFtYmRhVHJpZ2dlcnMpKSB7XG4gICAgICAgICAgICAgICAgY29uc3QgdHJpZ2dlciA9IHByb3BzLmxhbWJkYVRyaWdnZXJzW3RdO1xuICAgICAgICAgICAgICAgIGlmICh0cmlnZ2VyICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgICAgICAgICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKHRyaWdnZXIgYXMgbGFtYmRhLklGdW5jdGlvbiwgdCk7XG4gICAgICAgICAgICAgICAgICAgICh0aGlzLnRyaWdnZXJzIGFzIGFueSlbdF0gPSAodHJpZ2dlciBhcyBsYW1iZGEuSUZ1bmN0aW9uKS5mdW5jdGlvbkFybjtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgdmVyaWZpY2F0aW9uTWVzc2FnZVRlbXBsYXRlID0gdGhpcy52ZXJpZmljYXRpb25NZXNzYWdlQ29uZmlndXJhdGlvbihwcm9wcyk7XG4gICAgICAgIGxldCBlbWFpbFZlcmlmaWNhdGlvbk1lc3NhZ2U7XG4gICAgICAgIGxldCBlbWFpbFZlcmlmaWNhdGlvblN1YmplY3Q7XG4gICAgICAgIGlmICh2ZXJpZmljYXRpb25NZXNzYWdlVGVtcGxhdGUuZGVmYXVsdEVtYWlsT3B0aW9uID09PSBWZXJpZmljYXRpb25FbWFpbFN0eWxlLkNPREUpIHtcbiAgICAgICAgICAgIGVtYWlsVmVyaWZpY2F0aW9uTWVzc2FnZSA9IHZlcmlmaWNhdGlvbk1lc3NhZ2VUZW1wbGF0ZS5lbWFpbE1lc3NhZ2U7XG4gICAgICAgICAgICBlbWFpbFZlcmlmaWNhdGlvblN1YmplY3QgPSB2ZXJpZmljYXRpb25NZXNzYWdlVGVtcGxhdGUuZW1haWxTdWJqZWN0O1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNtc1ZlcmlmaWNhdGlvbk1lc3NhZ2UgPSB2ZXJpZmljYXRpb25NZXNzYWdlVGVtcGxhdGUuc21zTWVzc2FnZTtcbiAgICAgICAgY29uc3QgaW52aXRlTWVzc2FnZVRlbXBsYXRlOiBDZm5Vc2VyUG9vbC5JbnZpdGVNZXNzYWdlVGVtcGxhdGVQcm9wZXJ0eSA9IHtcbiAgICAgICAgICAgIGVtYWlsTWVzc2FnZTogcHJvcHMudXNlckludml0YXRpb24/LmVtYWlsQm9keSxcbiAgICAgICAgICAgIGVtYWlsU3ViamVjdDogcHJvcHMudXNlckludml0YXRpb24/LmVtYWlsU3ViamVjdCxcbiAgICAgICAgICAgIHNtc01lc3NhZ2U6IHByb3BzLnVzZXJJbnZpdGF0aW9uPy5zbXNNZXNzYWdlLFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBzZWxmU2lnblVwRW5hYmxlZCA9IHByb3BzLnNlbGZTaWduVXBFbmFibGVkICE9PSB1bmRlZmluZWQgPyBwcm9wcy5zZWxmU2lnblVwRW5hYmxlZCA6IGZhbHNlO1xuICAgICAgICBjb25zdCBhZG1pbkNyZWF0ZVVzZXJDb25maWc6IENmblVzZXJQb29sLkFkbWluQ3JlYXRlVXNlckNvbmZpZ1Byb3BlcnR5ID0ge1xuICAgICAgICAgICAgYWxsb3dBZG1pbkNyZWF0ZVVzZXJPbmx5OiAhc2VsZlNpZ25VcEVuYWJsZWQsXG4gICAgICAgICAgICBpbnZpdGVNZXNzYWdlVGVtcGxhdGU6IHByb3BzLnVzZXJJbnZpdGF0aW9uICE9PSB1bmRlZmluZWQgPyBpbnZpdGVNZXNzYWdlVGVtcGxhdGUgOiB1bmRlZmluZWQsXG4gICAgICAgIH07XG4gICAgICAgIGNvbnN0IHBhc3N3b3JkUG9saWN5ID0gdGhpcy5jb25maWd1cmVQYXNzd29yZFBvbGljeShwcm9wcyk7XG4gICAgICAgIGNvbnN0IHVzZXJQb29sID0gbmV3IENmblVzZXJQb29sKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIHVzZXJQb29sTmFtZTogcHJvcHMudXNlclBvb2xOYW1lLFxuICAgICAgICAgICAgdXNlcm5hbWVBdHRyaWJ1dGVzOiBzaWduSW4udXNlcm5hbWVBdHRycyxcbiAgICAgICAgICAgIGFsaWFzQXR0cmlidXRlczogc2lnbkluLmFsaWFzQXR0cnMsXG4gICAgICAgICAgICBhdXRvVmVyaWZpZWRBdHRyaWJ1dGVzOiBzaWduSW4uYXV0b1ZlcmlmeUF0dHJzLFxuICAgICAgICAgICAgbGFtYmRhQ29uZmlnOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdW5kZWZpbmVkSWZOb0tleXModGhpcy50cmlnZ2VycykgfSksXG4gICAgICAgICAgICBzbXNDb25maWd1cmF0aW9uOiB0aGlzLnNtc0NvbmZpZ3VyYXRpb24ocHJvcHMpLFxuICAgICAgICAgICAgYWRtaW5DcmVhdGVVc2VyQ29uZmlnLFxuICAgICAgICAgICAgZW1haWxWZXJpZmljYXRpb25NZXNzYWdlLFxuICAgICAgICAgICAgZW1haWxWZXJpZmljYXRpb25TdWJqZWN0LFxuICAgICAgICAgICAgc21zVmVyaWZpY2F0aW9uTWVzc2FnZSxcbiAgICAgICAgICAgIHZlcmlmaWNhdGlvbk1lc3NhZ2VUZW1wbGF0ZSxcbiAgICAgICAgICAgIHNjaGVtYTogdGhpcy5zY2hlbWFDb25maWd1cmF0aW9uKHByb3BzKSxcbiAgICAgICAgICAgIG1mYUNvbmZpZ3VyYXRpb246IHByb3BzLm1mYSxcbiAgICAgICAgICAgIGVuYWJsZWRNZmFzOiB0aGlzLm1mYUNvbmZpZ3VyYXRpb24ocHJvcHMpLFxuICAgICAgICAgICAgcG9saWNpZXM6IHBhc3N3b3JkUG9saWN5ICE9PSB1bmRlZmluZWQgPyB7IHBhc3N3b3JkUG9saWN5IH0gOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBlbWFpbENvbmZpZ3VyYXRpb246IHVuZGVmaW5lZElmTm9LZXlzKHtcbiAgICAgICAgICAgICAgICBmcm9tOiBwcm9wcy5lbWFpbFNldHRpbmdzPy5mcm9tLFxuICAgICAgICAgICAgICAgIHJlcGx5VG9FbWFpbEFkZHJlc3M6IHByb3BzLmVtYWlsU2V0dGluZ3M/LnJlcGx5VG8sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIHVzZXJuYW1lQ29uZmlndXJhdGlvbjogdW5kZWZpbmVkSWZOb0tleXMoe1xuICAgICAgICAgICAgICAgIGNhc2VTZW5zaXRpdmU6IHByb3BzLnNpZ25JbkNhc2VTZW5zaXRpdmUsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIGFjY291bnRSZWNvdmVyeVNldHRpbmc6IHRoaXMuYWNjb3VudFJlY292ZXJ5KHByb3BzKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMudXNlclBvb2xJZCA9IHVzZXJQb29sLnJlZjtcbiAgICAgICAgdGhpcy51c2VyUG9vbEFybiA9IHVzZXJQb29sLmF0dHJBcm47XG4gICAgICAgIHRoaXMudXNlclBvb2xQcm92aWRlck5hbWUgPSB1c2VyUG9vbC5hdHRyUHJvdmlkZXJOYW1lO1xuICAgICAgICB0aGlzLnVzZXJQb29sUHJvdmlkZXJVcmwgPSB1c2VyUG9vbC5hdHRyUHJvdmlkZXJVcmw7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGxhbWJkYSB0cmlnZ2VyIHRvIGEgdXNlciBwb29sIG9wZXJhdGlvblxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1pZGVudGl0eS1wb29scy13b3JraW5nLXdpdGgtYXdzLWxhbWJkYS10cmlnZ2Vycy5odG1sXG4gICAgICovXG4gICAgcHVibGljIGFkZFRyaWdnZXIob3BlcmF0aW9uOiBVc2VyUG9vbE9wZXJhdGlvbiwgZm46IGxhbWJkYS5JRnVuY3Rpb24pOiB2b2lkIHtcbiAgICAgICAgaWYgKG9wZXJhdGlvbi5vcGVyYXRpb25OYW1lIGluIHRoaXMudHJpZ2dlcnMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQSB0cmlnZ2VyIGZvciB0aGUgb3BlcmF0aW9uICR7b3BlcmF0aW9ufSBhbHJlYWR5IGV4aXN0cy5gKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmFkZExhbWJkYVBlcm1pc3Npb24oZm4sIG9wZXJhdGlvbi5vcGVyYXRpb25OYW1lKTtcbiAgICAgICAgKHRoaXMudHJpZ2dlcnMgYXMgYW55KVtvcGVyYXRpb24ub3BlcmF0aW9uTmFtZV0gPSBmbi5mdW5jdGlvbkFybjtcbiAgICB9XG4gICAgcHJpdmF0ZSBhZGRMYW1iZGFQZXJtaXNzaW9uKGZuOiBsYW1iZGEuSUZ1bmN0aW9uLCBuYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICAgICAgY29uc3QgY2FwaXRhbGl6ZSA9IG5hbWUuY2hhckF0KDApLnRvVXBwZXJDYXNlKCkgKyBuYW1lLnNsaWNlKDEpO1xuICAgICAgICBmbi5hZGRQZXJtaXNzaW9uKGAke2NhcGl0YWxpemV9Q29nbml0b2AsIHtcbiAgICAgICAgICAgIHByaW5jaXBhbDogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2NvZ25pdG8taWRwLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgIHNvdXJjZUFybjogdGhpcy51c2VyUG9vbEFybixcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHByaXZhdGUgdmVyaWZpY2F0aW9uTWVzc2FnZUNvbmZpZ3VyYXRpb24ocHJvcHM6IFVzZXJQb29sUHJvcHMpOiBDZm5Vc2VyUG9vbC5WZXJpZmljYXRpb25NZXNzYWdlVGVtcGxhdGVQcm9wZXJ0eSB7XG4gICAgICAgIGNvbnN0IENPREVfVEVNUExBVEUgPSAneyMjIyN9JztcbiAgICAgICAgY29uc3QgVkVSSUZZX0VNQUlMX1RFTVBMQVRFID0gJ3sjI1ZlcmlmeSBFbWFpbCMjfSc7XG4gICAgICAgIGNvbnN0IGVtYWlsU3R5bGUgPSBwcm9wcy51c2VyVmVyaWZpY2F0aW9uPy5lbWFpbFN0eWxlID8/IFZlcmlmaWNhdGlvbkVtYWlsU3R5bGUuQ09ERTtcbiAgICAgICAgY29uc3QgZW1haWxTdWJqZWN0ID0gcHJvcHMudXNlclZlcmlmaWNhdGlvbj8uZW1haWxTdWJqZWN0ID8/ICdWZXJpZnkgeW91ciBuZXcgYWNjb3VudCc7XG4gICAgICAgIGNvbnN0IHNtc01lc3NhZ2UgPSBwcm9wcy51c2VyVmVyaWZpY2F0aW9uPy5zbXNNZXNzYWdlID8/IGBUaGUgdmVyaWZpY2F0aW9uIGNvZGUgdG8geW91ciBuZXcgYWNjb3VudCBpcyAke0NPREVfVEVNUExBVEV9YDtcbiAgICAgICAgaWYgKGVtYWlsU3R5bGUgPT09IFZlcmlmaWNhdGlvbkVtYWlsU3R5bGUuQ09ERSkge1xuICAgICAgICAgICAgY29uc3QgZW1haWxNZXNzYWdlID0gcHJvcHMudXNlclZlcmlmaWNhdGlvbj8uZW1haWxCb2R5ID8/IGBUaGUgdmVyaWZpY2F0aW9uIGNvZGUgdG8geW91ciBuZXcgYWNjb3VudCBpcyAke0NPREVfVEVNUExBVEV9YDtcbiAgICAgICAgICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGVtYWlsTWVzc2FnZSkgJiYgZW1haWxNZXNzYWdlLmluZGV4T2YoQ09ERV9URU1QTEFURSkgPCAwKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBWZXJpZmljYXRpb24gZW1haWwgYm9keSBtdXN0IGNvbnRhaW4gdGhlIHRlbXBsYXRlIHN0cmluZyAnJHtDT0RFX1RFTVBMQVRFfSdgKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKHNtc01lc3NhZ2UpICYmIHNtc01lc3NhZ2UuaW5kZXhPZihDT0RFX1RFTVBMQVRFKSA8IDApIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFNNUyBtZXNzYWdlIG11c3QgY29udGFpbiB0aGUgdGVtcGxhdGUgc3RyaW5nICcke0NPREVfVEVNUExBVEV9J2ApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICBkZWZhdWx0RW1haWxPcHRpb246IFZlcmlmaWNhdGlvbkVtYWlsU3R5bGUuQ09ERSxcbiAgICAgICAgICAgICAgICBlbWFpbE1lc3NhZ2UsXG4gICAgICAgICAgICAgICAgZW1haWxTdWJqZWN0LFxuICAgICAgICAgICAgICAgIHNtc01lc3NhZ2UsXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgY29uc3QgZW1haWxNZXNzYWdlID0gcHJvcHMudXNlclZlcmlmaWNhdGlvbj8uZW1haWxCb2R5ID8/XG4gICAgICAgICAgICAgICAgYFZlcmlmeSB5b3VyIGFjY291bnQgYnkgY2xpY2tpbmcgb24gJHtWRVJJRllfRU1BSUxfVEVNUExBVEV9YDtcbiAgICAgICAgICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGVtYWlsTWVzc2FnZSkgJiYgZW1haWxNZXNzYWdlLmluZGV4T2YoVkVSSUZZX0VNQUlMX1RFTVBMQVRFKSA8IDApIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFZlcmlmaWNhdGlvbiBlbWFpbCBib2R5IG11c3QgY29udGFpbiB0aGUgdGVtcGxhdGUgc3RyaW5nICcke1ZFUklGWV9FTUFJTF9URU1QTEFURX0nYCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgIGRlZmF1bHRFbWFpbE9wdGlvbjogVmVyaWZpY2F0aW9uRW1haWxTdHlsZS5MSU5LLFxuICAgICAgICAgICAgICAgIGVtYWlsTWVzc2FnZUJ5TGluazogZW1haWxNZXNzYWdlLFxuICAgICAgICAgICAgICAgIGVtYWlsU3ViamVjdEJ5TGluazogZW1haWxTdWJqZWN0LFxuICAgICAgICAgICAgICAgIHNtc01lc3NhZ2UsXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgfVxuICAgIHByaXZhdGUgc2lnbkluQ29uZmlndXJhdGlvbihwcm9wczogVXNlclBvb2xQcm9wcykge1xuICAgICAgICBsZXQgYWxpYXNBdHRyczogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgICAgIGxldCB1c2VybmFtZUF0dHJzOiBzdHJpbmdbXSB8IHVuZGVmaW5lZDtcbiAgICAgICAgbGV0IGF1dG9WZXJpZnlBdHRyczogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgICAgIGNvbnN0IHNpZ25JbjogU2lnbkluQWxpYXNlcyA9IHByb3BzLnNpZ25JbkFsaWFzZXMgPz8geyB1c2VybmFtZTogdHJ1ZSB9O1xuICAgICAgICBpZiAoc2lnbkluLnByZWZlcnJlZFVzZXJuYW1lICYmICFzaWduSW4udXNlcm5hbWUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcigndXNlcm5hbWUgc2lnbkluIG11c3QgYmUgZW5hYmxlZCBpZiBwcmVmZXJyZWRVc2VybmFtZSBpcyBlbmFibGVkJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHNpZ25Jbi51c2VybmFtZSkge1xuICAgICAgICAgICAgYWxpYXNBdHRycyA9IFtdO1xuICAgICAgICAgICAgaWYgKHNpZ25Jbi5lbWFpbCkge1xuICAgICAgICAgICAgICAgIGFsaWFzQXR0cnMucHVzaChTdGFuZGFyZEF0dHJpYnV0ZU5hbWVzLmVtYWlsKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChzaWduSW4ucGhvbmUpIHtcbiAgICAgICAgICAgICAgICBhbGlhc0F0dHJzLnB1c2goU3RhbmRhcmRBdHRyaWJ1dGVOYW1lcy5waG9uZU51bWJlcik7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAoc2lnbkluLnByZWZlcnJlZFVzZXJuYW1lKSB7XG4gICAgICAgICAgICAgICAgYWxpYXNBdHRycy5wdXNoKFN0YW5kYXJkQXR0cmlidXRlTmFtZXMucHJlZmVycmVkVXNlcm5hbWUpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKGFsaWFzQXR0cnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAgICAgYWxpYXNBdHRycyA9IHVuZGVmaW5lZDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHVzZXJuYW1lQXR0cnMgPSBbXTtcbiAgICAgICAgICAgIGlmIChzaWduSW4uZW1haWwpIHtcbiAgICAgICAgICAgICAgICB1c2VybmFtZUF0dHJzLnB1c2goU3RhbmRhcmRBdHRyaWJ1dGVOYW1lcy5lbWFpbCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAoc2lnbkluLnBob25lKSB7XG4gICAgICAgICAgICAgICAgdXNlcm5hbWVBdHRycy5wdXNoKFN0YW5kYXJkQXR0cmlidXRlTmFtZXMucGhvbmVOdW1iZXIpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5hdXRvVmVyaWZ5KSB7XG4gICAgICAgICAgICBhdXRvVmVyaWZ5QXR0cnMgPSBbXTtcbiAgICAgICAgICAgIGlmIChwcm9wcy5hdXRvVmVyaWZ5LmVtYWlsKSB7XG4gICAgICAgICAgICAgICAgYXV0b1ZlcmlmeUF0dHJzLnB1c2goU3RhbmRhcmRBdHRyaWJ1dGVOYW1lcy5lbWFpbCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAocHJvcHMuYXV0b1ZlcmlmeS5waG9uZSkge1xuICAgICAgICAgICAgICAgIGF1dG9WZXJpZnlBdHRycy5wdXNoKFN0YW5kYXJkQXR0cmlidXRlTmFtZXMucGhvbmVOdW1iZXIpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKHNpZ25Jbi5lbWFpbCB8fCBzaWduSW4ucGhvbmUpIHtcbiAgICAgICAgICAgIGF1dG9WZXJpZnlBdHRycyA9IFtdO1xuICAgICAgICAgICAgaWYgKHNpZ25Jbi5lbWFpbCkge1xuICAgICAgICAgICAgICAgIGF1dG9WZXJpZnlBdHRycy5wdXNoKFN0YW5kYXJkQXR0cmlidXRlTmFtZXMuZW1haWwpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKHNpZ25Jbi5waG9uZSkge1xuICAgICAgICAgICAgICAgIGF1dG9WZXJpZnlBdHRycy5wdXNoKFN0YW5kYXJkQXR0cmlidXRlTmFtZXMucGhvbmVOdW1iZXIpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7IHVzZXJuYW1lQXR0cnMsIGFsaWFzQXR0cnMsIGF1dG9WZXJpZnlBdHRycyB9O1xuICAgIH1cbiAgICBwcml2YXRlIHNtc0NvbmZpZ3VyYXRpb24ocHJvcHM6IFVzZXJQb29sUHJvcHMpOiBDZm5Vc2VyUG9vbC5TbXNDb25maWd1cmF0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAocHJvcHMuZW5hYmxlU21zUm9sZSA9PT0gZmFsc2UgJiYgcHJvcHMuc21zUm9sZSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdlbmFibGVTbXNSb2xlIGNhbm5vdCBiZSBkaXNhYmxlZCB3aGVuIHNtc1JvbGUgaXMgc3BlY2lmaWVkJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLnNtc1JvbGUpIHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgc25zQ2FsbGVyQXJuOiBwcm9wcy5zbXNSb2xlLnJvbGVBcm4sXG4gICAgICAgICAgICAgICAgZXh0ZXJuYWxJZDogcHJvcHMuc21zUm9sZUV4dGVybmFsSWQsXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5lbmFibGVTbXNSb2xlID09PSBmYWxzZSkge1xuICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBtZmFDb25maWd1cmF0aW9uID0gdGhpcy5tZmFDb25maWd1cmF0aW9uKHByb3BzKTtcbiAgICAgICAgY29uc3QgcGhvbmVWZXJpZmljYXRpb24gPSBwcm9wcy5zaWduSW5BbGlhc2VzPy5waG9uZSA9PT0gdHJ1ZSB8fCBwcm9wcy5hdXRvVmVyaWZ5Py5waG9uZSA9PT0gdHJ1ZTtcbiAgICAgICAgY29uc3Qgcm9sZVJlcXVpcmVkID0gbWZhQ29uZmlndXJhdGlvbj8uaW5jbHVkZXMoJ1NNU19NRkEnKSB8fCBwaG9uZVZlcmlmaWNhdGlvbjtcbiAgICAgICAgaWYgKCFyb2xlUmVxdWlyZWQgJiYgcHJvcHMuZW5hYmxlU21zUm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNtc1JvbGVFeHRlcm5hbElkID0gdGhpcy5ub2RlLnVuaXF1ZUlkLnN1YnN0cigwLCAxMjIzKTsgLy8gc3RzOkV4dGVybmFsSWQgbWF4IGxlbmd0aCBvZiAxMjI0XG4gICAgICAgIGNvbnN0IHNtc1JvbGUgPSBwcm9wcy5zbXNSb2xlID8/IG5ldyBSb2xlKHRoaXMsICdzbXNSb2xlJywge1xuICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnY29nbml0by1pZHAuYW1hem9uYXdzLmNvbScsIHtcbiAgICAgICAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsczogeyAnc3RzOkV4dGVybmFsSWQnOiBzbXNSb2xlRXh0ZXJuYWxJZCB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgICAgICAgICAgLypcbiAgICAgICAgICAgICAgICAgICogVGhlIFVzZXJQb29sIGlzIHZlcnkgcGFydGljdWxhciB0aGF0IGl0IG11c3QgY29udGFpbiBhbiAnc25zOlB1Ymxpc2gnIGFjdGlvbiBhcyBhbiBpbmxpbmUgcG9saWN5LlxuICAgICAgICAgICAgICAgICAgKiBJZGVhbGx5LCBhIGNvbmRpdGlvbmFsIHRoYXQgcmVzdHJpY3RzIHRoaXMgYWN0aW9uIHRvICdzbXMnIHByb3RvY29sIG5lZWRzIHRvIGJlIGF0dGFjaGVkLCBidXQgdGhlIFVzZXJQb29sIGRlcGxveW1lbnQgZmFpbHMgdmFsaWRhdGlvbi5cbiAgICAgICAgICAgICAgICAgICogU2VlbXMgbGlrZSBhIGNhc2Ugb2YgYmVpbmcgZXhjZXNzaXZlbHkgc3RyaWN0LlxuICAgICAgICAgICAgICAgICAgKi9cbiAgICAgICAgICAgICAgICAnc25zLXB1Ymxpc2gnOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3NuczpQdWJsaXNoJ10sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBleHRlcm5hbElkOiBzbXNSb2xlRXh0ZXJuYWxJZCxcbiAgICAgICAgICAgIHNuc0NhbGxlckFybjogc21zUm9sZS5yb2xlQXJuLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIG1mYUNvbmZpZ3VyYXRpb24ocHJvcHM6IFVzZXJQb29sUHJvcHMpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmIChwcm9wcy5tZmEgPT09IHVuZGVmaW5lZCB8fCBwcm9wcy5tZmEgPT09IE1mYS5PRkYpIHtcbiAgICAgICAgICAgIC8vIHNpbmNlIGRlZmF1bHQgaXMgT0ZGLCB0cmVhdCB1bmRlZmluZWQgYW5kIE9GRiB0aGUgc2FtZSB3YXlcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAocHJvcHMubWZhU2Vjb25kRmFjdG9yID09PSB1bmRlZmluZWQgJiZcbiAgICAgICAgICAgIChwcm9wcy5tZmEgPT09IE1mYS5PUFRJT05BTCB8fCBwcm9wcy5tZmEgPT09IE1mYS5SRVFVSVJFRCkpIHtcbiAgICAgICAgICAgIHJldHVybiBbJ1NNU19NRkEnXTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IGVuYWJsZWRNZmFzID0gW107XG4gICAgICAgICAgICBpZiAocHJvcHMubWZhU2Vjb25kRmFjdG9yIS5zbXMpIHtcbiAgICAgICAgICAgICAgICBlbmFibGVkTWZhcy5wdXNoKCdTTVNfTUZBJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAocHJvcHMubWZhU2Vjb25kRmFjdG9yIS5vdHApIHtcbiAgICAgICAgICAgICAgICBlbmFibGVkTWZhcy5wdXNoKCdTT0ZUV0FSRV9UT0tFTl9NRkEnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHJldHVybiBlbmFibGVkTWZhcztcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIGNvbmZpZ3VyZVBhc3N3b3JkUG9saWN5KHByb3BzOiBVc2VyUG9vbFByb3BzKTogQ2ZuVXNlclBvb2wuUGFzc3dvcmRQb2xpY3lQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGNvbnN0IHRlbXBQYXNzd29yZFZhbGlkaXR5ID0gcHJvcHMucGFzc3dvcmRQb2xpY3k/LnRlbXBQYXNzd29yZFZhbGlkaXR5O1xuICAgICAgICBpZiAodGVtcFBhc3N3b3JkVmFsaWRpdHkgIT09IHVuZGVmaW5lZCAmJiB0ZW1wUGFzc3dvcmRWYWxpZGl0eS50b0RheXMoKSA+IER1cmF0aW9uLmRheXMoMzY1KS50b0RheXMoKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB0ZW1wUGFzc3dvcmRWYWxpZGl0eSBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIDM2NSBkYXlzIChyZWNlaXZlZDogJHt0ZW1wUGFzc3dvcmRWYWxpZGl0eS50b0RheXMoKX0pYCk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgbWluTGVuZ3RoID0gcHJvcHMucGFzc3dvcmRQb2xpY3kgPyBwcm9wcy5wYXNzd29yZFBvbGljeS5taW5MZW5ndGggPz8gOCA6IHVuZGVmaW5lZDtcbiAgICAgICAgaWYgKG1pbkxlbmd0aCAhPT0gdW5kZWZpbmVkICYmIChtaW5MZW5ndGggPCA2IHx8IG1pbkxlbmd0aCA+IDk5KSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBtaW5MZW5ndGggZm9yIHBhc3N3b3JkIG11c3QgYmUgYmV0d2VlbiA2IGFuZCA5OSAocmVjZWl2ZWQ6ICR7bWluTGVuZ3RofSlgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkSWZOb0tleXMoe1xuICAgICAgICAgICAgdGVtcG9yYXJ5UGFzc3dvcmRWYWxpZGl0eURheXM6IHRlbXBQYXNzd29yZFZhbGlkaXR5Py50b0RheXMoeyBpbnRlZ3JhbDogdHJ1ZSB9KSxcbiAgICAgICAgICAgIG1pbmltdW1MZW5ndGg6IG1pbkxlbmd0aCxcbiAgICAgICAgICAgIHJlcXVpcmVMb3dlcmNhc2U6IHByb3BzLnBhc3N3b3JkUG9saWN5Py5yZXF1aXJlTG93ZXJjYXNlLFxuICAgICAgICAgICAgcmVxdWlyZVVwcGVyY2FzZTogcHJvcHMucGFzc3dvcmRQb2xpY3k/LnJlcXVpcmVVcHBlcmNhc2UsXG4gICAgICAgICAgICByZXF1aXJlTnVtYmVyczogcHJvcHMucGFzc3dvcmRQb2xpY3k/LnJlcXVpcmVEaWdpdHMsXG4gICAgICAgICAgICByZXF1aXJlU3ltYm9sczogcHJvcHMucGFzc3dvcmRQb2xpY3k/LnJlcXVpcmVTeW1ib2xzLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcHJpdmF0ZSBzY2hlbWFDb25maWd1cmF0aW9uKHByb3BzOiBVc2VyUG9vbFByb3BzKTogQ2ZuVXNlclBvb2wuU2NoZW1hQXR0cmlidXRlUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGNvbnN0IHNjaGVtYTogQ2ZuVXNlclBvb2wuU2NoZW1hQXR0cmlidXRlUHJvcGVydHlbXSA9IFtdO1xuICAgICAgICBpZiAocHJvcHMuc3RhbmRhcmRBdHRyaWJ1dGVzKSB7XG4gICAgICAgICAgICBjb25zdCBzdGRBdHRyaWJ1dGVzID0gKE9iamVjdC5lbnRyaWVzKHByb3BzLnN0YW5kYXJkQXR0cmlidXRlcykgYXMgQXJyYXk8W2tleW9mIFN0YW5kYXJkQXR0cmlidXRlcywgU3RhbmRhcmRBdHRyaWJ1dGVdPilcbiAgICAgICAgICAgICAgICAuZmlsdGVyKChbLCBhdHRyXSkgPT4gISFhdHRyKVxuICAgICAgICAgICAgICAgIC5tYXAoKFthdHRyTmFtZSwgYXR0cl0pID0+ICh7XG4gICAgICAgICAgICAgICAgbmFtZTogU3RhbmRhcmRBdHRyaWJ1dGVOYW1lc1thdHRyTmFtZV0sXG4gICAgICAgICAgICAgICAgbXV0YWJsZTogYXR0ci5tdXRhYmxlID8/IHRydWUsXG4gICAgICAgICAgICAgICAgcmVxdWlyZWQ6IGF0dHIucmVxdWlyZWQgPz8gZmFsc2UsXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgICAgICBzY2hlbWEucHVzaCguLi5zdGRBdHRyaWJ1dGVzKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuY3VzdG9tQXR0cmlidXRlcykge1xuICAgICAgICAgICAgY29uc3QgY3VzdG9tQXR0cnMgPSBPYmplY3Qua2V5cyhwcm9wcy5jdXN0b21BdHRyaWJ1dGVzKS5tYXAoKGF0dHJOYW1lKSA9PiB7XG4gICAgICAgICAgICAgICAgY29uc3QgYXR0ckNvbmZpZyA9IHByb3BzLmN1c3RvbUF0dHJpYnV0ZXMhW2F0dHJOYW1lXS5iaW5kKCk7XG4gICAgICAgICAgICAgICAgY29uc3QgbnVtYmVyQ29uc3RyYWludHM6IENmblVzZXJQb29sLk51bWJlckF0dHJpYnV0ZUNvbnN0cmFpbnRzUHJvcGVydHkgPSB7XG4gICAgICAgICAgICAgICAgICAgIG1pblZhbHVlOiBhdHRyQ29uZmlnLm51bWJlckNvbnN0cmFpbnRzPy5taW4/LnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICAgIG1heFZhbHVlOiBhdHRyQ29uZmlnLm51bWJlckNvbnN0cmFpbnRzPy5tYXg/LnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICBjb25zdCBzdHJpbmdDb25zdHJhaW50czogQ2ZuVXNlclBvb2wuU3RyaW5nQXR0cmlidXRlQ29uc3RyYWludHNQcm9wZXJ0eSA9IHtcbiAgICAgICAgICAgICAgICAgICAgbWluTGVuZ3RoOiBhdHRyQ29uZmlnLnN0cmluZ0NvbnN0cmFpbnRzPy5taW5MZW4/LnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICAgIG1heExlbmd0aDogYXR0ckNvbmZpZy5zdHJpbmdDb25zdHJhaW50cz8ubWF4TGVuPy50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICAgICAgbmFtZTogYXR0ck5hbWUsXG4gICAgICAgICAgICAgICAgICAgIGF0dHJpYnV0ZURhdGFUeXBlOiBhdHRyQ29uZmlnLmRhdGFUeXBlLFxuICAgICAgICAgICAgICAgICAgICBudW1iZXJBdHRyaWJ1dGVDb25zdHJhaW50czogYXR0ckNvbmZpZy5udW1iZXJDb25zdHJhaW50c1xuICAgICAgICAgICAgICAgICAgICAgICAgPyBudW1iZXJDb25zdHJhaW50c1xuICAgICAgICAgICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgICAgIHN0cmluZ0F0dHJpYnV0ZUNvbnN0cmFpbnRzOiBhdHRyQ29uZmlnLnN0cmluZ0NvbnN0cmFpbnRzXG4gICAgICAgICAgICAgICAgICAgICAgICA/IHN0cmluZ0NvbnN0cmFpbnRzXG4gICAgICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICAgICAgbXV0YWJsZTogYXR0ckNvbmZpZy5tdXRhYmxlLFxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIHNjaGVtYS5wdXNoKC4uLmN1c3RvbUF0dHJzKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoc2NoZW1hLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gc2NoZW1hO1xuICAgIH1cbiAgICBwcml2YXRlIGFjY291bnRSZWNvdmVyeShwcm9wczogVXNlclBvb2xQcm9wcyk6IHVuZGVmaW5lZCB8IENmblVzZXJQb29sLkFjY291bnRSZWNvdmVyeVNldHRpbmdQcm9wZXJ0eSB7XG4gICAgICAgIGNvbnN0IGFjY291bnRSZWNvdmVyeSA9IHByb3BzLmFjY291bnRSZWNvdmVyeSA/PyBBY2NvdW50UmVjb3ZlcnkuUEhPTkVfV0lUSE9VVF9NRkFfQU5EX0VNQUlMO1xuICAgICAgICBzd2l0Y2ggKGFjY291bnRSZWNvdmVyeSkge1xuICAgICAgICAgICAgY2FzZSBBY2NvdW50UmVjb3ZlcnkuRU1BSUxfQU5EX1BIT05FX1dJVEhPVVRfTUZBOlxuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIHJlY292ZXJ5TWVjaGFuaXNtczogW1xuICAgICAgICAgICAgICAgICAgICAgICAgeyBuYW1lOiAndmVyaWZpZWRfZW1haWwnLCBwcmlvcml0eTogMSB9LFxuICAgICAgICAgICAgICAgICAgICAgICAgeyBuYW1lOiAndmVyaWZpZWRfcGhvbmVfbnVtYmVyJywgcHJpb3JpdHk6IDIgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgY2FzZSBBY2NvdW50UmVjb3ZlcnkuUEhPTkVfV0lUSE9VVF9NRkFfQU5EX0VNQUlMOlxuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIHJlY292ZXJ5TWVjaGFuaXNtczogW1xuICAgICAgICAgICAgICAgICAgICAgICAgeyBuYW1lOiAndmVyaWZpZWRfcGhvbmVfbnVtYmVyJywgcHJpb3JpdHk6IDEgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgIHsgbmFtZTogJ3ZlcmlmaWVkX2VtYWlsJywgcHJpb3JpdHk6IDIgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgY2FzZSBBY2NvdW50UmVjb3ZlcnkuRU1BSUxfT05MWTpcbiAgICAgICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgICAgICByZWNvdmVyeU1lY2hhbmlzbXM6IFt7IG5hbWU6ICd2ZXJpZmllZF9lbWFpbCcsIHByaW9yaXR5OiAxIH1dLFxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICBjYXNlIEFjY291bnRSZWNvdmVyeS5QSE9ORV9PTkxZX1dJVEhPVVRfTUZBOlxuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIHJlY292ZXJ5TWVjaGFuaXNtczogW3sgbmFtZTogJ3ZlcmlmaWVkX3Bob25lX251bWJlcicsIHByaW9yaXR5OiAxIH1dLFxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICBjYXNlIEFjY291bnRSZWNvdmVyeS5OT05FOlxuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIHJlY292ZXJ5TWVjaGFuaXNtczogW3sgbmFtZTogJ2FkbWluX29ubHknLCBwcmlvcml0eTogMSB9XSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgY2FzZSBBY2NvdW50UmVjb3ZlcnkuUEhPTkVfQU5EX0VNQUlMOlxuICAgICAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgQWNjb3VudFJlY292ZXJ5IHR5cGUgLSAke2FjY291bnRSZWNvdmVyeX1gKTtcbiAgICAgICAgfVxuICAgIH1cbn1cbmZ1bmN0aW9uIHVuZGVmaW5lZElmTm9LZXlzKHN0cnVjdDogb2JqZWN0KTogb2JqZWN0IHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBhbGxVbmRlZmluZWQgPSBPYmplY3QudmFsdWVzKHN0cnVjdCkucmVkdWNlKChhY2MsIHYpID0+IGFjYyAmJiAodiA9PT0gdW5kZWZpbmVkKSwgdHJ1ZSk7XG4gICAgcmV0dXJuIGFsbFVuZGVmaW5lZCA/IHVuZGVmaW5lZCA6IHN0cnVjdDtcbn1cbiJdfQ==