"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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